﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/friends/detail/friends_Async.h>
#include <nn/friends/detail/friends_ShimLibraryGlobal.h>
#include <nn/os/os_SdkMutex.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/result/result_HandlingUtility.h>
#include <mutex>

namespace nn { namespace friends { namespace detail  {

namespace
{
    nn::os::SdkMutexType g_Mutex = NN_OS_SDK_MUTEX_INITIALIZER();

    void* MallocFunction(size_t size) NN_NOEXCEPT
    {
        return std::malloc(size);
    }

    void FreeFunction(void* p, size_t) NN_NOEXCEPT
    {
        free(p);
    }

    nn::AllocateFunction g_AllocateFunction = MallocFunction;
    nn::DeallocateFunction g_FreeFunction = FreeFunction;
}

void SetAsyncAllocator(nn::AllocateFunction allocateFunction, nn::DeallocateFunction freeFunction) NN_NOEXCEPT
{
    std::lock_guard<decltype (g_Mutex)> lock(g_Mutex);

    g_AllocateFunction = allocateFunction;
    g_FreeFunction = freeFunction;
}

nn::Result CallAsync(AsyncContext* outAsync,
    const char* threadName, AsyncContextInternal::SyncCall function, void* param, size_t paramSize) NN_NOEXCEPT
{
    detail::ipc::IFriendService* session = nullptr;

    NN_RESULT_DO(ShimLibraryGlobal::GetInstance().CreateAsyncSession(&session));

    bool isSucceeded = false;

    NN_UTIL_SCOPE_EXIT
    {
        if (!isSucceeded)
        {
            nn::sf::ReleaseSharedObject(session);
        }
    };

    AsyncContextInternal* internal = nullptr;
    {
        std::lock_guard<decltype (g_Mutex)> lock(g_Mutex);

        NN_RESULT_DO(AsyncContextInternal::CreateInstance(&internal, session, g_AllocateFunction, g_FreeFunction));
    }

    internal->CallAsync(threadName, function, param, paramSize);
    outAsync->SetInternal(internal);

    isSucceeded = true;

    NN_RESULT_SUCCESS;
}

nn::Result CallAsync(AsyncContext* outAsync,
    const char* threadName, AsyncContextInternal::SyncCall function, AsyncContextInternal::DeepCopyCallback callback, const void* sourceParam) NN_NOEXCEPT
{
    detail::ipc::IFriendService* session = nullptr;

    NN_RESULT_DO(ShimLibraryGlobal::GetInstance().CreateAsyncSession(&session));

    bool isSucceeded = false;

    NN_UTIL_SCOPE_EXIT
    {
        if (!isSucceeded)
        {
            nn::sf::ReleaseSharedObject(session);
        }
    };

    AsyncContextInternal* internal = nullptr;
    {
        std::lock_guard<decltype (g_Mutex)> lock(g_Mutex);

        NN_RESULT_DO(AsyncContextInternal::CreateInstance(&internal, session, g_AllocateFunction, g_FreeFunction));
    }

    internal->CallAsync(threadName, function, callback, sourceParam);
    outAsync->SetInternal(internal);

    isSucceeded = true;

    NN_RESULT_SUCCESS;
}

}}}
