﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

// fontll2 は fontll と大部分のソースコードが同じであるため、メンテナンス性の観点から共通化しています。
// NN_FONTLL2_BUILD_FONTLL2 のマクロを定義すると fontll2 としてビルドされます。
// NN_FONTLL2_BUILD_FONTLL2 マクロは !.nact によって定義されます。
#if defined(NN_FONTLL2_BUILD_FONTLL2)
#include <nn/fontll2/fontll2_ScalableFontEngine.h>
#include "../fontll/detail/fontll_BfttfDecoder.h"
#else
#include <nn/fontll/fontll_ScalableFontEngine.h>
#include "detail/fontll_BfttfDecoder.h"
#endif

#include <nn/nn_StaticAssert.h>
#include <cmath>

#pragma warning(push)
extern "C"
{
#pragma warning(disable:4819)
#ifdef _WIN64
#pragma warning(disable:4324)
#endif// _WIN64
#include <fs_itype.h>
}
#pragma warning(pop)

// ファイル読み込みの機能は使わないため、空実装にしています。
namespace
{
    static FS_FILE* FsOpenFile(const char* path, const char* mode)
    {
        NN_UNUSED(path);
        NN_UNUSED(mode);
        return NULL;
    }

    static int FsCloseFile(FS_FILE* stream)
    {
        NN_UNUSED(stream);
        return 0;
    }

    static size_t FsReadFile(void* buffer, size_t size, size_t count, FS_FILE* stream)
    {
        NN_UNUSED(buffer);
        NN_UNUSED(size);
        NN_UNUSED(count);
        NN_UNUSED(stream);
        return 0;
    }

    static int FsSeekFile(FS_FILE* stream, FS_LONG offset, int origin)
    {
        NN_UNUSED(stream);
        NN_UNUSED(offset);
        NN_UNUSED(origin);
        return 0;
    }

    static FS_LONG FsTellFile(FS_FILE* stream)
    {
        NN_UNUSED(stream);
        return 0;
    }

    static int FsErrorFile(FS_FILE* stream)
    {
        NN_UNUSED(stream);
        return 0;
    }
}

namespace nn {
#if defined(NN_FONTLL2_BUILD_FONTLL2)
    namespace fontll2 {
#else
    namespace fontll {
#endif

        //---------------------------------------------------------------------------
        ScalableFontEngine::ScalableFontEngine()
            : m_pInternalBuffer(NULL)
        {
            std::memset(&m_Handle, 0, sizeof(Handle));

#if defined(NN_FONTLL2_BUILD_FONTLL2)
            // 「参照されていないローカル関数は削除されました」の警告への対処
            NN_UNUSED(&CountLeadingZeroes);
            NN_UNUSED(&UMUL64_HIGH32);
            NN_UNUSED(&UMUL64_MID32);
            NN_UNUSED(&UMUL64);
            NN_UNUSED(&I1616_ABS);
            NN_UNUSED(&I1616_NEGABS);
            NN_UNUSED(&I1616_NEGATE);
            NN_UNUSED(&I1616_FLOOR);
            NN_UNUSED(&I1616_TO_U32);
            NN_UNUSED(&U32_TO_I1616);
            NN_UNUSED(&I1616_TO_I32);
            NN_UNUSED(&I1616_ADD);
            NN_UNUSED(&I1616_SUB);
            NN_UNUSED(&I1616_SQR);
            NN_UNUSED(&I1616_MUL_I2408);
            NN_UNUSED(&I1616_MUL);
#endif
        }

        //---------------------------------------------------------------------------
        ScalableFontEngine::~ScalableFontEngine()
        {

        }

        //---------------------------------------------------------------------------
        int ScalableFontEngine::Initialize(void* pHeap, uint32_t heapSize)
        {
            nnfontllSetFsFunctions(
                FsOpenFile,
                FsCloseFile,
                FsReadFile,
                FsSeekFile,
                FsTellFile,
                FsErrorFile);
            m_pInternalBuffer = pHeap;
            return FS_init_ex(reinterpret_cast<FS_STATE*>(&m_Handle), pHeap, heapSize);
        }

        //---------------------------------------------------------------------------
        int ScalableFontEngine::Finalize()
        {
            return FS_exit(reinterpret_cast<FS_STATE*>(&m_Handle));
        }

        //---------------------------------------------------------------------------

        void* ScalableFontEngine::GetPointerToWorkBuffer()
        {
            return m_pInternalBuffer;
        }

        //---------------------------------------------------------------------------
        int ScalableFontEngine::GetFontMetrics(Metrics* pM)
        {
            return FS_font_metrics(reinterpret_cast<FS_STATE*>(&m_Handle), reinterpret_cast<FONT_METRICS*>(pM));
        }

        //---------------------------------------------------------------------------
        //ERR_FONT_NOT_FOUND  Both the 'path' and 'memptr' inputs were NULL
        //ERR_FONT_BUFFER_TOO_SMALL  Buffer supplied is too small to hold the font name that was generated
        //ERR_DUPLICATED_FONT_NAME  Could not generate a font name (internal error)
        int ScalableFontEngine::LoadFont(char* pNameBuffer, const void* pMem, uint32_t index, uint32_t nameBufferLen)
        {
            NN_STATIC_ASSERT(sizeof(FS_STATE) <= sizeof(Handle));
            return FS_load_font(
                reinterpret_cast<FS_STATE*>(&m_Handle),
                NULL,
                reinterpret_cast<FS_BYTE*>(const_cast<void*>(pMem)),
                index,
                nameBufferLen,
                pNameBuffer);
        }

        //---------------------------------------------------------------------------
        int ScalableFontEngine::SetScale(int32_t s00, int32_t s01, Fixed32 s10, Fixed32 s11)
        {
            // FS_FIXED => s32 FS_LONG =>
            // SUCCESS
            // ERR_NO_CURRENT_LFNT  no current loaded font
            // ERR_SCALE_LIMIT  characters scale would exceed limit of +/- 16,384).
            // ERR_SCALE_DEGENERATE  matrix is degenerate
            return FS_set_scale(reinterpret_cast<FS_STATE*>(&m_Handle), s00, s01, s10, s11);
        }

        //---------------------------------------------------------------------------
        //
        int ScalableFontEngine::SetBoldWeight(Fixed32 weight)
        {
            return FS_set_bold_pct(reinterpret_cast<FS_STATE*>(&m_Handle), weight);
        }

        //---------------------------------------------------------------------------
        //
        int ScalableFontEngine::SetFlags(Flags flag)
        {
            static const FS_ULONG flagsTable[] =
            {
                FLAGS_NO_EFFECT,
                FLAGS_OUTLINED_UNFILLED,
                FLAGS_OUTLINED_FILLED,
            };
            return FS_set_flags(reinterpret_cast<FS_STATE*>(&m_Handle), flagsTable[flag]);
        }

        //---------------------------------------------------------------------------
        //
        int ScalableFontEngine::SetOutlineWidth(uint16_t width)
        {
            return FS_set_outline_width(reinterpret_cast<FS_STATE*>(&m_Handle), width);
        }

        //---------------------------------------------------------------------------
        // FS_ULONG => u32 FS_USHORT => u16 FS_SHORT => s16
        // FS_MAP_GRAYMAP8
        int ScalableFontEngine::GetAdvance(int16_t* idx, int16_t* idy, Fixed32* dx, Fixed32* dy, uint32_t id, uint16_t type)
        {
            return FS_get_advance(
                reinterpret_cast<FS_STATE*>(&m_Handle),
                id,
                type,
                idx,
                idy,
                reinterpret_cast<FS_FIXED*>(dx),
                reinterpret_cast<FS_FIXED*>(dy));
        }

        //---------------------------------------------------------------------------
        int ScalableFontEngine::GetKerning(Fixed32* dx, Fixed32* dy, uint32_t id1, uint32_t id2)
        {
            return FS_get_kerning(
                reinterpret_cast<FS_STATE*>(&m_Handle),
                id1,
                id2,
                reinterpret_cast<FS_FIXED*>(dx),
                reinterpret_cast<FS_FIXED*>(dy));
        }

        //---------------------------------------------------------------------------
        bool ScalableFontEngine::CheckTtfKernTableExist()
        {
            FS_ULONG len;
            void* pKernTable = FS_get_table(reinterpret_cast<FS_STATE*>(&m_Handle), TAG_kern, TBL_QUERY, &len);
            if(pKernTable != NULL)
            {
                return true;
            }

            return false;
        }

        //---------------------------------------------------------------------------
        OtfKerningTable* ScalableFontEngine::InitializeOtfKerningTable(nn::AlignedAllocateFunctionWithUserData allocateFunction, void* pUserDataForAllocateFunction)
        {
            return InitializeOtfKerningTable(allocateFunction, pUserDataForAllocateFunction, false);
        }

        OtfKerningTable* ScalableFontEngine::InitializeOtfKerningTable(nn::AlignedAllocateFunctionWithUserData allocateFunction, void* pUserDataForAllocateFunction, bool ignorePalt)
        {
            // GPOS テーブルの取得
            void* pGposTable;
            {
                FS_ULONG len;
                pGposTable = FS_get_table(reinterpret_cast<FS_STATE*>(&m_Handle), TAG_GPOS, TBL_EXTRACT, &len);
                if (pGposTable == NULL)
                {
                    return NULL;
                }
            }

            // OTF カーニングテーブルの要素数の取得
            OtfKerningTable* pOtfKerningTable = static_cast<OtfKerningTable*>(allocateFunction(sizeof(OtfKerningTable), 4, pUserDataForAllocateFunction));
            pOtfKerningTable->kerningPairCount = PickKerningFromGposTable(NULL, pGposTable, ignorePalt);
            {
                size_t size = static_cast<size_t>(pOtfKerningTable->kerningPairCount) * sizeof(OtfKerningTable::KerningPair);
                pOtfKerningTable->pKerningPairs = static_cast<OtfKerningTable::KerningPair*>(allocateFunction(size, 4, pUserDataForAllocateFunction));
            }

            // OTF カーニングテーブルの取得
            PickKerningFromGposTable(pOtfKerningTable, pGposTable, ignorePalt);

            // あらかじめ Em あたりの単位の逆数を求めて保持しておく
            {
                Metrics metrics;
                GetFontMetrics(&metrics);
                pOtfKerningTable->reciprocalUnitsPerEm = 1.0f / static_cast<float>(metrics.unitsPerEm);
            }


            // GPOS テーブルの解放
            FS_free_table(reinterpret_cast<FS_STATE*>(&m_Handle), pGposTable);

            return pOtfKerningTable;
        }

        //---------------------------------------------------------------------------
        void ScalableFontEngine::FinalizeOtfKerningTable(OtfKerningTable* pOtfKerningTable, nn::FreeFunctionWithUserData freeFunction, void* pUserDataForFreeFunction)
        {
            if (pOtfKerningTable->pKerningPairs != NULL)
            {
                freeFunction(pOtfKerningTable->pKerningPairs, pUserDataForFreeFunction);
            }
            if (pOtfKerningTable != NULL)
            {
                freeFunction(pOtfKerningTable, pUserDataForFreeFunction);
            }
        }

        //---------------------------------------------------------------------------
        int ScalableFontEngine::AcquireOtfKerning(const OtfKerningTable* pOtfKerningTable, uint32_t id1, uint32_t id2, uint32_t fontSize)
        {
            // カーニングの生情報を取得
            float rawKerning = static_cast<float>(AcquireRawOtfKerning(pOtfKerningTable, id1, id2));

            // カーニングの取得
            int kerning = static_cast<int>(::std::floor(rawKerning * pOtfKerningTable->reciprocalUnitsPerEm * static_cast<float>(fontSize) + 0.5f));
            return kerning;
        }

        //---------------------------------------------------------------------------
        int ScalableFontEngine::AcquireOtfKerningFirst(const OtfKerningTable* pOtfKerningTable, uint32_t id, uint32_t fontSize)
        {
            // カーニングの生情報を取得
            float rawKerning = static_cast<float>(AcquireRawOtfKerningFirst(pOtfKerningTable, id));

            // カーニングの取得
            int kerning = static_cast<int>(::std::floor(rawKerning * pOtfKerningTable->reciprocalUnitsPerEm * static_cast<float>(fontSize) + 0.5f));
            return kerning;
        }

        //---------------------------------------------------------------------------
        int ScalableFontEngine::AcquireOtfKerningLast(const OtfKerningTable* pOtfKerningTable, uint32_t id, uint32_t fontSize)
        {
            // カーニングの生情報を取得
            float rawKerning = static_cast<float>(AcquireRawOtfKerningLast(pOtfKerningTable, id));

            // カーニングの取得
            int kerning = static_cast<int>(::std::floor(rawKerning * pOtfKerningTable->reciprocalUnitsPerEm * static_cast<float>(fontSize) + 0.5f));
            return kerning;
        }

        //---------------------------------------------------------------------------
        int ScalableFontEngine::AcquireRawOtfKerning(const OtfKerningTable* pOtfKerningTable, uint32_t id1, uint32_t id2)
        {
            if (pOtfKerningTable == NULL)
            {
                return 0;
            }

            // 文字コードをGlyphインデックスに変換
            uint16_t c1 = map_char(reinterpret_cast<FS_STATE*>(&m_Handle), id1, 0);
            uint16_t c2 = map_char(reinterpret_cast<FS_STATE*>(&m_Handle), id2, 0);

            int advance;
            int placement;

            // Palt の検索
            // Palt の値は、左の文字の advance と右の文字の placement を反映させる
            int palt = 0;
            BinarySearchForOtfKerning(&advance, &placement, pOtfKerningTable, c1, 0xffff);
            palt -= placement; // 原点基準になっているのでセル基準に補正する
            palt += advance;
            BinarySearchForOtfKerning(&advance, &placement, pOtfKerningTable, c2, 0xffff);
            palt += placement;

            // Kern の検索
            // Kern の値は、advance のみを反映させる
            int kern = 0;
            BinarySearchForOtfKerning(&advance, &placement, pOtfKerningTable, c1, c2);
            kern += advance;

            return static_cast<int>(static_cast<int16_t>(palt + kern));
        }

        //---------------------------------------------------------------------------
        int ScalableFontEngine::AcquireRawOtfKerningFirst(const OtfKerningTable* pOtfKerningTable, uint32_t id)
        {
            if (pOtfKerningTable == NULL)
            {
                return 0;
            }

            // 文字コードをGlyphインデックスに変換
            uint16_t c = map_char(reinterpret_cast<FS_STATE*>(&m_Handle), id, 0);

            int advance;
            int placement;

            // Palt の検索
            // Palt の値は、左の文字の advance と右の文字の placement を反映させる
            int palt = 0;
            BinarySearchForOtfKerning(&advance, &placement, pOtfKerningTable, c, 0xffff);
            palt += placement;

            return static_cast<int>(static_cast<int16_t>(palt));
        }

        //---------------------------------------------------------------------------
        int ScalableFontEngine::AcquireRawOtfKerningLast(const OtfKerningTable* pOtfKerningTable, uint32_t id)
        {
            if (pOtfKerningTable == NULL)
            {
                return 0;
            }

            // 文字コードをGlyphインデックスに変換
            uint16_t c = map_char(reinterpret_cast<FS_STATE*>(&m_Handle), id, 0);

            int advance;
            int placement;

            // Palt の検索
            // Palt の値は、左の文字の advance と右の文字の placement を反映させる
            int palt = 0;
            BinarySearchForOtfKerning(&advance, &placement, pOtfKerningTable, c, 0xffff);
            palt -= placement; // 原点基準になっているのでセル基準に補正する
            palt += advance;

            return static_cast<int>(static_cast<int16_t>(palt));
        }

        //---------------------------------------------------------------------------
        //ERR_FONT_NOT_FOUND  font by that name is not found
        //ERR_FONT_NOT_VALID  current font file is not a valid font
        //ERR_MALLOC_FAIL  not enough memory to load font.
        //ERR_NOT_A_TTF  font is not a TTF font
        //ERR_ACT3_DISK  font is a disk-based ACT3 compressed font
        //ERR_ACT3_UNDEF  font is ACT3 compressed but FS_ACT3 configuration not defined
        //ERR_PFR_UNDEF  font is in PFR format but PFR configuration not defined
        int ScalableFontEngine::SetFont(char* name)
        {
            int result = FS_set_font(reinterpret_cast<FS_STATE*>(&m_Handle), name);
            if(result != SUCCESS)
            {
                return result;
            }

            // まずは UCS-4 の cmap テーブルを読み込んでみて、存在しなければ UCS-2 を使う。
            // 指定する値は以下の通り。 ( https://www.microsoft.com/typography/otspec/cmap.htm より抜粋 )
            // | Platform ID | Encoding ID | Description   |
            // | 3           | 1           | Unicode UCS-2 |
            // | 3           | 10          | Unicode UCS-4 |
            result = FS_set_cmap(reinterpret_cast<FS_STATE*>(&m_Handle), 3, 10);
            if (result != SUCCESS)
            {
                result = FS_set_cmap(reinterpret_cast<FS_STATE*>(&m_Handle), 3, 1);
            }

            return result;
        }

        //---------------------------------------------------------------------------
        int ScalableFontEngine::GetError()
        {
            return FS_error(reinterpret_cast<FS_STATE*>(&m_Handle));
        }

        //---------------------------------------------------------------------------
        // type のバリエーション はどの程度提供しなければならないか...
        // とりあえず最初は、FS_MAP_GRAYMAP8 だけが指定できるようにしておく。
        //FS_MAP_ANY_EDGE_GRAYMAP  (FS_MAP_EDGE_GRAYMAP4 | FS_MAP_EDGE_GRAYMAP8 | FS_EDGE_GRAYMAP2)
        //FS_MAP_EDGE_TECH  (equivalent to FS_MAP_EDGE_GRAYMAP4)
        //FS_MAP_EDGE_GRAYMAP4  (4-bit Edge(TM) graymap)
        //FS_MAP_EDGE_GRAYMAP8  (8-bit Edge(TM) graymap)
        //FS_MAP_EDGE_GRAYMAP2  (2-bit Edge(TM) graymap)
        //FS_MAP_ANY_GRAYMAP  (FS_MAP_GRAYMAP4 | FS_MAP_GRAYMAP8 | FS_GRAYMAP2)
        //FS_MAP_GRAYMAP  (equivalent to FS_MAP_GRAYMAP4)
        //FS_MAP_GRAYMAP4  (4-bit graymap)
        //FS_MAP_GRAYMAP8  (8-bit graymap)
        //FS_MAP_GRAYMAP2  (2-bit graymap)
        //FS_MAP_BITMAP  (1-bit bitmap)
        //FS_MAP_RASTER_ICON  (size-dependent raster icon)
        //FS_MAP_VECTOR_ICON  (scalable vector icon)
        //FS_MAP_ANY_ICON  (FS_MAP_RASTER_ICON | FS_MAP_VECTOR_ICON)
        //FS_MAP_DISTANCEFIELD  (8-bit encoded distance field map)
        GlyphMap* ScalableFontEngine::AcquireGlyphmap(uint32_t id, uint16_t type)
        {
            return reinterpret_cast<GlyphMap*>(FS_get_glyphmap(reinterpret_cast<FS_STATE*>(&m_Handle), id, type));
        }

        //---------------------------------------------------------------------------
        bool ScalableFontEngine::CheckGlyphExist(uint32_t id)
        {
            return FS_map_char(reinterpret_cast<FS_STATE*>(&m_Handle), id) != 0;
        }

        //---------------------------------------------------------------------------
        int ScalableFontEngine::ReleasesGlyph(void * pGlyph)
        {
            return FS_free_char(reinterpret_cast<FS_STATE*>(&m_Handle), pGlyph);
        }

        //---------------------------------------------------------------------------
        void ScalableFontEngine::SetAutoHint(bool isAutoHint)
        {
            FS_set_flags(
                reinterpret_cast<FS_STATE*>(&m_Handle),
                isAutoHint ? FLAGS_FORCE_AUTOHINT : FLAGS_FORCE_AUTOHINT_OFF);
        }

        //---------------------------------------------------------------------------
        bool ScalableFontEngine::BinarySearchForOtfKerning(int* pAdvance, int* pPlacement, const OtfKerningTable* pOtfKerningTable, uint32_t c1, uint32_t c2)
        {
            *pAdvance = 0;
            *pPlacement = 0;

            // カーニング情報をテーブルから二分探索で求める
            int min = 0;
            int max = pOtfKerningTable->kerningPairCount - 1;
            while (min <= max)
            {
                int idx = (min + max) / 2;
                const OtfKerningTable::KerningPair* pKerningPair = &pOtfKerningTable->pKerningPairs[idx];
                if (pKerningPair->first > c1 || (pKerningPair->first == c1 && pKerningPair->second > c2))
                {
                    max = idx - 1;
                    continue;
                }
                if (pKerningPair->first < c1 || (pKerningPair->first == c1 && pKerningPair->second < c2))
                {
                    min = idx + 1;
                    continue;
                }
                // 見つかった
                *pAdvance = static_cast<int>(pKerningPair->advance);
                *pPlacement = static_cast<int>(pKerningPair->placement);
                return true;
            }

            // 見つからなかった場合は false を返す
            return false;
        }

        //---------------------------------------------------------------------------
        void* ScalableFontEngineHelper::Decode(const void* pData, uint32_t dataSize)
        {
            return detail::BfttfDecoder::Decode(const_cast<void*>(pData), dataSize);
        }

    }
}
