﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/os.h>
#include <nn/nn_Result.h>
#include <nn/nn_SystemThreadDefinition.h>
#include <nn/util/util_FormatString.h>

#include <nn/sf/sf_HipcServer.h>
#include <nn/sf/sf_ObjectFactory.h>
#include <nn/sf/sf_ExpHeapAllocator.h>

#include <nn/psc.h>
#include <nn/fgm/server/fgm_Server.h>
#include <nn/fgm/sfdl/fgm.sfdl.h>

#include "fgm_Resource.h"
#include "fgm_Debugger.h"
#include "detail/fgm_Hardware.h"

namespace nn     {
namespace fgm    {
namespace server {

nn::lmem::HeapHandle     g_HeapHandle;

namespace        {

const int SessionThreadStackSize = 8192;
const int SessionThreadPriority  = NN_SYSTEM_THREAD_PRIORITY(fgm, IpcServer);

nn::sf::ExpHeapAllocator g_ServiceFrameworkAllocator;

// Essentially SessionCount is the max number of processes
// using fgm at any single time.
enum SessionCount
{
    SessionCount_Priority0 = 4, // highest fgm priority
    SessionCount_Priority8 = 30, // default fgm priority
    SessionCount_Priority9 = 2,
    SessionCount_Debugger  = 2
};

const int SessionCount[NumberOfPriorities] =
{
    SessionCount_Priority0,
    SessionCount_Priority8,
    SessionCount_Priority9
};

const int SessionMaxCount =
    SessionCount_Debugger  +
    SessionCount_Priority0 +
    SessionCount_Priority8 +
    SessionCount_Priority9;

enum SessionPort
{
    SessionPort_Debug   = NumberOfPriorities,
    SessionPort_Highest = NumberOfPriorities + 1,
};

struct ServerOptions
{
    static const int SubDomainCountMax         = 60;
    static const int ObjectInSubDomainCountMax = 200;
};

class SessionManager :
    public nn::sf::ISharedObject
{
private:
    Priority m_Port;

public:
    nn::Result Initialize(nn::sf::Out<nn::sf::SharedPointer<nn::fgm::sf::IRequest>> pRequestOut)
    NN_NOEXCEPT
    {
        auto pRequest = nn::sf::ObjectFactory<
                           nn::sf::ExpHeapAllocator::Policy
                          >::CreateSharedEmplaced<
                           nn::fgm::sf::IRequest,
                           Request
                          >(&g_ServiceFrameworkAllocator, m_Port);

        if (pRequest != nullptr)
        {
            pRequestOut.Set(pRequest);
            return ResultSuccess();
        }

        return nn::fgm::ResultOutOfMemory();
    }

    explicit SessionManager(Priority port)
    NN_NOEXCEPT :
        m_Port(port)
    {

    }
};

class FgmService
    : public nn::sf::HipcSimpleAllInOneServerManager<SessionMaxCount, SessionPort_Highest, ServerOptions>
{
private:
    nn::os::ThreadType          m_SessionThread;
    uint8_t                     m_SessionThreadStack[SessionThreadStackSize] NN_ALIGNAS(os::StackRegionAlignment);
    nn::psc::PmModule           m_PmModule;
    nn::psc::PmModuleId*        m_pDependencyList;
    uint32_t                    m_DependencyCount;
    nn::os::MultiWaitHolderType m_PmModuleEventHolder;

    void FgmServerThread()
    NN_NOEXCEPT
    {
        nn::Result          result;
        nn::psc::PmState    state;
        nn::psc::PmFlagSet  flags;

        result = m_PmModule.Initialize(
                nn::psc::PmModuleId_Fgm,
                m_pDependencyList,
                m_DependencyCount,
                nn::os::EventClearMode_ManualClear);

        NN_ABORT_UNLESS_RESULT_SUCCESS(result);

        nn::os::InitializeMultiWaitHolder(
                &m_PmModuleEventHolder,
                m_PmModule.GetEventPointer()->GetBase());

        nn::os::SetMultiWaitHolderUserData(
                &m_PmModuleEventHolder,
                nn::psc::PmModuleId_Fgm);

        AddUserWaitHolder(&m_PmModuleEventHolder);

        while (NN_STATIC_CONDITION(true))
        {
            nn::os::MultiWaitHolderType* p = Wait();

            switch (nn::os::GetMultiWaitHolderUserData(p))
            {
            case AcceptTag:
            case InvokeTag:
                ProcessAuto(p);
                break;

            case nn::psc::PmModuleId_Fgm:
                // SF's Wait() handler removes holder from the multiwait.
                // Need to add it back after each event.
                m_PmModule.GetEventPointer()->Clear();
                AddUserWaitHolder(&m_PmModuleEventHolder);

                result = m_PmModule.GetRequest(&state, &flags);
                NN_ABORT_UNLESS_RESULT_SUCCESS(result);

                switch (state)
                {
                case nn::psc::PmState_FullAwake:
                case nn::psc::PmState_MinimumAwake:
                case nn::psc::PmState_SleepReady:
                    Request::SetPmState(state);
                    break;
                default:
                    break;
                }
                m_PmModule.Acknowledge(state, ResultSuccess());
                break;

            default:
                NN_UNEXPECTED_DEFAULT;
            }
        }
    }

    static void SessionThreadEntry(void* arg)
    NN_NOEXCEPT
    {
        FgmService* p = (FgmService*)arg;
        p->FgmServerThread();
    }

    nn::Result OnNeedsToAccept(int portIndex, PortObjectImpl* pPort)
    NN_NOEXCEPT
    {
        if (portIndex == SessionPort_Debug)
        {
            auto interface = nn::sf::ObjectFactory<
                              nn::sf::ExpHeapAllocator::Policy
                             >::CreateSharedEmplaced<
                              nn::fgm::sf::IDebugger,
                              Debugger
                             >(&g_ServiceFrameworkAllocator);

            return this->AcceptImpl(pPort, interface);
        }
        else if (portIndex >= 0 && portIndex < NumberOfPriorities)
        {
            auto interface = nn::sf::ObjectFactory<
                              nn::sf::ExpHeapAllocator::Policy
                             >::CreateSharedEmplaced<
                              nn::fgm::sf::ISession,
                              SessionManager
                             >(&g_ServiceFrameworkAllocator, static_cast<Priority>(portIndex));

            return this->AcceptImpl(pPort, interface);
        }

        return nn::fgm::ResultNotSupported();
    }

public:
    FgmService()
    NN_NOEXCEPT :
        m_pDependencyList(nullptr),
        m_DependencyCount(0)
    {

    }

    void WaitExit()
    NN_NOEXCEPT
    {
        nn::os::WaitThread(&m_SessionThread);
    }

    nn::Result Initialize()
    NN_NOEXCEPT
    {
        nn::Result result;

        // HW specific code, registers PM resources with FGM
        result = detail::Initialize(&m_pDependencyList, &m_DependencyCount);
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);

        for (int i = 0; i < NumberOfPriorities; i++)
        {
            result = this->InitializePort(i, SessionCount[i], PortName[i]);
            NN_ABORT_UNLESS_RESULT_SUCCESS(result);
        }

        #if defined(NN_SDK_BUILD_DEBUG) || defined(NN_SDK_BUILD_DEVELOP)
        result = this->InitializePort(SessionPort_Debug, SessionCount_Debugger, FgmDebugPortName);
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);
        #endif

        Start();
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateThread(&m_SessionThread, SessionThreadEntry, this, m_SessionThreadStack, SessionThreadStackSize, SessionThreadPriority));
        nn::os::SetThreadNamePointer(&m_SessionThread, NN_SYSTEM_THREAD_NAME(fgm, IpcServer));
        nn::os::StartThread(&m_SessionThread);

        return ResultSuccess();
    }

} g_FgmServer;

} // namespace

nn::Result Initialize(void* pFgmMemoryIn, size_t fgmMemorySize)
NN_NOEXCEPT
{
    g_HeapHandle = nn::lmem::CreateExpHeap(
                            reinterpret_cast<uint8_t*>(pFgmMemoryIn),
                            fgmMemorySize,
                            nn::lmem::CreationOption_ThreadSafe);

    NN_ABORT_UNLESS(g_HeapHandle != nullptr);

    g_ServiceFrameworkAllocator.Attach(g_HeapHandle);

    return g_FgmServer.Initialize();
}

void Wait()
NN_NOEXCEPT
{
    g_FgmServer.WaitExit();
}

}}}
