﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/os.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/os/os_TransferMemory.h>

#include <nn/sf/sf_HipcClient.h>
#include <nn/sf/sf_ExpHeapAllocator.h>

#include <nn/fgm/sfdl/fgm.sfdl.h>
#include <nn/fgm/fgm_Debugger.h>

namespace nn   {
namespace fgm  {
namespace      {

const int SessionHeapSize = (16 * 1024);

class InitExpHeap
{
public:
    InitExpHeap()
    NN_NOEXCEPT
    {
        nn::sf::ExpHeapStaticAllocator
                <SessionHeapSize, Debugger>::
                Initialize(nn::lmem::CreationOption_ThreadSafe);
    }
} g_InitExpHeap;

} // namespace

nn::Result Debugger::Initialize(void* pMemoryPool, size_t memoryPoolSize, nn::os::EventClearMode eventClearMode)
NN_NOEXCEPT
{
    nn::Result result;
    nn::sf::NativeHandle sfHandle;
    nn::os::NativeHandle memoryHandle;
    nn::os::TransferMemoryType transferMemory;

    NN_ABORT_UNLESS(m_DebuggerImpl == nullptr);

    result = nn::sf::CreateHipcProxyByName
                    <nn::fgm::sf::IDebugger,
                     nn::sf::ExpHeapStaticAllocator
                    <SessionHeapSize, Debugger>::Policy>
                    (&m_DebuggerImpl, FgmDebugPortName);

    if (result.IsFailure())
    {
        return result;
    }

    nn::os::CreateTransferMemory(
                    &transferMemory,
                    pMemoryPool,
                    memoryPoolSize,
                    nn::os::MemoryPermission_None);

    memoryHandle = nn::os::DetachTransferMemory(&transferMemory);

    result = m_DebuggerImpl->Initialize(
                    &sfHandle,
                    nn::sf::NativeHandle(memoryHandle, true),
                    memoryPoolSize);

    if (result.IsSuccess())
    {
        m_SystemEvent.AttachReadableHandle(sfHandle.GetOsHandle(), sfHandle.IsManaged(), eventClearMode);
        sfHandle.Detach();
    }

    return result;
}

nn::Result Debugger::Initialize(void* pMemoryPool, size_t memoryPoolSize)
NN_NOEXCEPT
{
    return Initialize(pMemoryPool, memoryPoolSize, nn::os::EventClearMode_ManualClear);
}

nn::Result Debugger::Finalize()
NN_NOEXCEPT
{
    nn::os::DestroySystemEvent(m_SystemEvent.GetBase());
    m_DebuggerImpl = nullptr;
    return ResultSuccess();
}

nn::Result Debugger::Read(DebugEntry* pEntries, uint32_t count, uint32_t* pRead, uint32_t* pLost, uint32_t* pRemaining)
NN_NOEXCEPT
{
    NN_ABORT_UNLESS(m_DebuggerImpl != nullptr);
    nn::sf::OutBuffer buffer(reinterpret_cast<char*>(pEntries), count * sizeof(DebugEntry));
    return m_DebuggerImpl->Read(buffer, pRead, pLost, pRemaining);
}

void Debugger::ClearEvent()
NN_NOEXCEPT
{
    m_SystemEvent.Clear();
}

nn::os::SystemEvent* Debugger::GetSystemEventPointer()
NN_NOEXCEPT
{
    NN_ABORT_UNLESS(m_DebuggerImpl != nullptr);
    return &m_SystemEvent;
}

nn::Result Debugger::Wait()
NN_NOEXCEPT
{
    NN_ABORT_UNLESS(m_DebuggerImpl != nullptr);
    m_SystemEvent.Wait();
    m_SystemEvent.Clear();
    return ResultSuccess();
}

nn::Result Debugger::WaitWithTimeout(nn::TimeSpan timeout)
NN_NOEXCEPT
{
    NN_ABORT_UNLESS(m_DebuggerImpl != nullptr);

    if (m_SystemEvent.TimedWait(timeout))
    {
        m_SystemEvent.Clear();
        return ResultSuccess();
    }

    return nn::fgm::ResultTimeout();
}

nn::Result Debugger::Cancel() NN_NOEXCEPT
{
    NN_ABORT_UNLESS(m_DebuggerImpl != nullptr);
    return m_DebuggerImpl->Cancel();
}

Debugger::Debugger()
NN_NOEXCEPT : m_DebuggerImpl(nullptr)
{

}

Debugger::~Debugger()
NN_NOEXCEPT
{
    m_DebuggerImpl = nullptr;
}

}}

