﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/os.h>
#include <nn/os/os_SystemEvent.h>

#include <nn/sf/sf_ShimLibraryUtility.h>
#include <nn/util/util_FormatString.h>

#include <nn/fgm.h>
#include <nn/fgm/sfdl/fgm.sfdl.h>

#include "fgm_SessionManager.h"

namespace nn   {
namespace fgm  {
namespace      {

// sf subdomain manager for each port
SessionManager sessionManager[NumberOfPriorities] =
{
    {nn::fgm::PortName[0]},
    {nn::fgm::PortName[1]},
    {nn::fgm::PortName[2]}
};

} // namespace

nn::Result Request::Initialize(Module id, Priority priority, nn::os::EventClearMode eventClearMode)
NN_NOEXCEPT
{
    nn::Result result;
    nn::sf::NativeHandle sfHandle;
    nn::sf::SharedPointer<nn::fgm::sf::ISession> sessionObject;

    if (m_RequestImpl != nullptr)
    {
        return nn::fgm::ResultAlreadyInitialized();
    }

    // Convert Priority to port index
    int portIndex = -1;
    for (int i = 0; i < NumberOfPriorities; i++)
    {
        if (nn::fgm::FgmPortPriority[i] == static_cast<int>(priority))
        {
            portIndex = i;
            break;
        }
    }
    if (portIndex < 0)
    {
        return nn::fgm::ResultInvalidPriority();
    }

    if ((result = sessionManager[portIndex].GetObject(&sessionObject)).IsSuccess())
    {
        if ((result = sessionObject->Initialize(&m_RequestImpl)).IsSuccess() &&
            (result = m_RequestImpl->Initialize(&sfHandle, id, 0)).IsSuccess())
        {
            m_SystemEvent.AttachReadableHandle(sfHandle.GetOsHandle(), sfHandle.IsManaged(), eventClearMode);
            sfHandle.Detach();
            m_PortIndex = portIndex;
        }
        else
        {
            m_RequestImpl = nullptr;
            sessionObject = nullptr;
            sessionManager[portIndex].Release();
        }
    }

    return result;
}

nn::Result Request::Initialize(Module id, Priority priority)
NN_NOEXCEPT
{
    return Initialize(id, priority, nn::os::EventClearMode_ManualClear);
}

nn::Result Request::Finalize()
NN_NOEXCEPT
{
    if (m_RequestImpl != nullptr)
    {
        m_RequestImpl = nullptr;
        nn::os::DestroySystemEvent(m_SystemEvent.GetBase());
        sessionManager[m_PortIndex].Release();
    }

    return ResultSuccess();
}

nn::Result Request::Set(Setting min, Setting max)
NN_NOEXCEPT
{
    if (m_RequestImpl == nullptr)
    {
        return nn::fgm::ResultNotInitialized();
    }

    m_SystemEvent.Clear();

    return m_RequestImpl->Set(min, max);
}

nn::Result Request::SetAndWait(Setting min, Setting max)
NN_NOEXCEPT
{
    nn::Result result;

    if (m_RequestImpl == nullptr)
    {
        return nn::fgm::ResultNotInitialized();
    }

    if ((result = Set(min, max)).IsSuccess())
    {
        result = Wait();
    }

    return result;
}

nn::Result Request::SetAndWaitWithTimeout(Setting min, Setting max, nn::TimeSpan timeout)
NN_NOEXCEPT
{
    nn::Result result;

    if (m_RequestImpl == nullptr)
    {
        return nn::fgm::ResultNotInitialized();
    }

    if ((result = Set(min, max)).IsSuccess())
    {
        result = WaitWithTimeout(timeout);
    }

    return result;
}

nn::Result Request::Get(Setting* pCurrent)
NN_NOEXCEPT
{
    if (m_RequestImpl == nullptr)
    {
        return nn::fgm::ResultNotInitialized();
    }

    return m_RequestImpl->Get(pCurrent);
}

nn::Result Request::Wait()
NN_NOEXCEPT
{
    if (m_RequestImpl == nullptr)
    {
        return nn::fgm::ResultNotInitialized();
    }

    m_SystemEvent.Wait();
    m_SystemEvent.Clear();

    return ResultSuccess();
}

nn::Result Request::WaitWithTimeout(nn::TimeSpan timeout)
NN_NOEXCEPT
{
    if (m_RequestImpl == nullptr)
    {
        return nn::fgm::ResultNotInitialized();
    }

    if (m_SystemEvent.TimedWait(timeout))
    {
        m_SystemEvent.Clear();
        return ResultSuccess();
    }

    return nn::fgm::ResultTimeout();
}

nn::os::SystemEvent* Request::GetSystemEventPointer()
NN_NOEXCEPT
{
    return &m_SystemEvent;
}

void Request::ClearEvent()
NN_NOEXCEPT
{
    if (m_RequestImpl != nullptr)
    {
        m_SystemEvent.Clear();
    }
}

nn::Result Request::Cancel()
NN_NOEXCEPT
{
    if (m_RequestImpl == nullptr)
    {
        return nn::fgm::ResultNotInitialized();
    }

    return m_RequestImpl->Cancel();
}

Request::Request()
NN_NOEXCEPT :
    m_RequestImpl(nullptr),
    m_PortIndex(0),
    m_Reserved(0)
{

}

Request::~Request()
NN_NOEXCEPT
{
    Finalize();
}

}}

