﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/


#include <nn/nn_Common.h>
#include <nn/fatalsrv/fatalsrv_Service.h>
#include <nn/fatalsrv/fatalsrv_Server.h>
#include <nn/fatalsrv/fatalsrv_ServiceName.h>
#include <nn/sf/sf_HipcSimpleAllInOneServerManager.h>
#include <nn/sf/sf_ObjectFactory.h>
#include <nn/settings/system/settings_Fatal.h>
#include <nn/nn_SystemThreadDefinition.h>
#include <nn/fatal/detail/fatal_Log.h>
#include <nn/fs/fs_MemoryManagement.h>

#include "fatalsrv_Environment.h"
#include "fatalsrv_Settings.h"
#include "fatalsrv_Font.h"
#include "fatalsrv_MessageManager.h"
#include "fatalsrv_Memory.h"
#include "fatalsrv_Repair.h"

namespace nn { namespace fatalsrv {
    namespace {
        class ServerManager : public nn::sf::HipcSimpleAllInOneServerManager<5, 2>
        {
        };

        std::aligned_storage<sizeof(ServerManager), NN_ALIGNOF(ServerManager)>::type g_ServerManagerStorage;
        ServerManager* g_pServerManager;

        const size_t ServerStackSize = 1024 * 32;
        NN_OS_ALIGNAS_THREAD_STACK Bit8 g_ServerStack[ServerStackSize];
        nn::os::ThreadType g_ServerThread;

        fatalsrv::Service g_ServiceImpl;
        nn::sf::UnmanagedServiceObjectByPointer<nn::fatalsrv::IService, nn::fatalsrv::Service> g_ServiceObject(&g_ServiceImpl);
        nn::sf::UnmanagedServiceObjectByPointer<nn::fatalsrv::IPrivateService, nn::fatalsrv::Service> g_PrivateServiceObject(&g_ServiceImpl);

        SettingsEventManager g_SettingsEventManager;

        void IpcDispatcher(void*) NN_NOEXCEPT
        {
            NN_DETAIL_FATAL_TRACE("IpcDispatcher running.\n");

            NN_DETAIL_FATAL_TRACE("Initialize environment info.\n");
            InitializeEnvironmentInfo();

            NN_DETAIL_FATAL_TRACE("Initialize font.\n");
            InitializeSharedFont();

            nn::fs::SetAllocator(AllocateForFs, FreeForFs);
            NN_DETAIL_FATAL_TRACE("Load text data.\n");
            NN_ABORT_UNLESS_RESULT_SUCCESS(InitializeTextData());

            ProcessRepairStatus(&g_ServiceImpl);

            while (auto p = g_pServerManager->Wait())
            {
                auto userData = nn::os::GetMultiWaitHolderUserData(p);

                switch (userData)
                {
                case ServerManager::InvokeTag:
                case ServerManager::AcceptTag:
                    {
                        NN_ABORT_UNLESS_RESULT_SUCCESS(
                            g_pServerManager->ProcessAuto(p));
                    }
                    break;

                default:
                    {
                        auto waitHolder = reinterpret_cast<nn::os::MultiWaitHolderType*>(userData);
                        if (waitHolder == g_SettingsEventManager.GetEventWaitHolder())
                        {
                            g_SettingsEventManager.OnSettingsUpdated();
                        }
                        g_pServerManager->AddUserWaitHolder(waitHolder);
                    }
                    break;
                }
            }
        }

        void InitializeServer() NN_NOEXCEPT
        {
            NN_SDK_ASSERT(!g_pServerManager);

            g_pServerManager = new (&g_ServerManagerStorage) ServerManager;

            auto sessionCountMax = 4;
            NN_ABORT_UNLESS_RESULT_SUCCESS(g_pServerManager->RegisterObjectForPort(
                g_ServiceObject.GetShared(), sessionCountMax, ServicePortName));

            auto privateSessionCountMax = 4;
            NN_ABORT_UNLESS_RESULT_SUCCESS(g_pServerManager->RegisterObjectForPort(
                g_PrivateServiceObject.GetShared(), privateSessionCountMax, PrivateServicePortName));

            g_pServerManager->AddUserWaitHolder(g_SettingsEventManager.GetEventWaitHolder());
            g_pServerManager->Start();
        }
    }

    void RegisterService() NN_NOEXCEPT
    {
        InitializeServer();
    }

    void StartServer() NN_NOEXCEPT
    {
        NN_SDK_ASSERT(g_pServerManager);
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateThread(
            &g_ServerThread,
            IpcDispatcher,
            nullptr,
            g_ServerStack,
            ServerStackSize,
            NN_SYSTEM_THREAD_PRIORITY(fatalsrv, IpcDispatcher)));

        nn::os::SetThreadNamePointer(&g_ServerThread, NN_SYSTEM_THREAD_NAME(fatalsrv, IpcDispatcher));
        nn::os::StartThread(&g_ServerThread);
    }

    void StopServer() NN_NOEXCEPT
    {
        g_pServerManager->RequestStop();

        nn::os::WaitThread(&g_ServerThread);
        nn::os::DestroyThread(&g_ServerThread);
        g_pServerManager->~ServerManager();
        g_pServerManager = nullptr;
    }

    void WaitServerStop() NN_NOEXCEPT
    {
        nn::os::WaitThread(&g_ServerThread);
    }

}} // namespace nn::fatalsrv
