﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <nn/util/util_ScopeExit.h>
#include <nn/gpio/gpio.h>
#include <nn/settings/system/settings_Boot.h>
#include <nn/nn_SdkLog.h>
#include <nn/os/os_Result.h>
#include <nn/svc/svc_Base.h>
#include <nn/fatal/fatal_Result.private.h>
#include <nn/os/os_Tick.h>
#include <nn/os/os_Thread.h>
#include <nn/nn_TimeSpan.h>
#include <nn/fs/fs_GameCard.h>

#include "fatalsrv_Repair.h"

namespace nn { namespace fatalsrv {

nn::Result IsRepairTimeReviserCardInserted()
{
    if ( !nn::fs::IsGameCardInserted() )
    {
        return nn::fatal::ResultInRepairProcessWithoutRunningRepairTimeReviser();
    }

    nn::fs::GameCardHandle handle;
    NN_RESULT_DO( nn::fs::GetGameCardHandle( &handle ) );
    nn::fs::GameCardAttribute attribute;
    NN_RESULT_DO( nn::fs::GetGameCardAttribute( &attribute, handle ) );
    if ( ( static_cast<uint8_t>( attribute )
           & static_cast<uint8_t>( nn::fs::GameCardAttribute::RepairToolFlag ) )
         == static_cast<uint8_t>( nn::fs::GameCardAttribute::RepairToolFlag ) )
    {
        NN_RESULT_SUCCESS;
    }

    return nn::fatal::ResultInRepairProcessWithoutRunningRepairTimeReviser();
}

bool IsRepairInProgressWithoutRunningRepairTimeReviser()
{
    if (::nn::settings::system::RequiresRunRepairTimeReviser())
    {
        if(IsRepairTimeReviserCardInserted().IsFailure())
        {
            return true;
        }
    }
    return false;
}

bool IsRepairInProgressWithoutSpecialInput()
{
    // 修理中かチェック
    if (::nn::settings::system::IsInRepairProcess())
    {
        gpio::Initialize();
        NN_UTIL_SCOPE_EXIT { gpio::Finalize(); };

        gpio::GpioPadSession volUp;
        gpio::OpenSession(&volUp, gpio::GpioPadName_ButtonVolUp);
        NN_UTIL_SCOPE_EXIT { gpio::CloseSession(&volUp); };
        gpio::SetDirection(&volUp, gpio::Direction_Input);

        auto start = nn::os::GetSystemTick();
        do
        {
            if (gpio::GetValue(&volUp) == gpio::GpioValue_Low)
            {
                NN_SDK_LOG("[fatal] vol is pressed\n");
            }
            else
            {
                NN_SDK_LOG("[fatal] vol isn't pressed\n");
                return true;
            }
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));
        } while (nn::os::ConvertToTimeSpan(nn::os::GetSystemTick() - start) < nn::TimeSpan::FromSeconds(1));
    }
    return false;
}

void ProcessRepairStatus(fatalsrv::Service* service)
{
    if(IsRepairInProgressWithoutSpecialInput())
    {
        fatalsrv::FatalContext fatalContext;
        fatalContext.lastResult = nn::fatal::ResultInRepairProcessWithoutSpecialInput();
        fatalsrv::CpuContext cpuContext;
        cpuContext.Clear();
        Bit64 processId = 0;
        nn::svc::GetProcessId(&processId, svc::PSEUDO_HANDLE_CURRENT_PROCESS);

        service->ThrowFatalWithCpuContext(fatalContext, processId, nn::fatal::FatalPolicy_WithoutErrorReport, cpuContext);
    }
    if(IsRepairInProgressWithoutRunningRepairTimeReviser())
    {
        fatalsrv::FatalContext fatalContext;
        fatalContext.lastResult = nn::fatal::ResultInRepairProcessWithoutRunningRepairTimeReviser();
        fatalsrv::CpuContext cpuContext;
        cpuContext.Clear();
        Bit64 processId = 0;
        nn::svc::GetProcessId(&processId, svc::PSEUDO_HANDLE_CURRENT_PROCESS);

        service->ThrowFatalWithCpuContext(fatalContext, processId, nn::fatal::FatalPolicy_WithoutErrorReport, cpuContext);
    }
}

}}  // namespace nn::fatalsrv
