﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/fatal/fatal_Result.h>
#include <nn/fatal/detail/fatal_Log.h>
#include <nn/fs/fs_File.h>
#include <nn/fs/fs_FileSystem.h>
#include <nn/fs/fs_Mount.h>
#include <nn/fs/fs_SystemData.h>
#include <nn/ncm/ncm_ProgramId.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/settings/system/settings_SystemApplication.h>
#include <nn/util/util_FormatString.h>
#include <nn/util/util_ScopeExit.h>

#include "fatalsrv_Environment.h"
#include "fatalsrv_Memory.h"
#include "fatalsrv_MessageManager.h"

#include <memory>

namespace nn { namespace fatalsrv {
    namespace
    {
        // INFO: http://spdlybra.nintendo.co.jp/jira/browse/SWHUID-520
        const size_t ErrorCodeTextSize = 40 * 3;
        const size_t MessageTextSize   = 31 * 8 * 3;

        struct TextData
        {
            nn::settings::Language language;
            char errorCodeText[ErrorCodeTextSize + 1];
            char messageText[MessageTextSize + 1];
        };
        NN_STATIC_ASSERT(std::is_pod<TextData>::value);

        const nn::ncm::SystemDataId FatalMessageProgramId = { 0x010000000000081d };
        const char MountName[] = "fatal_msg";

        Result ReadFile(void* out, size_t outBufferSize, const nn::settings::LanguageCode& languageCode, const char* fileName) NN_NOEXCEPT
        {
            nn::fs::FileHandle file;
            char pathBuffer[64] = {};
            util::SNPrintf(pathBuffer, sizeof(pathBuffer), "%s:/%s/%s", MountName, languageCode.string, fileName);

            NN_RESULT_DO(nn::fs::OpenFile(&file, pathBuffer, fs::OpenMode_Read));
            NN_UTIL_SCOPE_EXIT{ nn::fs::CloseFile(file); };

            int64_t fileSize;
            NN_RESULT_DO(nn::fs::GetFileSize(&fileSize, file));
            NN_RESULT_THROW_UNLESS(static_cast<int64_t>(outBufferSize) >= fileSize, fatal::ResultAllocationFailed());
            NN_RESULT_DO(nn::fs::ReadFile(file, 0, out, static_cast<size_t>(fileSize)));

            NN_RESULT_SUCCESS;
        }

        const char* GetMessageFileName(bool questFlag) NN_NOEXCEPT
        {
            return questFlag ? "QuestMessage" : "GeneralMessage";
        }

        Result ReadMessageAll(TextData out[], int languageCount, bool questFlag) NN_NOEXCEPT
        {
            size_t cacheSize;
            NN_RESULT_DO(nn::fs::QueryMountSystemDataCacheSize(&cacheSize, FatalMessageProgramId));
            std::unique_ptr<char[]> mountCache(new char[cacheSize]);
            NN_RESULT_THROW_UNLESS(mountCache != nullptr, fatal::ResultAllocationFailed());

            NN_RESULT_DO(nn::fs::MountSystemData(MountName, FatalMessageProgramId, mountCache.get(), cacheSize));
            NN_UTIL_SCOPE_EXIT{ nn::fs::Unmount(MountName); };

            std::unique_ptr<nn::settings::LanguageCode[]> codes(new nn::settings::LanguageCode[languageCount]);
            NN_RESULT_THROW_UNLESS(codes != nullptr, nn::fatal::ResultAllocationFailed());

            nn::settings::GetAvailableLanguageCodes(codes.get(), languageCount);

            for (int i = 0; i < languageCount; ++i)
            {
                auto& textData = out[i];
                auto& code = codes[i];
                NN_RESULT_DO(ReadFile(textData.errorCodeText, sizeof(textData.errorCodeText), code, "ErrorCode"));
                NN_RESULT_DO(ReadFile(textData.messageText, sizeof(textData.messageText), code, GetMessageFileName(questFlag)));
                textData.language = static_cast<nn::settings::Language>(i);
            }

            NN_RESULT_SUCCESS;
        }

        class MessageManager
        {
        public:
            MessageManager() NN_NOEXCEPT
                : m_LanguageCodeCount(0), m_pTextData(nullptr)
            {}
            Result Initialize() NN_NOEXCEPT
            {
                m_LanguageCodeCount = nn::settings::GetAvailableLanguageCodeCount();
                m_pTextData.reset(new TextData[m_LanguageCodeCount]);
                std::memset(m_pTextData.get(), 0, sizeof(TextData) * m_LanguageCodeCount);

                NN_RESULT_THROW_UNLESS(m_pTextData != nullptr, nn::fatal::ResultAllocationFailed());

                NN_RESULT_DO(ReadMessageAll(m_pTextData.get(), m_LanguageCodeCount, GetEnvironmentInfo().questFlag));

                NN_RESULT_SUCCESS;
            }
            const char* GetErrorCodeLabel(const nn::settings::LanguageCode& languageCode) NN_NOEXCEPT
            {
                auto index = FindIndex(languageCode);
                return index >= 0 ? m_pTextData.get()[index].errorCodeText : nullptr;
            }
            const char* GetGeneralMessage(const nn::settings::LanguageCode& languageCode) NN_NOEXCEPT
            {
                auto index = FindIndex(languageCode);
                return index >= 0 ? m_pTextData.get()[index].messageText : nullptr;
            }
        private:
            int FindIndex(const nn::settings::LanguageCode& languageCode)
            {
                for (int i = 0; i < m_LanguageCodeCount; ++i)
                {
                    auto language = static_cast<nn::settings::Language>(i);
                    if (language == languageCode)
                    {
                        return i;
                    }
                }
                return -1;
            }

            int m_LanguageCodeCount;
            std::unique_ptr<TextData> m_pTextData;
        };

        MessageManager g_MessageManager;
    } // namespace


    Result InitializeTextData() NN_NOEXCEPT
    {
        NN_RESULT_DO(g_MessageManager.Initialize());
        NN_RESULT_SUCCESS;
    }

    const char* GetErrorCodeLabel(const nn::settings::LanguageCode& languageCode) NN_NOEXCEPT
    {
        return g_MessageManager.GetErrorCodeLabel(languageCode);
    }

    const char* GetMessage(const nn::settings::LanguageCode& languageCode, const nn::err::ErrorCode& errorCode) NN_NOEXCEPT
    {
        NN_UNUSED(errorCode);

        return g_MessageManager.GetGeneralMessage(languageCode);
    }
}} // namespace nn::fatalsrv
