﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <nn/mem/mem_StandardAllocator.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_SdkLog.h>

#include "fatalsrv_Environment.h"
#include "fatalsrv_Memory.h"

//#define NN_FATALSRV_MEASURE_HEAP_USAGE


namespace nn { namespace fatalsrv {

#if defined(NN_BUILD_CONFIG_COMPILER_VC)
#define NN_FATALSRV_ALIGNED_ALLOC(size, align) _aligned_malloc((size), (align))
#else
#define NN_FATALSRV_ALIGNED_ALLOC(size, align) std::aligned_alloc((align), (size))
#endif

    void* Allocate(size_t size, size_t alignment, void* userPtr) NN_NOEXCEPT
    {
        NN_UNUSED(userPtr);
        return NN_FATALSRV_ALIGNED_ALLOC(size, alignment);
    }

    void Free(void* ptr, void* userPtr) NN_NOEXCEPT
    {
        NN_UNUSED(userPtr);
        std::free(ptr);
    }

    void* Reallocate(void* ptr, size_t size, void* userPtr) NN_NOEXCEPT
    {
        NN_UNUSED(userPtr);
        return std::realloc(ptr, size);
    }

    void*   AllocateForFs(size_t size) NN_NOEXCEPT
    {
        return Allocate(size, 4, nullptr);
    }

    void    FreeForFs(void* ptr, size_t size) NN_NOEXCEPT
    {
        NN_UNUSED(size);
        Free(ptr, nullptr);
    }

#if defined(NN_FATALSRV_MEASURE_HEAP_USAGE)

    size_t GetTotalAlloc() NN_NOEXCEPT
    {
        size_t totalAlloc = 0;

        g_Allocator.WalkAllocatedBlocks([](void* addr, size_t size, void* param) {
            *reinterpret_cast<size_t*>(param) += size;
            return 1;
        }, &totalAlloc);

        return totalAlloc;
    }

    void DumpHeapState(const char* title) NN_NOEXCEPT
    {
        NN_SDK_LOG("[fatalsrv] Allocation state: %s\n", title);
        NN_SDK_LOG("[fatalsrv]   - Total allocation: %lld\n", GetTotalAlloc());
    }
#else

    void DumpHeapState(const char*) NN_NOEXCEPT
    {
    }

#endif
}} // namespace nn::fatalsrv
