﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <cstring>
#include <nn/erpt.h>
#include <nn/err/detail/err_ErrorCodeConvert.h>
#include <nn/err/err_Result.h>
#include <nn/fatal/detail/fatal_Log.h>
#include <nn/fatalsrv/fatalsrv_Service.h>
#include <nn/fs/fs_ErrorReport.h>
#include <nn/ncm/ncm_ProgramId.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_FormatString.h>

#include "fatalsrv_ErrorReport.h"

namespace nn { namespace fatalsrv {
    namespace
    {
#if defined(NN_BUILD_CONFIG_OS_HORIZON)
        template<typename PtrType>
        Result WriteBacktrace(
            nn::erpt::Context& erptContext,
            nn::erpt::FieldId fieldId,
            const PtrType* srcBacktrace,
            int32_t srcBacktraceCount,
            PtrType programMappedAddr) NN_NOEXCEPT
        {
            const int MaxBacktraceCount = fatalsrv::Aarch64Context::MaxBacktraceCount;
            PtrType backtrace[MaxBacktraceCount];
            auto backtraceCount = static_cast<uint32_t>(std::min(srcBacktraceCount, MaxBacktraceCount));

            for (int i = 0; i < backtraceCount; ++i)
            {
                backtrace[i] = srcBacktrace[i] - programMappedAddr;
            }

            NN_RESULT_DO(
                erptContext.Add(fieldId, backtrace, backtraceCount)
            );

            NN_RESULT_SUCCESS;
        }

        template<typename ContextType, typename ValueType>
        Result WriteCpuContextImpl(
            erpt::Context& erptContext,
            const ContextType& context,
            erpt::FieldId backtraceId,
            erpt::FieldId registerId,
            erpt::FieldId flagId,
            erpt::FieldId programMappedAddrId) NN_NOEXCEPT
        {
            NN_RESULT_DO(
                erptContext.Add(backtraceId, context.backtrace, context.backtraceCount));

            NN_RESULT_DO(
                erptContext.Add(registerId, context.values, sizeof(context.values) / sizeof(ValueType)));

            NN_RESULT_DO(
                erptContext.Add(flagId, context.setFlag._storage[0])
            );

            NN_RESULT_DO(
                erptContext.Add(programMappedAddrId, context.programMappedAddr)
            );

            NN_RESULT_SUCCESS;
        }

        Result WriteCpuContext(nn::erpt::Context& erptContext, const CpuContext& cpuContext) NN_NOEXCEPT
        {
            if (cpuContext.archType == CpuContext::ArchType::Aarch64)
            {
                NN_RESULT_DO(
                    (WriteCpuContextImpl<Aarch64Context, nn::Bit64>(
                        erptContext, cpuContext.aarch64,
                        erpt::FieldId::StackBacktrace64,
                        erpt::FieldId::GeneralRegisterAarch64,
                        erpt::FieldId::RegisterSetFlag64,
                        erpt::FieldId::ProgramMappedAddr64)));
            }
            else if(cpuContext.archType == CpuContext::ArchType::Aarch32)
            {
                NN_RESULT_DO(
                    (WriteCpuContextImpl<Aarch32Context, nn::Bit32>(
                        erptContext, cpuContext.aarch32,
                        erpt::FieldId::StackBacktrace32,
                        erpt::FieldId::GeneralRegisterAarch32,
                        erpt::FieldId::RegisterSetFlag32,
                        erpt::FieldId::ProgramMappedAddr32)));
            }
            NN_RESULT_SUCCESS;
        }
#else
        Result WriteCpuContext(nn::erpt::Context& erptContext, const CpuContext& cpuContext) NN_NOEXCEPT
        {
            NN_UNUSED(erptContext);
            NN_UNUSED(cpuContext);
            NN_RESULT_SUCCESS;
        }
#endif

        class WriteErrorReportTask : public ITask
        {
        public:
            WriteErrorReportTask() NN_NOEXCEPT :
                m_Context(nullptr), m_WrittenEvent(nullptr), m_NeedsErrorReport(false), m_Id(0)
            {}

            void Initialize(const Service::Context& context, Bit64 id, bool needsErrorReport, os::Event* writtenEvent) NN_NOEXCEPT
            {
                m_Context = &context;
                m_Id = id;
                m_NeedsErrorReport = needsErrorReport;
                m_WrittenEvent = writtenEvent;
            }

            virtual Result Run() NN_NOEXCEPT
            {
                NN_SDK_ASSERT(m_Context);
                NN_SDK_ASSERT(m_WrittenEvent);

                if (m_NeedsErrorReport)
                {
                    NN_RESULT_DO(WriteErrorReport(m_Context->fatalContext, m_Context->cpuContext, m_Id));
                }
                m_WrittenEvent->Signal();
                NN_RESULT_SUCCESS;
            }
            virtual const char* GetTaskName() const
            {
                return "WriteErrorReport";
            }
            virtual size_t GetRequiredStackSize() const
            {
                return 8192;
            }
        private:
            const Service::Context* m_Context;
            os::Event* m_WrittenEvent;
            bool m_NeedsErrorReport;
            Bit64 m_Id;
        };

        WriteErrorReportTask g_WriteErrorReportTask;
    }// namespace

    nn::err::ErrorCode CreateErrorCodeFromResult(const Result& result) NN_NOEXCEPT
    {
        auto resultToConvert = result.IsSuccess() ? nn::err::ResultSystemProgramAbort() : result;
        return nn::err::detail::ConvertResultToErrorCode(resultToConvert);
    }

    Result WriteErrorReport(const fatalsrv::FatalContext& fatalContext, const fatalsrv::CpuContext& cpuContext, nn::Bit64 programId) NN_NOEXCEPT
    {
        auto errorCode = CreateErrorCodeFromResult(fatalContext.lastResult);
        char errorCodeString[nn::err::ErrorCode::StringLengthMax];
        nn::util::SNPrintf(errorCodeString, sizeof(errorCodeString), "%04d-%04d", errorCode.category, errorCode.number);

        char programIdString[17];
        nn::util::SNPrintf(programIdString, sizeof(programIdString), "%016llx", programId);

        const int ErptContextBufferSize = 768;
        uint8_t erptContextBuffer[ErptContextBufferSize];
        nn::erpt::Context abortContext(nn::erpt::CategoryId::ErrorInfo, erptContextBuffer, ErptContextBufferSize);

        NN_RESULT_DO(
            abortContext.Add(nn::erpt::FieldId::ProgramId, programIdString, static_cast<uint32_t>(sizeof(programIdString))));
        NN_RESULT_DO(
            abortContext.Add(nn::erpt::FieldId::AbortType, cpuContext.exceptionType));
        NN_RESULT_DO(
            abortContext.Add(nn::erpt::FieldId::AbortFlag, true));
        NN_RESULT_DO(
            abortContext.Add(nn::erpt::FieldId::SystemAbortFlag, true));
        NN_RESULT_DO(
            abortContext.Add(nn::erpt::FieldId::ErrorCode, errorCodeString, static_cast<uint32_t>(sizeof(errorCodeString))));
        NN_RESULT_DO(
            abortContext.Add(nn::erpt::FieldId::FatalFlag, true));
        nn::fs::SetErrorReportFileSystemInfo();
        NN_RESULT_DO(
            WriteCpuContext(abortContext, cpuContext));
        NN_RESULT_DO(
            abortContext.CreateReport(nn::erpt::ReportType::ReportType_Visible));

        nn::fs::ClearErrorReportFileSystemInfo();

        NN_DETAIL_FATAL_TRACE("WriteErrorReport Success");

        NN_RESULT_SUCCESS;
    }

    ITask* InitializeAndGetErrorReportTask(const Service::Context& context, Bit64 id, bool needsErrorReport, nn::os::Event* writtenEvent) NN_NOEXCEPT
    {
        g_WriteErrorReportTask.Initialize(context, id, needsErrorReport, writtenEvent);
        return &g_WriteErrorReportTask;
    }
}} // namespace nn::fatalsrv
