﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>

#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_Abort.h>
#include <nn/os.h>

#include <nn/fan/fan_ControlTypes.h>
#include "fan_ControllerResourceManager.h"

namespace {

// ControllerHolder は、Controller が利用するオブジェクトの実体のリソースを保持する構造体

nn::fan::impl::detail::ControllerHolder g_ControllerHolderTable[nn::fan::FanNameCountMax];
nn::os::Mutex                           g_ControllerHolderTableMutex(false);

int FanNameToHolderIndex(nn::fan::FanName name) NN_NOEXCEPT
{
    NN_SDK_ASSERT(static_cast<int>(name) > 0);
    return static_cast<int>(name) - 1;
}

}

namespace nn {
namespace fan {
namespace impl {
namespace detail {

ControllerHolder* AllocControllerHolder(FanName name, SettingsHolder& settingsHolder) NN_NOEXCEPT
{
    // CPU ファン以外は実装が存在しない
    NN_SDK_ASSERT(name == FanName_Cpu);

    std::lock_guard<nn::os::Mutex>  lock(g_ControllerHolderTableMutex);

    ControllerHolder& holder = g_ControllerHolderTable[FanNameToHolderIndex(name)];

    if ( holder.m_RefCount == 0 )
    {
        // DeviceAccessor オブジェクトを構築
#if defined(NN_BUILD_CONFIG_HARDWARE_NX)
        if (settingsHolder.HasFan(name) && settingsHolder.GetFanInterface(name) == SettingsHolder::FanInterface::Pwm)
        {
            new (holder.GetDeviceAccessorHolder()) DeviceAccessorPwm(name);
        }
        else if (settingsHolder.HasFan(name) && settingsHolder.GetFanInterface(name) == SettingsHolder::FanInterface::Gpio)
        {
            new (holder.GetDeviceAccessorHolder()) DeviceAccessorGpio(name);
        }
        else
        {
            new (holder.GetDeviceAccessorHolder()) DeviceAccessorDummy(name);
        }
#else
        new (holder.GetDeviceAccessorHolder()) DeviceAccessorDummy(name);
#endif
    }
    holder.m_RefCount ++;

    return &holder;
}

void FreeControllerHolder(ControllerHolder* pControllerHolder) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex>  lock(g_ControllerHolderTableMutex);

    NN_ABORT_UNLESS(pControllerHolder->m_RefCount > 0);
    pControllerHolder->m_RefCount --;

    if ( pControllerHolder->m_RefCount == 0 )
    {
        // DeviceAccessor オブジェクトを破棄
        static_cast<DeviceAccessor*>(pControllerHolder->GetDeviceAccessorHolder())->~DeviceAccessor();
    }
}

// マルチスレッドを想定して g_ControllerHolderTableMutex のロックを取得するために fan_ControllerResourceManager.cpp を経由する
void NotifyElapsedTime(int64_t elapsedTimeUsec) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex>  lock(g_ControllerHolderTableMutex);

    for ( int index = 0; index < nn::fan::FanNameCountMax; index ++ )
    {
        if ( g_ControllerHolderTable[index].m_RefCount > 0 )
        {
            static_cast<DeviceAccessor*>(g_ControllerHolderTable[index].GetDeviceAccessorHolder())->NotifyElapsedTime(elapsedTimeUsec);
        }
    }
}

void SetFanEnabledFromServer(bool enabled) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex>  lock(g_ControllerHolderTableMutex);

    for ( int index = 0; index < nn::fan::FanNameCountMax; index ++ )
    {
        if ( g_ControllerHolderTable[index].m_RefCount > 0 )
        {
            static_cast<DeviceAccessor*>(g_ControllerHolderTable[index].GetDeviceAccessorHolder())->SetFanEnabledFromServer(enabled);
        }
    }
}

} // detail
} // impl
} // fan
} // nn
