﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>
#include <nn/os.h>
#include <nn/os/os_SdkMutex.h>
#include <nn/util/util_LockGuard.h>
#include <nn/eupld/eupld_Result.h>
#include <nn/erpt/erpt_Report.h>

#include "eupldsrv_Request.h"
#include "eupldsrv_Worker.h"

namespace nn    {
namespace eupld {
namespace srv   {

nn::util::IntrusiveList<Request, nn::util::IntrusiveListBaseNodeTraits<Request>> Request::m_RequestList;
nn::os::SdkMutex Request::m_Lock;

Request* Request::GetRequest()
NN_NOEXCEPT
{
    Request* pRequest = nullptr;

    NN_UTIL_LOCK_GUARD(m_Lock);

    if (!(m_RequestList.empty()))
    {
        pRequest = &m_RequestList.front();
        pRequest->AddReference();
        pRequest->m_State = RequestState_UploadInProgress;
        m_RequestList.pop_front();
    }

    return pRequest;
}

void Request::CompleteRequest(nn::Result result)
NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_Lock);

    m_State  = RequestState_Completed;
    m_Result = result;

    if (m_pEvent != nullptr)
    {
        m_pEvent->Signal();
    }
}

ReportUploadList* Request::GetUploadList()
NN_NOEXCEPT
{
    return &m_ReportList;
}

nn::Result Request::Cancel()
NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_Lock);

    if (m_State == RequestState_UploadInProgress)
    {
        m_State = RequestState_CancelInProgress;
        return nn::eupld::ResultCancelInProgress();
    }
    else if (m_State == RequestState_Scheduled)
    {
        m_RequestList.erase(m_RequestList.iterator_to(*this));
        m_State = RequestState_Completed;
    }

    return ResultSuccess();
}

nn::Result Request::GetResult()
NN_NOEXCEPT
{
    return m_Result;
}

bool Request::IsAuto()
NN_NOEXCEPT
{
    return m_Auto;
}

nn::Result Request::Initialize(const nn::erpt::ReportList& reportList, bool autoUpload)
NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_Lock);

    if (m_State != RequestState_Completed)
    {
        return nn::eupld::ResultAlreadyScheduled();
    }

    m_ReportList.reportCount = 0;

    for (uint32_t i = 0;
                  i < reportList.reportCount;
                  i++)
    {
        if (!(reportList.Report[i].reportFlags.Test<nn::erpt::ReportFlag::Transmitted>()))
        {
            m_ReportList.Report[i].reportId = reportList.Report[i].reportId;
            m_ReportList.Report[i].result   = nn::eupld::ResultReportUploadPending();
            m_ReportList.reportCount++;
        }
    }

    if (m_ReportList.reportCount == 0)
    {
        m_Result = nn::eupld::ResultNoReportsToSend();
        return m_Result;
    }

    m_Auto  = autoUpload;
    m_State = RequestState_Scheduled;
    m_RequestList.push_back(*this);
    g_Worker.SignalNewRequest();

    return ResultSuccess();
}

nn::Result Request::Initialize(const nn::erpt::ReportId* pReportIdArray, uint32_t reportCount, bool autoUpload)
NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_Lock);

    if (m_State != RequestState_Completed)
    {
        return nn::eupld::ResultAlreadyScheduled();
    }

    m_ReportList.reportCount = reportCount;

    for (uint32_t i = 0;
                  i < reportCount;
                  i++)
    {
        m_ReportList.Report[i].reportId = pReportIdArray[i];
        m_ReportList.Report[i].result   = nn::eupld::ResultReportUploadPending();
    }

    m_Auto  = autoUpload;
    m_State = RequestState_Scheduled;
    m_RequestList.push_back(*this);
    g_Worker.SignalNewRequest();

    return ResultSuccess();
}

void Request::Finalize()
NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_Lock);

    if (m_State == RequestState_Scheduled)
    {
        m_RequestList.erase(m_RequestList.iterator_to(*this));
        m_State = RequestState_Completed;
    }
    else if (m_State == RequestState_UploadInProgress)
    {
        m_State = RequestState_CancelInProgress;
    }

    m_pEvent = nullptr;
}

Request::Request(nn::os::SystemEvent* pEvent)
NN_NOEXCEPT :
    m_pEvent(pEvent),
    m_State(RequestState_Completed),
    m_Result(ResultSuccess()),
    m_Auto(true)
{
    m_ReportList.reportCount = 0;
}

Request::~Request()
NN_NOEXCEPT
{
    Finalize();
}

}}}
