﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/os.h>
#include <nn/nn_SdkLog.h>
#include <nn/eupld/eupld_Result.h>

#include "eupldsrv_ReportStream.h"

namespace nn    {
namespace eupld {
namespace srv   {

nn::Result ReportStream::Read(uint32_t* pReadCountOut, uint8_t* pBuffer, uint32_t size)
NN_NOEXCEPT
{
    uint32_t readCount;

    *pReadCountOut = 0;

    // Batch array header.
    if (m_AddHeader)
    {
        if (size < sizeof(ArrayHeader))
        {
            return nn::eupld::ResultTransportFailed();
        }

        pBuffer[0]      = Tag_Array16384;
        pBuffer[1]      = 0;
        pBuffer[2]      = static_cast<uint8_t>(m_ReportCount);
        pBuffer        += sizeof(ArrayHeader);
        *pReadCountOut += sizeof(ArrayHeader);
        size           -= sizeof(ArrayHeader);
        m_AddHeader     = false;
    }

    // Data from all reports.
    while (size > 0 && m_CurrentReportIndex >= 0)
    {
        if ((m_Result = m_Report[m_CurrentReportIndex].Read(&readCount, pBuffer, size)).IsFailure())
        {
            m_Report[m_CurrentReportIndex].Close();
            m_pList->Report[m_CurrentReportIndex].result = m_Result;
            return m_Result;
        }
        if (readCount == 0) // EOF
        {
            // Reports are being sent in reverse order - see SIGLO-38831.
            while (--m_CurrentReportIndex >= 0)
            {
                if (m_pList->Report[m_CurrentReportIndex].result.IsSuccess())
                {
                    break;
                }
            }
        }
        else
        {
            *pReadCountOut += readCount;
            pBuffer        += readCount;
            size           -= readCount;
        }
    }

    return ResultSuccess();
}

uint32_t ReportStream::GetStreamSize()
NN_NOEXCEPT
{
    return m_TotalSize;
}

ReportStream::ReportStream(ReportUploadList* pList, bool autoUpload)
NN_NOEXCEPT :
    m_pList(pList),
    m_CurrentReportIndex(0),
    m_Result(ResultSuccess()),
    m_TotalSize(0),
    m_ReportCount(0),
    m_AddHeader(false),
    m_AutoUpload(autoUpload)
{
    for (uint32_t i = 0;
                  i < m_pList->reportCount;
                  i++)
    {
        int64_t    size;
        nn::Result result;

        // It's possible that report operations may fail,
        // for example, due to fs corruption. Rather than
        // aborting transmition, skip failed report and at
        // least try sending all other reports.
        if ((result = m_Report[i].Open(m_pList->Report[i].reportId)).IsFailure() ||
            (result = m_Report[i].GetSize(&size)).IsFailure())
        {
            m_pList->Report[i].result = result;
            m_Report[i].Close();
            continue;
        }

        m_pList->Report[i].result = result;
        m_TotalSize += static_cast<uint32_t>(size);
        m_ReportCount++;

        if (m_ReportCount == 1)
        {
            // Successfully opened at least one report.
            m_AddHeader          = true;
            m_TotalSize         += sizeof(ArrayHeader);
        }

        // Reports are being sent in reverse order - see SIGLO-38831.
        m_CurrentReportIndex = i;
    }
}

void ReportStream::MarkAsTransmitted()
NN_NOEXCEPT
{
    if (m_AutoUpload)
    {
        nn::erpt::ReportFlagSet flags;

        flags.Set<nn::erpt::ReportFlag::Transmitted>(1);

        for (uint32_t i = 0;
                      i < m_pList->reportCount;
                      i++)
        {
            if (m_pList->Report[i].result.IsSuccess())
            {
                if (m_Result.IsSuccess())
                {
                    m_Report[i].SetFlags(flags);
                }
            }
        }
    }
}

ReportStream::~ReportStream()
NN_NOEXCEPT
{
    for (uint32_t i = 0;
                  i < m_pList->reportCount;
                  i++)
    {
        if (m_pList->Report[i].result.IsSuccess())
        {
            m_Report[i].Close();
        }
    }
}

}}}
