﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_StaticAssert.h>
#include <nn/es/es_Types.h>
#include <nn/es/es_ELicenseTypes.h>
#include "es_FixedSizeVector.h"

namespace nn { namespace es {

// 登録可能アカウント最大数（本来は account::UserCountMax と同値であるべき）
// TORIAEZU:
//  現実装では想定ユースケースに合わせて登録可能な上限数を 1 にしておく
//
const int ELicenseUserCountMax = 1;
NN_STATIC_ASSERT(ELicenseUserCountMax <= account::UserCountMax);

// ActiveRightsContext 内で管理する RightsId の数（v0 + patch + AOC）
const int UsingRightsCountMax = 1 + 1 + 2000;

// UsingRights の状態を表す列挙型
enum class UsingRightsState : uint8_t
{
    NotChecked,     // eLicense との照合が未実施
    NoELicense,     // 対応する eLicense が見つからなかった
    Valid,          // 対応する eLicense が見つかった
};

// ActiveRightsContext 内で管理する RightsId と関連フラグを抱えた要素構造体
struct UsingRights
{
    // ns から登録された RightsId
    RightsId rightsId;

    // 状態
    UsingRightsState state;

    // 必ず IsPermanent(), SetPermanent() を使ってアクセスすること
    Bit8 _isTemporaryFlags;

    uint8_t _padding[6];

    // 対応する eLicenseId、アカウント数分保持する
    ELicenseId eLicenseId[ELicenseUserCountMax];

    // 全 userIndex の eLicense 全てが無期限か否かを返す
    bool IsPermanentAll() const NN_NOEXCEPT
    {
        return this->_isTemporaryFlags == 0;
    }

    // 指定した userIndex の eLicense が無期限か否かを返す
    bool IsPermanent(int userIndex) const NN_NOEXCEPT
    {
        NN_SDK_ASSERT(0 <= userIndex && userIndex < ELicenseUserCountMax);
        return (this->_isTemporaryFlags & (0x1u << userIndex)) ? false : true;
    }

    // 指定した userIndex の eLicense が無期限か否かを設定する
    void SetPermanent(int userIndex, bool isPermanent) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(0 <= userIndex && userIndex < ELicenseUserCountMax);
        Bit8 mask  = 0x1u << userIndex;
        Bit8 value = this->_isTemporaryFlags & ~mask;
        this->_isTemporaryFlags = isPermanent ? value : (value | mask);
    }
};
NN_STATIC_ASSERT(ELicenseUserCountMax <= sizeof(UsingRights::_isTemporaryFlags) * 8);
NN_STATIC_ASSERT(sizeof(UsingRights) == 8 + 8 + 16 * ELicenseUserCountMax);


// ActiveRightsContext 内で保持する UsingRights
typedef fixed_size_vector<UsingRights, UsingRightsCountMax> UsingRightsArray;

}}  // namespace nn::es
