﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <memory>
#include <mutex>
#include <nn/es.h>
#include <nn/fs.h>
#include <nn/os.h>
#include <nn/kvdb/kvdb_BoundedString.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_BitArray.h>
#include <nn/util/util_FormatString.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/nn_SdkLog.h>

namespace nn { namespace es {

namespace detail
{
    const int BufferSize = 16000;

    // TicketList で使われる共有バッファ
    // ある TicketList にアクセスしている間は他の TicketList は使われないため、共有する
    extern char Buffer[BufferSize];
}

typedef kvdb::BoundedString<64> Path;

template <typename RecordT>
class TicketList
{
public:
    explicit TicketList(int maxRecordCount, const char* path) NN_NOEXCEPT : m_MaxRecordCount(maxRecordCount)
    {
        m_ListFilePath.AssignFormat("%s/ticket_list.bin", path);
    }

    void Initialize() NN_NOEXCEPT;

    // ファイルに保存可能なレコード数を取得します。
    // return: レコード数
    int MaxRecordCount() const NN_NOEXCEPT;

    // ファイルに保存されているレコード数を取得します。
    // return: 保存されているレコード数
    int Count() const NN_NOEXCEPT;

    // ファイルに保存されているレコードの中で、RightsId が一致するレコードの数を取得します。
    // return: RightsId が一致するレコードの数
    int Count(RightsId rightsId) const NN_NOEXCEPT;

    // ファイルに保存されているレコードの中で、RightsId が一致するレコードの数を取得します。
    // return: RightsId が一致するレコードの数
    int Count(const RightsIdIncludingKeyId& rightsId) const NN_NOEXCEPT;

    // ファイルの index 番目にレコードがあるか調べます。
    // return: レコードの有無
    bool Exists(int index) const NN_NOEXCEPT;

    // ファイルの index 番目のレコードを取得します。
    // return: レコードの有無
    bool Get(RecordT* outRecord, int index) const NN_NOEXCEPT;

    // ファイルにレコードを追加します。
    // ファイルの先頭からレコードがない領域を探し、あった場合はその領域に追加します。
    // 追加するレコードの重複チェックは行いません。
    // return: レコードを追記した index (追加できなかった場合、-1)
    int InsertWithoutCheckingDuplication(const RecordT& record) NN_NOEXCEPT;

    // ファイルの index 番目のレコード削除します。
    // return: レコードの削除が成功したか
    bool Erase(int index) NN_NOEXCEPT;

    // 128bit RightsId と TicketId が一致するレコードをすべて削除します。
    // return: レコードの削除が成功したか
    bool Erase(const RightsIdIncludingKeyId& rightsId, TicketId ticketId) NN_NOEXCEPT;

    // 全てのレコードを削除します。
    void Clear() NN_NOEXCEPT;

    // ファイルにレコードがあるか調べます。
    // return: レコードの index (レコードがなかった場合、-1)
    int Find(RightsId rightsId, TicketId ticketId) const NN_NOEXCEPT;

    // ファイルにレコードがあるか調べます。
    // return: レコードの index (レコードがなかった場合、-1)
    int Find(const RightsIdIncludingKeyId& rightsId, TicketId ticketId) const NN_NOEXCEPT;

    // RightsId が一致する TicketId である値を探します。
    // ファイルの先頭から探し、初めて見つかった値を返却します。
    // return: 一致した値が見つかったかどうか
    bool GetTicketId(TicketId* outTicketId, RightsId rightsId) const NN_NOEXCEPT;

    // 128bit RightsId が一致する TicketId である値を探します。
    // ファイルの先頭から探し、初めて見つかった値を返却します。
    // return: 一致した値が見つかったかどうか
    bool GetTicketId(TicketId* outTicketId, const RightsIdIncludingKeyId& rightsId) const NN_NOEXCEPT;

    int ReadTicketList(RecordT** outList, int startIndex) const NN_NOEXCEPT;
private:
    void Write(int index, RecordT record) NN_NOEXCEPT;
    RecordT Read(int index) const NN_NOEXCEPT;

    void InsertImpl(int index, RecordT) NN_NOEXCEPT;
    void EraseImpl(int index) NN_NOEXCEPT;

    bool IsValidIndex(int index) const NN_NOEXCEPT;
    void SetTerminator(int index) NN_NOEXCEPT;

    Path m_ListFilePath;
    int m_RecordCount = 0;
    int m_MaxRecordCount = 0;
    bool m_HasTerminator = false;
    int m_TerminatorIndex = 0;
};

template <typename RecordT>
class TicketDatabaseBase
{
public:
    explicit TicketDatabaseBase(int maxRecordCount, const char* databasePath) NN_NOEXCEPT : m_TicketList(maxRecordCount, databasePath) {}

    void Initialize() NN_NOEXCEPT;

    // チケットデータベースの保存数の上限に達しているか調べます。
    bool IsFull() const NN_NOEXCEPT;

    // チケットを削除します。
    bool DeleteTicket(const RightsIdIncludingKeyId& rightsId, TicketId ticketId) NN_NOEXCEPT;

    // すべてのチケットを削除します。
    void DeleteAllTicket() NN_NOEXCEPT;

    // チケットの数を数えます。
    int CountTicket() const NN_NOEXCEPT;

    // RightsId が一致するチケットの数を数えます。
    int CountTicket(RightsId rightsId) const NN_NOEXCEPT;

    // RightsId が一致するチケットの数を数えます。
    int CountTicket(const RightsIdIncludingKeyId& rightsId) const NN_NOEXCEPT;

    // チケットが存在しているか調べます。
    Result ExistTicket(RightsId rightsId, TicketId ticketId) const NN_NOEXCEPT;

    // チケットが存在しているか調べます。
    Result ExistTicket(const RightsIdIncludingKeyId& rightsId, TicketId ticketId) const NN_NOEXCEPT;

    // RightsId に対応する TicketId を探します。
    Result FindTicket(TicketId* outTicketId, RightsId rightsId) const NN_NOEXCEPT;

    // RightsId に対応する TicketId を探します。
    Result FindTicket(TicketId* outTicketId, const RightsIdIncludingKeyId& rightsId) const NN_NOEXCEPT;

    void DeleteTicketByRightsId(const RightsIdIncludingKeyId rightsIdList[], size_t count) NN_NOEXCEPT;

    void DeleteTicketOfRightsId(const RightsIdIncludingKeyId rightsIdList[], size_t count) NN_NOEXCEPT;

    void DeleteTicketByAccountId(AccountId accountId) NN_NOEXCEPT;

    void DeleteAllTicketExcludingList(const TicketId exclusionTicketIdList[], size_t count) NN_NOEXCEPT;

    int GetRightsIdList(RightsIdIncludingKeyId outRightsIdList[], size_t count) const NN_NOEXCEPT;

    int GetRightsIdList(RightsIdIncludingKeyId outRightsIdList[], size_t count, RightsId rightsId) const NN_NOEXCEPT;

    int ListMissingTicket(TicketId outTicketIdList[], size_t outTicketIdListSize, const TicketId ticketIdList[], size_t ticketIdListSize) const NN_NOEXCEPT;

    void OwnTicketList(bool outOwnTicketList[], const RightsIdIncludingKeyId rightsIdList[], size_t count) const NN_NOEXCEPT;

    void OwnTicketList(bool outOwnTicketList[], const TicketId ticketIdList[], size_t count) const NN_NOEXCEPT;

protected:
    TicketList<RecordT> m_TicketList;
};

template <typename RecordT>
void TicketList<RecordT>::Initialize() NN_NOEXCEPT
{
    Result result = fs::CreateFile(m_ListFilePath, m_MaxRecordCount * sizeof(RecordT));
    NN_ABORT_UNLESS(result.IsSuccess() || fs::ResultPathAlreadyExists::Includes(result));

    // リストファイルを作成した場合、チケットデータベースをクリア
    if (result.IsSuccess())
    {
        Clear();
    }

    int index = 0;
    // バッファの大きさだけ一度に読み込む
    RecordT* list;
    while (int readRecordCount = ReadTicketList(&list, index))
    {
        // バッファの各レコードに対し実行
        for (int i = 0; i < readRecordCount; i++, index++)
        {
            if (list[i].IsTerminator())
            {
                m_HasTerminator = true;
                m_TerminatorIndex = index;
                return;
            }

            // 空でないレコードならカウントする
            if (!list[i].IsEmpty())
            {
                m_RecordCount++;
            }
        }
    }
}

template <typename RecordT>
int TicketList<RecordT>::MaxRecordCount() const NN_NOEXCEPT
{
    return m_MaxRecordCount;
}

template <typename RecordT>
int TicketList<RecordT>::Count() const NN_NOEXCEPT
{
    return m_RecordCount;
}

template <typename RecordT>
int TicketList<RecordT>::Count(RightsId rightsId) const NN_NOEXCEPT
{
    int count = 0;
    int index = 0;
    RecordT* list;
    while (int readRecordCount = ReadTicketList(&list, index))
    {
        for (int i = 0; i < readRecordCount; i++, index++)
        {
            if (!list[i].IsEmpty())
            {
                if (rightsId == list[i].rightsId.GetRightsId())
                {
                    count++;
                }
            }
        }
    }

    return count;
}

template <typename RecordT>
int TicketList<RecordT>::Count(const RightsIdIncludingKeyId& rightsId) const NN_NOEXCEPT
{
    int count = 0;
    int index = 0;
    RecordT* list;
    while (int readRecordCount = ReadTicketList(&list, index))
    {
        for (int i = 0; i < readRecordCount; i++, index++)
        {
            if (!list[i].IsEmpty())
            {
                if (rightsId == list[i].rightsId)
                {
                    count++;
                }
            }
        }
    }

    return count;
}

template <typename RecordT>
bool TicketList<RecordT>::Exists(int index) const NN_NOEXCEPT
{
    // 有効でない index のレコードは存在しない
    if (!IsValidIndex(index))
    {
        return false;
    }

    RecordT record = Read(index);

    return !record.IsEmpty();
}

template <typename RecordT>
bool TicketList<RecordT>::Get(RecordT* outRecord, int index) const NN_NOEXCEPT
{
     *outRecord = Read(index);

     if (outRecord->IsEmpty())
     {
         *outRecord = RecordT::NullRecord;
         return false;
     }

     return true;
}

template <typename RecordT>
int TicketList<RecordT>::InsertWithoutCheckingDuplication(const RecordT& record) NN_NOEXCEPT
{
    int index = 0;
    RecordT* list;
    while (int readRecordCount = ReadTicketList(&list, index))
    {
        for (int i = 0; i < readRecordCount; i++, index++)
        {
            // 最初に見つかった空のレコードの領域にレコードを挿入する
            if (list[i].IsEmpty())
            {
                InsertImpl(index, record);

                return index;
            }
        }
    }

    return -1;
}

template <typename RecordT>
bool TicketList<RecordT>::Erase(int index) NN_NOEXCEPT
{
    // 有効でない index のレコードは存在しない
    if (!IsValidIndex(index))
    {
        return false;
    }

    EraseImpl(index);

    return true;
}

template <typename RecordT>
bool TicketList<RecordT>::Erase(const RightsIdIncludingKeyId& rightsId, TicketId ticketId) NN_NOEXCEPT
{
    int index = Find(rightsId, ticketId);

    if (index == -1)
    {
        return false;
    }

    return Erase(index);
}

template <typename RecordT>
void TicketList<RecordT>::Clear() NN_NOEXCEPT
{
    m_RecordCount = 0;

    // 終端レコードを表すターミネーターをリストファイルの先頭に設定する
    SetTerminator(0);
}

template <typename RecordT>
int TicketList<RecordT>::Find(RightsId rightsId, TicketId ticketId) const NN_NOEXCEPT
{
    int index = 0;
    RecordT* list;
    while (int readRecordCount = ReadTicketList(&list, index))
    {
        for (int i = 0; i < readRecordCount; i++, index++)
        {
            if (!list[i].IsEmpty())
            {
                if (rightsId == list[i].rightsId.GetRightsId() && ticketId == list[i].ticketId)
                {
                    return index;
                }
            }
        }
    }

    return -1;
}


template <typename RecordT>
int TicketList<RecordT>::Find(const RightsIdIncludingKeyId& rightsId, TicketId ticketId) const NN_NOEXCEPT
{
    int index = 0;
    RecordT* list;
    while (int readRecordCount = ReadTicketList(&list, index))
    {
        for (int i = 0; i < readRecordCount; i++, index++)
        {
            if (!list[i].IsEmpty())
            {
                if (rightsId == list[i].rightsId && ticketId == list[i].ticketId)
                {
                    return index;
                }
            }
        }
    }

    return -1;
}

template <typename RecordT>
bool TicketList<RecordT>::GetTicketId(TicketId* outTicketId, const RightsIdIncludingKeyId& rightsId) const NN_NOEXCEPT
{
    int index = 0;
    RecordT* list;
    while (int readRecordCount = ReadTicketList(&list, index))
    {
        for (int i = 0; i < readRecordCount; i++, index++)
        {
            if (!list[i].IsEmpty())
            {
                if (rightsId == list[i].rightsId)
                {
                    RecordT record;
                    Get(&record, index);

                    *outTicketId = record.ticketId;
                    return true;
                }
            }
        }
    }

    return false;
}

template <typename RecordT>
bool TicketList<RecordT>::GetTicketId(TicketId* outTicketId, RightsId rightsId) const NN_NOEXCEPT
{
    int index = 0;
    RecordT* list;
    while (int readRecordCount = ReadTicketList(&list, index))
    {
        for (int i = 0; i < readRecordCount; i++, index++)
        {
            if (!list[i].IsEmpty())
            {
                if (rightsId == list[i].rightsId.GetRightsId())
                {
                    RecordT record;
                    Get(&record, index);

                    *outTicketId = record.ticketId;
                    return true;
                }
            }
        }
    }

    return false;
}

template <typename RecordT>
int TicketList<RecordT>::ReadTicketList(RecordT** outList, int startIndex) const NN_NOEXCEPT
{
    if (!IsValidIndex(startIndex))
    {
        return 0;
    }

    int readSize;
    if (m_HasTerminator)
    {
        readSize = std::min(detail::BufferSize, static_cast<int>((m_TerminatorIndex + 1 - startIndex) * sizeof(RecordT)));
    }
    else
    {
        readSize = std::min(detail::BufferSize, static_cast<int>((m_MaxRecordCount - startIndex) * sizeof(RecordT)));
    }

    fs::FileHandle fileHandle;
    NN_ABORT_UNLESS_RESULT_SUCCESS(fs::OpenFile(&fileHandle, m_ListFilePath.Get(), fs::OpenMode_Read));
    NN_UTIL_SCOPE_EXIT{ fs::CloseFile(fileHandle); };

    NN_ABORT_UNLESS_RESULT_SUCCESS(fs::ReadFile(fileHandle, startIndex * sizeof(RecordT), detail::Buffer, readSize));
    *outList = reinterpret_cast<RecordT*>(detail::Buffer);
    return readSize / sizeof(RecordT);
}

template <typename RecordT>
void TicketList<RecordT>::Write(int index, RecordT record) NN_NOEXCEPT
{
    fs::FileHandle fileHandle;
    NN_ABORT_UNLESS_RESULT_SUCCESS(fs::OpenFile(&fileHandle, m_ListFilePath.Get(), fs::OpenMode_Write));
    NN_UTIL_SCOPE_EXIT{ fs::CloseFile(fileHandle); };

    NN_ABORT_UNLESS_RESULT_SUCCESS(fs::WriteFile(fileHandle, index * sizeof(RecordT), &record, sizeof(RecordT), fs::WriteOption::MakeValue(fs::WriteOptionFlag_Flush)));
}

template <typename RecordT>
RecordT TicketList<RecordT>::Read(int index) const NN_NOEXCEPT
{
    RecordT record;

    fs::FileHandle fileHandle;
    NN_ABORT_UNLESS_RESULT_SUCCESS(fs::OpenFile(&fileHandle, m_ListFilePath.Get(), fs::OpenMode_Read));
    NN_UTIL_SCOPE_EXIT{ fs::CloseFile(fileHandle); };

    NN_ABORT_UNLESS_RESULT_SUCCESS(fs::ReadFile(fileHandle, index * sizeof(RecordT), &record, sizeof(RecordT)));

    return record;
}

template <typename RecordT>
void TicketList<RecordT>::InsertImpl(int index, RecordT record) NN_NOEXCEPT
{
    Write(index, record);
    m_RecordCount++;

    if (m_TerminatorIndex == index)
    {
        SetTerminator(index + 1);
    }
}

template <typename RecordT>
void TicketList<RecordT>::EraseImpl(int index) NN_NOEXCEPT
{
    Write(index, RecordT::NullRecord);
    m_RecordCount--;

    if (index == m_TerminatorIndex - 1)
    {
        SetTerminator(index);
    }
}

template <typename RecordT>
bool TicketList<RecordT>::IsValidIndex(int index) const NN_NOEXCEPT
{
    if (m_HasTerminator)
    {
        return index >= 0 && index <= m_TerminatorIndex;
    }
    else
    {
        return index >= 0 && index < m_MaxRecordCount;
    }
}

template <typename RecordT>
void TicketList<RecordT>::SetTerminator(int index) NN_NOEXCEPT
{
    if (index >= m_MaxRecordCount)
    {
        m_HasTerminator = false;
        m_TerminatorIndex = 0;
    }
    else
    {
        Write(index, RecordT::TerminatorRecord);
        m_HasTerminator = true;
        m_TerminatorIndex = index;
    }
}

template <typename RecordT>
void TicketDatabaseBase<RecordT>::Initialize() NN_NOEXCEPT
{
    m_TicketList.Initialize();
}

template <typename RecordT>
bool TicketDatabaseBase<RecordT>::IsFull() const NN_NOEXCEPT
{
    return m_TicketList.Count() >= m_TicketList.MaxRecordCount();
}

template <typename RecordT>
bool TicketDatabaseBase<RecordT>::DeleteTicket(const RightsIdIncludingKeyId& rightsId, TicketId ticketId) NN_NOEXCEPT
{
    return m_TicketList.Erase(rightsId, ticketId);
}

template <typename RecordT>
void TicketDatabaseBase<RecordT>::DeleteAllTicket() NN_NOEXCEPT
{
    m_TicketList.Clear();
}

template <typename RecordT>
int TicketDatabaseBase<RecordT>::CountTicket() const NN_NOEXCEPT
{
    return m_TicketList.Count();
}

template <typename RecordT>
int TicketDatabaseBase<RecordT>::CountTicket(RightsId rightsId) const NN_NOEXCEPT
{
    return m_TicketList.Count(rightsId);
}

template <typename RecordT>
int TicketDatabaseBase<RecordT>::CountTicket(const RightsIdIncludingKeyId& rightsId) const NN_NOEXCEPT
{
    return m_TicketList.Count(rightsId);
}

template <typename RecordT>
Result TicketDatabaseBase<RecordT>::ExistTicket(const RightsId rightsId, TicketId ticketId) const NN_NOEXCEPT
{
    int index = m_TicketList.Find(rightsId, ticketId);
    NN_RESULT_THROW_UNLESS(index != -1, ResultRightsIdNotFound());

    NN_RESULT_SUCCESS;
}

template <typename RecordT>
Result TicketDatabaseBase<RecordT>::ExistTicket(const RightsIdIncludingKeyId& rightsId, TicketId ticketId) const NN_NOEXCEPT
{
    int index = m_TicketList.Find(rightsId, ticketId);
    NN_RESULT_THROW_UNLESS(index != -1, ResultRightsIdNotFound());

    NN_RESULT_SUCCESS;
}

template <typename RecordT>
Result TicketDatabaseBase<RecordT>::FindTicket(TicketId* outTicketId, RightsId rightsId) const NN_NOEXCEPT
{
    bool isFound = m_TicketList.GetTicketId(outTicketId, rightsId);
    NN_RESULT_THROW_UNLESS(isFound, ResultRightsIdNotFound());

    NN_RESULT_SUCCESS;
}

template <typename RecordT>
Result TicketDatabaseBase<RecordT>::FindTicket(TicketId* outTicketId, const RightsIdIncludingKeyId& rightsId) const NN_NOEXCEPT
{
    bool isFound = m_TicketList.GetTicketId(outTicketId, rightsId);
    NN_RESULT_THROW_UNLESS(isFound, ResultRightsIdNotFound());

    NN_RESULT_SUCCESS;
}

template <typename RecordT>
void TicketDatabaseBase<RecordT>::DeleteTicketByRightsId(const RightsIdIncludingKeyId rightsIdList[], size_t count) NN_NOEXCEPT
{
    int index = 0;
    RecordT* list;
    while (int readRecordCount = m_TicketList.ReadTicketList(&list, index))
    {
        for (int i = 0; i < readRecordCount; i++, index++)
        {
            if (list[i].IsEmpty())
            {
                continue;
            }

            for (size_t j = 0; j < count; j++)
            {
                if (list[i].rightsId == rightsIdList[j])
                {
                    m_TicketList.Erase(index);
                    break;
                }
            }
        }
    }
}

template <typename RecordT>
void TicketDatabaseBase<RecordT>::DeleteTicketOfRightsId(const RightsIdIncludingKeyId rightsIdList[], size_t count) NN_NOEXCEPT
{
    int index = 0;
    RecordT* list;
    while (int readRecordCount = m_TicketList.ReadTicketList(&list, index))
    {
        for (int i = 0; i < readRecordCount; i++, index++)
        {
            if (list[i].IsEmpty())
            {
                continue;
            }

            for (size_t j = 0; j < count; j++)
            {
                if (list[i].rightsId.GetRightsId() == rightsIdList[j].GetRightsId())
                {
                    m_TicketList.Erase(index);
                    break;
                }
            }
        }
    }
}


template <typename RecordT>
void TicketDatabaseBase<RecordT>::DeleteTicketByAccountId(AccountId accountId) NN_NOEXCEPT
{
    int index = 0;
    RecordT* list;
    while (int readRecordCount = m_TicketList.ReadTicketList(&list, index))
    {
        for (int i = 0; i < readRecordCount; i++, index++)
        {
            if (list[i].IsEmpty())
            {
                continue;
            }

            if (list[i].accountId == accountId)
            {
                // チケットを削除
                m_TicketList.Erase(index);
            }
        }
    }
}

template <typename RecordT>
void TicketDatabaseBase<RecordT>::DeleteAllTicketExcludingList(const TicketId exclusionTicketIdList[], size_t count) NN_NOEXCEPT
{
    int index = 0;
    RecordT* list;
    while (int readRecordCount = m_TicketList.ReadTicketList(&list, index))
    {
        for (int i = 0; i < readRecordCount; i++, index++)
        {
            if (list[i].IsEmpty())
            {
                continue;
            }

            bool isDelete = true;

            for (size_t j = 0; j < count; j++)
            {
                if (list[i].ticketId == exclusionTicketIdList[j])
                {
                    isDelete = false;
                    break;
                }
            }

            if (isDelete)
            {
                // チケットを削除
                m_TicketList.Erase(index);
            }
        }
    }
}

template <typename RecordT>
int TicketDatabaseBase<RecordT>::GetRightsIdList(RightsIdIncludingKeyId outRightsIdList[], size_t count) const NN_NOEXCEPT
{
    int recordCount = 0;
    int index = 0;
    RecordT* list;
    while (int readRecordCount = m_TicketList.ReadTicketList(&list, index))
    {
        for (int i = 0; i < readRecordCount; i++, index++)
        {
            if (!list[i].IsEmpty())
            {
                outRightsIdList[recordCount] = list[i].rightsId;
                recordCount++;

                if (recordCount >= static_cast<int>(count))
                {
                    return recordCount;
                }
            }
        }
    }

    return recordCount;
}

template <typename RecordT>
int TicketDatabaseBase<RecordT>::GetRightsIdList(RightsIdIncludingKeyId outRightsIdList[], size_t count, RightsId rightsId) const NN_NOEXCEPT
{
    int recordCount = 0;
    int index = 0;
    RecordT* list;
    while (int readRecordCount = m_TicketList.ReadTicketList(&list, index))
    {
        for (int i = 0; i < readRecordCount; i++, index++)
        {
            if (!list[i].IsEmpty() && rightsId == list[i].rightsId.GetRightsId())
            {
                outRightsIdList[recordCount] = list[i].rightsId;
                recordCount++;

                if (recordCount >= static_cast<int>(count))
                {
                    return recordCount;
                }
            }
        }
    }

    return recordCount;
}

template <typename RecordT>
int TicketDatabaseBase<RecordT>::ListMissingTicket(TicketId outTicketIdList[], size_t outTicketIdListCount, const TicketId ticketIdList[], size_t ticketIdListCount) const NN_NOEXCEPT
{
    int recordCount = 0;

    int length = static_cast<int>(ticketIdListCount);
    size_t bitArrayMemorySize = util::BitArray::CalculateWorkMemorySize(length);
    std::unique_ptr<uint8_t[]> bitArrayMemory(new uint8_t[bitArrayMemorySize]);
    util::BitArray matchList(bitArrayMemory.get(), bitArrayMemorySize, length);
    matchList.reset();

    int index = 0;
    RecordT* list;
    while (int readRecordCount = m_TicketList.ReadTicketList(&list, index))
    {
        for (int i = 0; i < readRecordCount; i++, index++)
        {
            if (list[i].IsEmpty())
            {
                continue;
            }

            for (int j = 0; j < length; j++)
            {
                if (list[i].ticketId == ticketIdList[j])
                {
                    matchList[j] = true;
                }
            }
        }
    }

    // 保有していなかった Personalized チケットを出力配列に格納する
    for (int i = 0; i < length; i++)
    {
        if (!matchList[i])
        {
            outTicketIdList[recordCount] = ticketIdList[i];
            recordCount++;

            if (recordCount >= static_cast<int>(outTicketIdListCount))
            {
                return recordCount;
            }
        }
    }

    return recordCount;
}

template <typename RecordT>
void TicketDatabaseBase<RecordT>::OwnTicketList(bool outOwnTicketList[], const RightsIdIncludingKeyId rightsIdList[], size_t count) const NN_NOEXCEPT
{
    int index = 0;
    RecordT* list;
    while (int readRecordCount = m_TicketList.ReadTicketList(&list, index))
    {
        for (int i = 0; i < readRecordCount; i++, index++)
        {
            // 空のレコードだったらチェックしない
            if (list[i].IsEmpty())
            {
                continue;
            }

            // rightsIdList の各エントリと一致しているか確認する
            for (size_t j = 0; j < count; j++)
            {
                // 既に保有しているならチェックしない
                if (outOwnTicketList[j])
                {
                    continue;
                }

                if (list[i].rightsId == rightsIdList[j])
                {
                    outOwnTicketList[j] = true;
                }
            }
        }
    }
}

template <typename RecordT>
void TicketDatabaseBase<RecordT>::OwnTicketList(bool outOwnTicketList[], const TicketId ticketIdList[], size_t count) const NN_NOEXCEPT
{
    int index = 0;
    RecordT* list;
    while (int readRecordCount = m_TicketList.ReadTicketList(&list, index))
    {
        for (int i = 0; i < readRecordCount; i++, index++)
        {
            // 空のレコードだったらチェックしない
            if (list[i].IsEmpty())
            {
                continue;
            }

            // ticketIdList の各エントリと一致しているか確認する
            for (size_t j = 0; j < count; j++)
            {
                // 既に保有しているならチェックしない
                if (outOwnTicketList[j])
                {
                    continue;
                }

                if (list[i].ticketId == ticketIdList[j])
                {
                    outOwnTicketList[j] = true;
                }
            }
        }
    }
}

}} // namespace nn::es
