﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <memory>
#include <nn/es.h>
#include <nn/fs.h>
#include <nn/fs/fs_SystemSaveData.h>
#include <nn/escore/escore.h>
#include <nn/util/util_FormatString.h>
#include "es_Istorage.h"
#include "es_TicketDatabaseBase.h"

namespace nn { namespace es {

struct TicketRecord;
struct TicketMetaRecord;
class ETicketServiceDatabase;
class CertificateDatabase;
class TicketDatabase;
class TicketMetaRecordDatabase;

struct TicketRecord
{
    static const TicketRecord NullRecord;
    static const TicketRecord TerminatorRecord;

    RightsIdIncludingKeyId rightsId;
    TicketId ticketId;
    AccountId accountId;
    uint16_t propertyMask;
    uint16_t padding;

    bool IsEmpty() const NN_NOEXCEPT
    {
        return (rightsId == NullRecord.rightsId && ticketId == NullRecord.ticketId && accountId == NullRecord.accountId) || IsTerminator();
    }

    bool IsTerminator() const NN_NOEXCEPT
    {
        return rightsId == TerminatorRecord.rightsId && ticketId == TerminatorRecord.ticketId && accountId == TerminatorRecord.accountId;
    }

    bool IsElicenseRequired() const NN_NOEXCEPT
    {
        return (propertyMask & static_cast<std::underlying_type<PropertyMaskFlag>::type>(PropertyMaskFlag::IsElicenseRequired)) != 0;
    }
};

enum TicketMetaRecordType : uint8_t
{
    TicketMetaRecordType_Prepurchase,
};

struct TicketMetaRecord
{
    static const TicketMetaRecord NullRecord;
    static const TicketMetaRecord TerminatorRecord;

    RightsIdIncludingKeyId rightsId;
    TicketId ticketId;
    AccountId accountId;
    TicketMetaRecordType type;
    int8_t reserved[3];
    union {
        struct {
            /*!
            * @brief 配信開始予定時刻です。
            */
            time::PosixTime deliveryScheduledTime;

            /*!
            * @brief 予約領域です。
            */
            int8_t reserved[216];
        } prepurchase;
    };

    bool IsEmpty() const NN_NOEXCEPT
    {
        return (rightsId == NullRecord.rightsId && ticketId == NullRecord.ticketId && accountId == NullRecord.accountId) || IsTerminator();
    }

    bool IsTerminator() const NN_NOEXCEPT
    {
        return rightsId == TerminatorRecord.rightsId && ticketId == TerminatorRecord.ticketId && accountId == TerminatorRecord.accountId;
    }
};
NN_STATIC_ASSERT(sizeof(TicketMetaRecord) == 256);

TicketMetaRecord GetTicketMetaRecordFromPrepurchaseRecord(PrepurchaseRecord record) NN_NOEXCEPT;

PrepurchaseRecord GetPrepurchaseRecordFromTicketMetaRecord(TicketMetaRecord metaRecord) NN_NOEXCEPT;

class ETicketServiceDatabase
{
public:
    explicit ETicketServiceDatabase(const char* databasePath) NN_NOEXCEPT
    {
        m_DatabasePath.AssignFormat("%s", databasePath);
        m_VersionFilePath.AssignFormat("%s/version.bin", databasePath);
    }

    Path GetDatabasePath() NN_NOEXCEPT
    {
        return m_DatabasePath;
    }

    void SetVersion(int version) NN_NOEXCEPT;

    int GetVersion() const NN_NOEXCEPT;

protected:
    void Initialize() NN_NOEXCEPT;

private:
    Path m_DatabasePath;
    Path m_VersionFilePath;
};

class CertificateDatabase : public ETicketServiceDatabase
{
public:
    explicit CertificateDatabase(const char* certificateDatabasePath) NN_NOEXCEPT : ETicketServiceDatabase(certificateDatabasePath)
    {
        m_CertificatePath.AssignFormat("%s/certificate", certificateDatabasePath);
    }

    void Initialize() NN_NOEXCEPT;

    void ImportCertificate(const void* certificate, size_t certificateSize, const char* subjectName) NN_NOEXCEPT;

    int GetCertificateSize(const char* subjectName) NN_NOEXCEPT;

    Result GetCertificate(int* outSize, void* outBuffer, size_t outBufferSize, const char* subjectName) NN_NOEXCEPT;

private:
    Path m_CertificatePath;
};

class TicketDatabase : public TicketDatabaseBase<TicketRecord>, public ETicketServiceDatabase
{
public:
    // チケットの情報を TicketInfo 構造体で取得します。
    static void CreateTicketInfo(TicketInfo* outTicketInfo, const void* ticket, size_t ticketSize) NN_NOEXCEPT;

    TicketDatabase(int maxRecordCount, const char* ticketDatabasePath, CertificateDatabase* certificateDatabaseRef) NN_NOEXCEPT
        : TicketDatabaseBase(maxRecordCount, ticketDatabasePath),
        ETicketServiceDatabase(ticketDatabasePath),
        m_CertificateDatabaseRef(certificateDatabaseRef)
    {
        m_TicketFilePath.AssignFormat("%s/ticket.bin", ticketDatabasePath);
    }

    void Initialize() NN_NOEXCEPT;

    // チケットをインポートします。
    Result ImportTicket(const void* ticket, size_t ticketSize) NN_NOEXCEPT;

    // チケットのタイトルキーを取得します。
    Result GetTitleKey(AesKey* outTitleKey, const RightsIdIncludingKeyId& rightsId, TicketId ticketId, int keyGeneration) const NN_NOEXCEPT;

    // チケットのサイズを取得します。
    size_t GetTicketSize(const RightsIdIncludingKeyId& rightsId, TicketId ticketId) const NN_NOEXCEPT;

    // チケットのデータを取得します。
    Result GetTicketData(size_t* outSize, void* outTicketBuffer, size_t ticketBufferSize, const RightsIdIncludingKeyId& rightsId, TicketId ticketId) const NN_NOEXCEPT;

    // 暗号化されたチケットのサイズを取得します。
    size_t GetEncryptedTicketSize(const RightsIdIncludingKeyId& rightsId, TicketId ticketId) const NN_NOEXCEPT;

    // 暗号化されたチケットのデータと RSA で暗号化された AES 暗号鍵を取得します
    Result GetEncryptedTicketData(size_t* outSize, void* outTicketBuffer, size_t ticketBufferSize, void* outKey, size_t outKeySize, const RightsIdIncludingKeyId& rightsId, TicketId ticketId) const NN_NOEXCEPT;

    // チケットに対応する証明書名を取得します。
    bool GetCertificateName(char* outCaName, char* outXsName, size_t caNameBufferSize, size_t xsNameBufferSize, const RightsIdIncludingKeyId& rightsId, TicketId ticketId) const NN_NOEXCEPT;

    // チケットに対応する証明書のサイズを取得します。
    int GetCertificateSize(const RightsIdIncludingKeyId& rightsId, TicketId ticketId) const NN_NOEXCEPT;

    // チケットに対応する証明書を取得します。
    int GetCertificateData(void* outCertificate, size_t outCertificateSize, const RightsIdIncludingKeyId& rightsId, TicketId ticketId) const NN_NOEXCEPT;

    // チケットの情報を TicketInfo 構造体で取得します。
    bool GetTicketInfo(TicketInfo* outTicketInfo, const RightsIdIncludingKeyId& rightsId, TicketId ticketId) const NN_NOEXCEPT;

    // 機器認証されたチケットがあるかを判定します。
    bool HasDeviceLinkedTicket(RightsId rightsId) const NN_NOEXCEPT;

    // ユーティリティ関数
    int GetTicketInfoList(TicketInfo outTicketInfoList[], size_t ticketInfoListCount, const RightsIdIncludingKeyId rightsIdList[], size_t rightsIdListCount) const NN_NOEXCEPT;

    int ListLightTicketInfo(LightTicketInfo outTicketInfoList[], size_t count, const RightsIdIncludingKeyId& rightsId) const NN_NOEXCEPT;

    void DeleteAllDeviceLinkedTicketExcludingList(const TicketId exclusionTicketIdList[], size_t count) NN_NOEXCEPT;

private:
    // チケットを証明書 DB に保存されている証明書で検証します。
    Result VerifyTicket(const void* ticket, size_t ticketSize) const NN_NOEXCEPT;

    bool GetCertificateNameImpl(char* outCaName, char* outXsName, size_t caNameBufferSize, size_t xsNameBufferSize, const void* ticket, size_t ticketSize) const NN_NOEXCEPT;

    void Write(int index, const void* ticket, size_t ticketSize) NN_NOEXCEPT;
    void Read(void* outTicket, size_t ticketSize, int index) const NN_NOEXCEPT;

    CertificateDatabase* m_CertificateDatabaseRef;
    Path m_TicketFilePath;
};

class TicketMetaRecordDatabase : public TicketDatabaseBase<TicketMetaRecord>, public ETicketServiceDatabase
{
public:
    TicketMetaRecordDatabase(int maxRecordCount, const char* ticketDatabasePath) NN_NOEXCEPT : TicketDatabaseBase(maxRecordCount, ticketDatabasePath), ETicketServiceDatabase(ticketDatabasePath) {}

    void Initialize() NN_NOEXCEPT;

    // チケットメタ記録をインポートします。
    Result ImportTicket(TicketMetaRecord record) NN_NOEXCEPT;

    // チケットメタ記録があるかを確認します。
    bool HasTicketMetaRecord(RightsId rightsId, TicketMetaRecordType type) const NN_NOEXCEPT;

    // ユーティリティ関数
    int ListTicketMetaRecordInfo(TicketMetaRecord ticketMetaRecordList[], size_t count, const RightsIdIncludingKeyId& rightsId, TicketMetaRecordType type) const NN_NOEXCEPT;

private:
    bool IsSameRightsId(const RightsIdIncludingKeyId& lhs, const RightsIdIncludingKeyId& rhs, TicketMetaRecordType type) const NN_NOEXCEPT;
};

}} // namespace nn::es
