﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <array>
#include <algorithm>
#include <nn/nn_Common.h>
#include <nn/result/result_HandlingUtility.h>

namespace nn { namespace es {

template<typename T, size_t Size>
class fixed_size_vector
{
public:
    typedef T                   value_type;
    typedef value_type*         iterator;
    typedef const value_type*   const_iterator;

    fixed_size_vector() NN_NOEXCEPT = default;

    NN_IMPLICIT fixed_size_vector(std::initializer_list<int> list) NN_NOEXCEPT
    {
        for (auto itr = list.begin(); itr != list.end(); itr++)
        {
            push_back(*itr);
        }
    }

    T& operator[](int index) NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_LESS(index, m_size);
        return m_data[index];
    }

    const T& operator[](int index) const NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_LESS(index, m_size);
        return m_data[index];
    }

    iterator begin() NN_NOEXCEPT
    {
        return &m_data[0];
    }

    iterator end() NN_NOEXCEPT
    {
        return &m_data[m_size];
    }

    const_iterator cbegin() NN_NOEXCEPT
    {
        return &m_data[0];
    }

    const_iterator cend() NN_NOEXCEPT
    {
        return &m_data[m_size];
    }

    T& front() NN_NOEXCEPT
    {
        NN_ABORT_UNLESS(!empty());
        return m_data[0];
    }

    const T& front() const NN_NOEXCEPT
    {
        NN_ABORT_UNLESS(!empty());
        return m_data[0];
    }

    T& back() NN_NOEXCEPT
    {
        NN_ABORT_UNLESS(!empty());
        return m_data[m_size - 1];
    }

    const T& back() const NN_NOEXCEPT
    {
        NN_ABORT_UNLESS(!empty());
        return m_data[m_size - 1];
    }

    void clear() NN_NOEXCEPT
    {
        m_size = 0;
    }

    T* data() NN_NOEXCEPT
    {
        return m_data.data();
    }

    bool empty() const NN_NOEXCEPT
    {
        return m_size == 0;
    }

    iterator erase(iterator position) NN_NOEXCEPT
    {
        if (!(begin() <= position && position < end()))
        {
            return end();
        }

        std::move(position + 1, end(), position);
        m_size--;

        return position;
    }

    iterator insert(iterator position, const T& value) NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_LESS(m_size, Size);

        if (!(begin() <= position && position < end()))
        {
            return end();
        }

        m_size++;
        std::move_backward(position, end() - 1, end());
        *position = value;

        return position;
    }

    size_t max_size() const NN_NOEXCEPT
    {
        return m_data.max_size();
    }

    void pop_back() NN_NOEXCEPT
    {
        NN_ABORT_UNLESS(!empty());
        m_size--;
    }

    void push_back(const T& value) NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_LESS(m_size, Size);
        m_data[m_size] = value;
        m_size++;
    }

    size_t size() const NN_NOEXCEPT
    {
        return m_size;
    }

private:
    std::array<T, Size> m_data;
    size_t m_size = 0;
};

}} // namespace nn::es
