﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <memory>

#include <nn/fs.h>
#include <nn/nn_Log.h>
#include <nn/escore/escore.h>
#include <nn/es/es_Configuration.h>
#include <nn/fs/fs_SdCardForDebug.h>
#include <nn/fs/fs_SystemSaveData.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/settings/factory/settings_DeviceCertificate.h>
#include <nn/settings/factory/settings_DeviceKey.h>
#include <nn/settings/fwdbg/settings_SettingsGetterApi.h>
#include <nn/settings/setting.h>
#include <nn/sf/sf_ObjectFactory.h>
#include <nn/spl/spl_Api.h>
#include <nn/es/es_ServiceLog.h>
#include "es_CreateETicketService.h"
#include "es_DatabaseVersion.h"
#include "es_ETicketServiceImpl.h"
#include "es_Istorage.h"
#include "es_MaxTicketCount.h"

namespace nn { namespace es {
namespace {
    const int CertificateListNumber = 4;

    static const char* CertificateDbPath           = "escertificate:/";
    static const char* CommonTicketDbPath          = "escommon:/";
    static const char* PersonalizedTicketDbPath    = "espersonalized:/";
    static const char* TicketMetaRecordDbPath      = "esmetarecord:/";
    static const char* ELicenseArchiveStorePath    = "eselicense:/";
}

ETicketServiceImpl::ETicketServiceImpl() NN_NOEXCEPT :
    m_CommonTicketDatabase(MaxCommonTicketCount, CommonTicketDbPath, &m_CertificateDatabase),
    m_PersonalizedTicketDatabase(MaxPersonalizedTicketCount, PersonalizedTicketDbPath, &m_CertificateDatabase),
    m_CertificateDatabase(CertificateDbPath),
    m_TicketMetaRecordDatabase(MaxTicketMetaRecordCount, TicketMetaRecordDbPath),
    m_ELicenseArchiveStore(ELicenseArchiveStorePath),
    m_ELicenseManager(&m_ELicenseArchiveStore, &m_ELicenseList)
{
    GetRightsAvailabilityMaster().Setup(&m_CommonTicketDatabase, &m_PersonalizedTicketDatabase, &m_TicketMetaRecordDatabase, &m_ELicenseList);
}

// チケットのインポートを行います。
Result ETicketServiceImpl::ImportTicket(const sf::InBuffer& ticket, const sf::InBuffer& certificate) NN_NOEXCEPT
{
    NN_ES_SERVICE_LOG_TRACE("ETicketServiceImpl::ImportTicket()\n");

    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    MemoryInputStream ticketReader(ticket.GetPointerUnsafe(), static_cast<unsigned int>(ticket.GetSize()));

    // チケットを検証する
    {
        // 証明書を分割する
        void* certificateList[CertificateListNumber];
        unsigned int outCertificateSize[CertificateListNumber];
        unsigned int certificateCount = static_cast<unsigned int>(CertificateListNumber);

        nn::escore::Certificate eCertificate;
        nn::escore::ESError esError = eCertificate.ParseCertList(certificate.GetPointerUnsafe(), static_cast<unsigned int>(certificate.GetSize()), certificateList, outCertificateSize, &certificateCount);
        NN_RESULT_THROW_UNLESS(esError == ES_ERR_OK, ResultCertificateInvalid());

        // チケットを検証する
        nn::escore::ETicket eTicket;
        esError = eTicket.Set(ticketReader, const_cast<const void**>(certificateList), certificateCount, true);

        // チケットの検証に失敗した場合
        if (esError != ES_ERR_OK)
        {
            // チケットが無効な場合
            NN_RESULT_THROW_UNLESS(!(esError > ES_ERR_CERT_BASE), ResultTicketInvalid());
            // 証明書が無効な場合
            NN_RESULT_THROW(ResultCertificateInvalid());
        }

        // 証明書の subject 名を取得する
        unsigned char* subject[CertificateListNumber];
        unsigned char* issuer[CertificateListNumber];

        for (unsigned int i = 0; i < certificateCount; i++)
        {
            esError = eCertificate.GetCertNames(certificateList[i], &issuer[i], &subject[i]);
            NN_RESULT_THROW_UNLESS(esError == ES_ERR_OK, ResultCertificateInvalid());
        }

        // 証明書を保存する
        for (unsigned int i = 0; i < certificateCount; i++)
        {
            m_CertificateDatabase.ImportCertificate(certificateList[i], outCertificateSize[i], reinterpret_cast<char*>(subject[i]));
        }
    }

    // チケットの情報を取得
    TicketInfo ticketInfo;
    TicketDatabase::CreateTicketInfo(&ticketInfo, ticket.GetPointerUnsafe(), ticket.GetSize());

    // Common チケットの場合
    if (ticketInfo.accountId == 0)
    {
        NN_RESULT_DO(m_CommonTicketDatabase.ImportTicket(ticket.GetPointerUnsafe(), ticket.GetSize()));
    }
    // Personalized チケットの場合
    else
    {
        NN_RESULT_DO(m_PersonalizedTicketDatabase.ImportTicket(ticket.GetPointerUnsafe(), ticket.GetSize()));
    }

    NN_ABORT_UNLESS_RESULT_SUCCESS(fs::CommitSaveData(CertificateDbMountName));
    NN_ABORT_UNLESS_RESULT_SUCCESS(fs::CommitSaveData(CommonTicketDbMountName));
    NN_ABORT_UNLESS_RESULT_SUCCESS(fs::CommitSaveData(PersonalizedTicketDbMountName));
    NN_ABORT_UNLESS_RESULT_SUCCESS(fs::CommitSaveData(TicketMetaRecordDbMountName));

    NN_RESULT_SUCCESS;
} // NOLINT(impl/function_size)

// チケットのインポートを行います。
Result ETicketServiceImpl::ImportTicketCertificateSet(const sf::InBuffer& ticketCertificateSet) NN_NOEXCEPT
{
    NN_ES_SERVICE_LOG_TRACE("ETicketServiceImpl::ImportTicketCertificateSet()\n");

    MemoryInputStream ticketReader(ticketCertificateSet.GetPointerUnsafe(), static_cast<unsigned int>(ticketCertificateSet.GetSize()));

    nn::escore::ETicket eTicket;
    nn::escore::ESError esError = eTicket.Set(ticketReader, nullptr, 0, false);
    if (esError != ES_ERR_OK)
    {
        NN_RESULT_THROW_UNLESS(!(esError > ES_ERR_CERT_BASE), ResultTicketInvalid());
        NN_RESULT_THROW(ResultCertificateInvalid());
    }

    unsigned int ticketSize;
    esError = eTicket.GetSize(&ticketSize);
    NN_RESULT_THROW_UNLESS(esError == ES_ERR_OK, ResultTicketInvalid());

    size_t certificateSize = ticketCertificateSet.GetSize() - static_cast<size_t>(ticketSize);

    sf::InBuffer pInTicketBuffer(ticketCertificateSet.GetPointerUnsafe(), static_cast<size_t>(ticketSize));
    sf::InBuffer pInCertificateBuffer(ticketCertificateSet.GetPointerUnsafe() + ticketSize, certificateSize);

    return ImportTicket(pInTicketBuffer, pInCertificateBuffer);
}

// RightsID でチケットの複数削除を行います。
Result ETicketServiceImpl::DeleteTicket(const sf::InArray<RightsIdIncludingKeyId>& rightsIdList) NN_NOEXCEPT
{
    NN_ES_SERVICE_LOG_TRACE("ETicketServiceImpl::DeleteTicket()\n");

    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    // CommonTicket から削除
    m_CommonTicketDatabase.DeleteTicketByRightsId(rightsIdList.GetData(), rightsIdList.GetLength());

    // PersonalizedTicket から削除
    m_PersonalizedTicketDatabase.DeleteTicketByRightsId(rightsIdList.GetData(), rightsIdList.GetLength());

    // チケットメタから削除
    // RightsId が一致しているチケットメタを削除する
    m_TicketMetaRecordDatabase.DeleteTicketOfRightsId(rightsIdList.GetData(), rightsIdList.GetLength());

    NN_ABORT_UNLESS_RESULT_SUCCESS(fs::CommitSaveData(CommonTicketDbMountName));
    NN_ABORT_UNLESS_RESULT_SUCCESS(fs::CommitSaveData(PersonalizedTicketDbMountName));
    NN_ABORT_UNLESS_RESULT_SUCCESS(fs::CommitSaveData(TicketMetaRecordDbMountName));

    NN_RESULT_SUCCESS;
}

// AccountId が一致する全ての Personalized チケットの削除を行います。
Result ETicketServiceImpl::DeletePersonalizedTicket(AccountId accountId) NN_NOEXCEPT
{
    NN_ES_SERVICE_LOG_TRACE("ETicketServiceImpl::DeletePersonalizedTicket()\n");

    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    m_PersonalizedTicketDatabase.DeleteTicketByAccountId(accountId);
    m_TicketMetaRecordDatabase.DeleteTicketByAccountId(accountId);

    NN_ABORT_UNLESS_RESULT_SUCCESS(fs::CommitSaveData(PersonalizedTicketDbMountName));
    NN_ABORT_UNLESS_RESULT_SUCCESS(fs::CommitSaveData(TicketMetaRecordDbMountName));

    NN_RESULT_SUCCESS;
}

// 全ての Common チケットの削除を行います。
Result ETicketServiceImpl::DeleteAllCommonTicket() NN_NOEXCEPT
{
    NN_ES_SERVICE_LOG_TRACE("ETicketServiceImpl::DeleteAllCommonTicket()\n");

    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    // Coomon チケットデータベースをクリア
    m_CommonTicketDatabase.DeleteAllTicket();

    NN_ABORT_UNLESS_RESULT_SUCCESS(fs::CommitSaveData(CommonTicketDbMountName));

    NN_RESULT_SUCCESS;
}

// 全ての Personalized チケットの削除を行います。
Result ETicketServiceImpl::DeleteAllPersonalizedTicket() NN_NOEXCEPT
{
    NN_ES_SERVICE_LOG_TRACE("ETicketServiceImpl::DeleteAllPersonalizedTicket()\n");

    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    // Personalized チケットデータベースをクリア
    m_PersonalizedTicketDatabase.DeleteAllTicket();

    NN_ABORT_UNLESS_RESULT_SUCCESS(fs::CommitSaveData(PersonalizedTicketDbMountName));

    NN_RESULT_SUCCESS;
}

// ticketIdExclusionList[] にない全ての Personalized チケットの削除を行います。
Result ETicketServiceImpl::DeleteAllPersonalizedTicketExcludingList(const sf::InArray<TicketId>& ticketIdExclusionList) NN_NOEXCEPT
{
    NN_ES_SERVICE_LOG_TRACE("ETicketServiceImpl::DeleteAllPersonalizedTicketExcludingList()\n");

    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    // パーソナライズドチケットの場合は動的チケットは除いて削除する
    m_PersonalizedTicketDatabase.DeleteAllDeviceLinkedTicketExcludingList(ticketIdExclusionList.GetData(), ticketIdExclusionList.GetLength());
    m_TicketMetaRecordDatabase.DeleteAllTicketExcludingList(ticketIdExclusionList.GetData(), ticketIdExclusionList.GetLength());

    NN_ABORT_UNLESS_RESULT_SUCCESS(fs::CommitSaveData(PersonalizedTicketDbMountName));
    NN_ABORT_UNLESS_RESULT_SUCCESS(fs::CommitSaveData(TicketMetaRecordDbMountName));

    NN_RESULT_SUCCESS;
}

// Common チケットの数を取得します。
Result ETicketServiceImpl::CountCommonTicket(sf::Out<std::int32_t> pOutCount) NN_NOEXCEPT
{
    NN_ES_SERVICE_LOG_TRACE("ETicketServiceImpl::CountCommonTicket()\n");

    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    *pOutCount = m_CommonTicketDatabase.CountTicket();

    NN_RESULT_SUCCESS;
}

// Personalized チケットの数を取得します。
Result ETicketServiceImpl::CountPersonalizedTicket(sf::Out<std::int32_t> pOutCount) NN_NOEXCEPT
{
    NN_ES_SERVICE_LOG_TRACE("ETicketServiceImpl::CountPersonalizedTicket()\n");

    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    *pOutCount = m_PersonalizedTicketDatabase.CountTicket();

    NN_RESULT_SUCCESS;
}

// Common チケットのリストを RightsID で取得します。
Result ETicketServiceImpl::ListCommonTicket(sf::Out<std::int32_t> pOutCount, const sf::OutArray<RightsIdIncludingKeyId>& outRightsIdList) NN_NOEXCEPT
{
    NN_ES_SERVICE_LOG_TRACE("ETicketServiceImpl::ListCommonTicket()\n");

    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    int count = m_CommonTicketDatabase.GetRightsIdList(outRightsIdList.GetData(), outRightsIdList.GetLength());

    // RightsId の重複を削除
    std::sort(outRightsIdList.GetData(), outRightsIdList.GetData() + count);
    auto pArrayEnd = std::unique(outRightsIdList.GetData(), outRightsIdList.GetData() + count);
    *pOutCount = static_cast<int32_t>(pArrayEnd - outRightsIdList.GetData());

    NN_RESULT_SUCCESS;
}

// Personalized チケットのリストを RightsID で取得します。
Result ETicketServiceImpl::ListPersonalizedTicket(sf::Out<std::int32_t> pOutCount, const sf::OutArray<RightsIdIncludingKeyId>& outRightsIdList) NN_NOEXCEPT
{
    NN_ES_SERVICE_LOG_TRACE("ETicketServiceImpl::ListPersonalizedTicket()\n");

    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    int count = m_PersonalizedTicketDatabase.GetRightsIdList(outRightsIdList.GetData(), outRightsIdList.GetLength());

    // RightsId の重複を削除
    std::sort(outRightsIdList.GetData(), outRightsIdList.GetData() + count);
    auto pArrayEnd = std::unique(outRightsIdList.GetData(), outRightsIdList.GetData() + count);
    *pOutCount = static_cast<int32_t>(pArrayEnd - outRightsIdList.GetData());

    NN_RESULT_SUCCESS;
}

// TicketIdList[] の中から保有していない Personalized チケットのリストを取得します。
Result ETicketServiceImpl::ListMissingPersonalizedTicket(sf::Out<std::int32_t> pOutCount, const sf::OutArray<TicketId>& outTicketIdList, const sf::InArray<TicketId>& ticketIdList) NN_NOEXCEPT
{
    NN_ES_SERVICE_LOG_TRACE("ETicketServiceImpl::ListMissingPersonalizedTicket()\n");

    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    *pOutCount = m_PersonalizedTicketDatabase.ListMissingTicket(outTicketIdList.GetData(), outTicketIdList.GetLength(), ticketIdList.GetData(), ticketIdList.GetLength());

    NN_RESULT_SUCCESS;
}

// Common チケットのサイズを取得します。(最初に見つかったチケットのデータを取得します。)
Result ETicketServiceImpl::GetCommonTicketSize(sf::Out<std::uint64_t> pOutSize, const RightsIdIncludingKeyId& rightsId) NN_NOEXCEPT
{
    NN_ES_SERVICE_LOG_TRACE("ETicketServiceImpl::GetCommonTicketSize()\n");

    // 内部鍵に相当する RightsId チケットは チケット DB に存在しない
    NN_RESULT_THROW_UNLESS(rightsId.IsExternalKey(), ResultRightsIdNotFound());

    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    // RightsId に対応した TicketId を探す
    TicketId ticketId;
    NN_RESULT_DO(m_CommonTicketDatabase.FindTicket(&ticketId, rightsId));

    // チケットのサイズを取得
    *pOutSize = m_CommonTicketDatabase.GetTicketSize(rightsId, ticketId);

    NN_RESULT_SUCCESS;
}

// Common チケットのデータを取得します。(最初に見つかったチケットのデータを取得します。)
Result ETicketServiceImpl::GetCommonTicketData(sf::Out<std::uint64_t> pOutSize, const sf::OutBuffer& pOutBuffer, const RightsIdIncludingKeyId& rightsId) NN_NOEXCEPT
{
    NN_ES_SERVICE_LOG_TRACE("ETicketServiceImpl::GetCommonTicketData()\n");

    // 内部鍵に相当する RightsId チケットは チケット DB に存在しない
    NN_RESULT_THROW_UNLESS(rightsId.IsExternalKey(), ResultRightsIdNotFound());

    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    // RightsId に対応した TicketId を探す
    TicketId ticketId;
    NN_RESULT_DO(m_CommonTicketDatabase.FindTicket(&ticketId, rightsId));

    // チケットのデータを取得
    size_t ticketSize;
    NN_RESULT_DO(m_CommonTicketDatabase.GetTicketData(&ticketSize, pOutBuffer.GetPointerUnsafe(), pOutBuffer.GetSize(), rightsId, ticketId));

    *pOutSize = static_cast<int64_t>(ticketSize);

    NN_RESULT_SUCCESS;
}

// チケットがあるかどうかを確認します。
Result ETicketServiceImpl::OwnTicket(const sf::OutArray<bool>& outOwnTicketList, const sf::InArray<RightsIdIncludingKeyId>& rightsIdList) NN_NOEXCEPT
{
    NN_ES_SERVICE_LOG_TRACE("ETicketServiceImpl::OwnTicket()\n");

    NN_RESULT_THROW_UNLESS(outOwnTicketList.GetLength() == rightsIdList.GetLength(), ResultInvalidParameter());

    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    std::fill(outOwnTicketList.GetData(), outOwnTicketList.GetData() + outOwnTicketList.GetLength(), false);

    // Common チケットから探す
    m_CommonTicketDatabase.OwnTicketList(outOwnTicketList.GetData(), rightsIdList.GetData(), rightsIdList.GetLength());

    // Personalized チケットから探す
    m_PersonalizedTicketDatabase.OwnTicketList(outOwnTicketList.GetData(), rightsIdList.GetData(), rightsIdList.GetLength());

    NN_RESULT_SUCCESS;
}

// チケットの情報を取得します。
Result ETicketServiceImpl::GetTicketInfo(sf::Out<std::int32_t> pOutCount, const sf::OutArray<TicketInfo>& outTicketInfoList, const sf::InArray<RightsIdIncludingKeyId>& rightsIdList) NN_NOEXCEPT
{
    NN_ES_SERVICE_LOG_TRACE("ETicketServiceImpl::GetTicketInfo()\n");

    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    int count = 0;

    // Common チケットから探す
    count += m_CommonTicketDatabase.GetTicketInfoList(outTicketInfoList.GetData(), outTicketInfoList.GetLength(), rightsIdList.GetData(), rightsIdList.GetLength());

    // Personalized チケットから探す
    count += m_PersonalizedTicketDatabase.GetTicketInfoList(outTicketInfoList.GetData() + count, outTicketInfoList.GetLength() - count, rightsIdList.GetData(), rightsIdList.GetLength());

    *pOutCount = static_cast<int32_t>(count);

    NN_RESULT_SUCCESS;
}

Result ETicketServiceImpl::ListLightTicketInfo(sf::Out<std::int32_t> pOutCount, const sf::OutArray<LightTicketInfo>& outTicketInfoList, const RightsIdIncludingKeyId& rightsId) NN_NOEXCEPT
{
    NN_ES_SERVICE_LOG_TRACE("ETicketServiceImpl::ListLightTicketInfo()\n");

    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    int count = 0;

    // Common チケットから探す
    count += m_CommonTicketDatabase.ListLightTicketInfo(outTicketInfoList.GetData(), outTicketInfoList.GetLength(), rightsId);

    // Personalized チケットから探す
    count += m_PersonalizedTicketDatabase.ListLightTicketInfo(outTicketInfoList.GetData() + count, outTicketInfoList.GetLength() - count, rightsId);

    *pOutCount = static_cast<int32_t>(count);

    NN_RESULT_SUCCESS;
}

// Challenge データに対して署名と証明書を取得します。
Result ETicketServiceImpl::SignData(sf::Out<Sign> pOutSign, sf::Out<Certificate> pOutCertificate, const sf::InBuffer& pData) NN_NOEXCEPT
{
    NN_ES_SERVICE_LOG_TRACE("ETicketServiceImpl::SignData()\n");

    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    const nn::escore::ESTitleId ShopTitleId = 0x010000000000100b;
    MemoryInputStream dataReader(pData.GetPointerUnsafe(), static_cast<unsigned int>(pData.GetSize()));
    nn::escore::ETicketService signer;
    nn::escore::ESError esError;

    // 署名と証明書の取得
    esError = signer.Sign(ShopTitleId, dataReader, static_cast<unsigned int>(pData.GetSize()), pOutSign.GetPointer(), sizeof(Sign), pOutCertificate.GetPointer(), sizeof(Certificate));
    NN_ABORT_UNLESS(esError == ES_ERR_OK, "A device certificate or a private key isn't established right.");

    NN_RESULT_SUCCESS;
}

Result ETicketServiceImpl::GetCommonTicketAndCertificateSize(sf::Out<std::uint64_t> pOutTicketSize, sf::Out<std::uint64_t> pOutCertificateSize, const es::RightsIdIncludingKeyId& rightsId) NN_NOEXCEPT
{
    NN_ES_SERVICE_LOG_TRACE("ETicketServiceImpl::GetCommonTicketAndCertificateSize()\n");

    // 内部鍵に相当する RightsId チケットは チケット DB に存在しない
    NN_RESULT_THROW_UNLESS(rightsId.IsExternalKey(), ResultRightsIdNotFound());

    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    // RightsId に対応した TicketId を探す
    TicketId ticketId;
    NN_RESULT_DO(m_CommonTicketDatabase.FindTicket(&ticketId, rightsId));

    // チケットと証明書のサイズを取得
    *pOutTicketSize = m_CommonTicketDatabase.GetTicketSize(rightsId, ticketId);
    *pOutCertificateSize = m_CommonTicketDatabase.GetCertificateSize(rightsId, ticketId);

    NN_RESULT_SUCCESS;
}

Result ETicketServiceImpl::GetCommonTicketAndCertificateData(nn::sf::Out<std::uint64_t> pOutTicketSize, nn::sf::Out<std::uint64_t> pOutCertificateSize, const nn::sf::OutBuffer& pOutTicketBuffer, const nn::sf::OutBuffer& pOutCertificateBuffer, const nn::es::RightsIdIncludingKeyId& rightsId) NN_NOEXCEPT
{
    NN_ES_SERVICE_LOG_TRACE("ETicketServiceImpl::GetCommonTicketAndCertificateData()\n");

    // 内部鍵に相当する RightsId チケットは チケット DB に存在しない
    NN_RESULT_THROW_UNLESS(rightsId.IsExternalKey(), ResultRightsIdNotFound());

    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    // RightsId に対応した TicketId を探す
    TicketId ticketId;
    NN_RESULT_DO(m_CommonTicketDatabase.FindTicket(&ticketId, rightsId));

    // チケットと証明書のサイズを取得
    size_t ticketSize;
    NN_RESULT_DO(m_CommonTicketDatabase.GetTicketData(&ticketSize, pOutTicketBuffer.GetPointerUnsafe(), pOutTicketBuffer.GetSize(), rightsId, ticketId));

    *pOutTicketSize = static_cast<int64_t>(ticketSize);
    *pOutCertificateSize = m_CommonTicketDatabase.GetCertificateData(pOutCertificateBuffer.GetPointerUnsafe(),pOutCertificateBuffer.GetSize(), rightsId, ticketId);

    NN_RESULT_SUCCESS;
}

Result ETicketServiceImpl::ImportPrepurchaseRecord(const nn::es::PrepurchaseRecord& record) NN_NOEXCEPT
{
    NN_ES_SERVICE_LOG_TRACE("ETicketServiceImpl::ImportPrepurchaseRecord()\n");

    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    NN_RESULT_DO(m_TicketMetaRecordDatabase.ImportTicket(GetTicketMetaRecordFromPrepurchaseRecord(record)));

    NN_ABORT_UNLESS_RESULT_SUCCESS(fs::CommitSaveData(TicketMetaRecordDbMountName));

    NN_RESULT_SUCCESS;
}

Result ETicketServiceImpl::DeletePrepurchaseRecord(const nn::es::PrepurchaseRecord& record) NN_NOEXCEPT
{
    NN_ES_SERVICE_LOG_TRACE("ETicketServiceImpl::DeletePrepurchaseRecord()\n");

    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    // 本来はメタタイプが Prepurchase のレコードのみを操作する必要があるが、メタタイプが Prepurchase 以外のタイプが存在しないため処理を省略
    m_TicketMetaRecordDatabase.DeleteTicket(record.rightsId, record.ticketId);

    NN_ABORT_UNLESS_RESULT_SUCCESS(fs::CommitSaveData(TicketMetaRecordDbMountName));

    NN_RESULT_SUCCESS;
}

Result ETicketServiceImpl::DeleteAllPrepurchaseRecord() NN_NOEXCEPT
{
    NN_ES_SERVICE_LOG_TRACE("ETicketServiceImpl::DeleteAllPrepurchaseRecord()\n");

    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    // 本来はメタタイプが Prepurchase のレコードのみを操作する必要があるが、メタタイプが Prepurchase 以外のタイプが存在しないため処理を省略
    m_TicketMetaRecordDatabase.DeleteAllTicket();

    NN_ABORT_UNLESS_RESULT_SUCCESS(fs::CommitSaveData(TicketMetaRecordDbMountName));

    NN_RESULT_SUCCESS;
}

Result ETicketServiceImpl::CountPrepurchaseRecord(nn::sf::Out<std::int32_t> pOutCount) NN_NOEXCEPT
{
    NN_ES_SERVICE_LOG_TRACE("ETicketServiceImpl::CountPrepurchaseRecord()\n");

    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    // 本来はメタタイプが Prepurchase のレコードのみを操作する必要があるが、メタタイプが Prepurchase 以外のタイプが存在しないため処理を省略
    *pOutCount = m_TicketMetaRecordDatabase.CountTicket();

    NN_RESULT_SUCCESS;
}

Result ETicketServiceImpl::ListPrepurchaseRecord(nn::sf::Out<std::int32_t> pOutCount, const nn::sf::OutArray<nn::es::RightsIdIncludingKeyId>& outRightsIdList) NN_NOEXCEPT
{
    NN_ES_SERVICE_LOG_TRACE("ETicketServiceImpl::ListPrepurchaseRecord()\n");

    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    // 本来はメタタイプが Prepurchase のレコードのみを操作する必要があるが、メタタイプが Prepurchase 以外のタイプが存在しないため処理を省略
    int count = m_TicketMetaRecordDatabase.GetRightsIdList(outRightsIdList.GetData(), outRightsIdList.GetLength());

    // RightsId の重複を削除
    std::sort(outRightsIdList.GetData(), outRightsIdList.GetData() + count);
    auto pArrayEnd = std::unique(outRightsIdList.GetData(), outRightsIdList.GetData() + count);
    *pOutCount = static_cast<int32_t>(pArrayEnd - outRightsIdList.GetData());

    NN_RESULT_SUCCESS;
}

Result ETicketServiceImpl::ListPrepurchaseRecordInfo(nn::sf::Out<std::int32_t> pOutCount, const nn::sf::OutArray<nn::es::PrepurchaseRecord>& outPrepurchaseRecordList, const nn::es::RightsIdIncludingKeyId& rightsId) NN_NOEXCEPT
{
    NN_ES_SERVICE_LOG_TRACE("ETicketServiceImpl::ListPrepurchaseRecordInfo()\n");

    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    std::unique_ptr<TicketMetaRecord[]> list(new TicketMetaRecord[outPrepurchaseRecordList.GetLength()]);

    // RightsId が一致している予約購入情報を取得する
    *pOutCount = m_TicketMetaRecordDatabase.ListTicketMetaRecordInfo(list.get(), outPrepurchaseRecordList.GetLength(), rightsId, TicketMetaRecordType_Prepurchase);

    for (int i = 0; i < *pOutCount; i++)
    {
        outPrepurchaseRecordList[i] = GetPrepurchaseRecordFromTicketMetaRecord(list[i]);
    }

    NN_RESULT_SUCCESS;
}

nn::Result ETicketServiceImpl::CountDesignatedTicket(nn::sf::Out<std::int32_t> pOutCount, nn::es::RightsId rightsId) NN_NOEXCEPT
{
    NN_ES_SERVICE_LOG_TRACE("ETicketServiceImpl::CountDesignatedTicket()\n");

    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    *pOutCount = m_CommonTicketDatabase.CountTicket(rightsId) + m_PersonalizedTicketDatabase.CountTicket(rightsId);

    NN_RESULT_SUCCESS;
}

nn::Result ETicketServiceImpl::ListDesignatedTicket(nn::sf::Out<std::int32_t> pOutCount, const nn::sf::OutArray<nn::es::RightsIdIncludingKeyId>& outRightsIdList, nn::es::RightsId rightsId) NN_NOEXCEPT
{
    NN_ES_SERVICE_LOG_TRACE("ETicketServiceImpl::ListDesignatedTicket()\n");

    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    int count = 0;
    count += m_CommonTicketDatabase.GetRightsIdList(outRightsIdList.GetData(), outRightsIdList.GetLength(), rightsId);
    count += m_PersonalizedTicketDatabase.GetRightsIdList(outRightsIdList.GetData() + count, outRightsIdList.GetLength() - count, rightsId);

    // RightsId の重複を削除
    std::sort(outRightsIdList.GetData(), outRightsIdList.GetData() + count);
    auto pArrayEnd = std::unique(outRightsIdList.GetData(), outRightsIdList.GetData() + count);
    *pOutCount = static_cast<int32_t>(pArrayEnd - outRightsIdList.GetData());

    NN_RESULT_SUCCESS;
}

nn::Result ETicketServiceImpl::CountDesignatedPrepurchaseRecord(nn::sf::Out<std::int32_t> pOutCount, nn::es::RightsId rightsId) NN_NOEXCEPT
{
    NN_ES_SERVICE_LOG_TRACE("ETicketServiceImpl::CountDesignatedPrepurchaseRecord()\n");

    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    // 本来はメタタイプが Prepurchase のレコードのみを操作する必要があるが、メタタイプが Prepurchase 以外のタイプが存在しないため処理を省略
    *pOutCount = m_TicketMetaRecordDatabase.CountTicket(rightsId);

    NN_RESULT_SUCCESS;
}

nn::Result ETicketServiceImpl::ListDesignatedPrepurchaseRecord(nn::sf::Out<std::int32_t> pOutCount, const nn::sf::OutArray<nn::es::RightsIdIncludingKeyId>& outRightsIdList, nn::es::RightsId rightsId) NN_NOEXCEPT
{
    NN_ES_SERVICE_LOG_TRACE("ETicketServiceImpl::ListDesignatedPrepurchaseRecord()\n");

    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    // 本来はメタタイプが Prepurchase のレコードのみを操作する必要があるが、メタタイプが Prepurchase 以外のタイプが存在しないため処理を省略
    int count = m_TicketMetaRecordDatabase.GetRightsIdList(outRightsIdList.GetData(), outRightsIdList.GetLength(), rightsId);

    // RightsId の重複を削除
    std::sort(outRightsIdList.GetData(), outRightsIdList.GetData() + count);
    auto pArrayEnd = std::unique(outRightsIdList.GetData(), outRightsIdList.GetData() + count);
    *pOutCount = static_cast<int32_t>(pArrayEnd - outRightsIdList.GetData());

    NN_RESULT_SUCCESS;
}

Result ETicketServiceImpl::GetEncryptedTicketSize(nn::sf::Out<std::uint64_t> pOutSize, const nn::es::RightsIdIncludingKeyId& rightsId) NN_NOEXCEPT
{
    NN_ES_SERVICE_LOG_TRACE("ETicketServiceImpl::GetEncryptedTicketSize()\n");

    // 内部鍵に相当する RightsId チケットは チケット DB に存在しない
    NN_RESULT_THROW_UNLESS(rightsId.IsExternalKey(), ResultRightsIdNotFound());

    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    // RightsId に対応した TicketId を探す
    TicketId ticketId;
    NN_RESULT_TRY(m_CommonTicketDatabase.FindTicket(&ticketId, rightsId))
        NN_RESULT_CATCH(ResultRightsIdNotFound)
        {
            NN_RESULT_DO(m_PersonalizedTicketDatabase.FindTicket(&ticketId, rightsId));
            *pOutSize = m_PersonalizedTicketDatabase.GetEncryptedTicketSize(rightsId, ticketId);

            NN_RESULT_SUCCESS;
        }
    NN_RESULT_END_TRY

    *pOutSize = m_CommonTicketDatabase.GetEncryptedTicketSize(rightsId, ticketId);

    NN_RESULT_SUCCESS;
}

Result ETicketServiceImpl::GetEncryptedTicketData(nn::sf::Out<std::uint64_t> pOutTicketId, nn::sf::Out<std::uint64_t> pOutSize, const nn::sf::OutBuffer& pOutBuffer, const nn::sf::OutBuffer& pOutKey, const nn::es::RightsIdIncludingKeyId& rightsId) NN_NOEXCEPT
{
    NN_ES_SERVICE_LOG_TRACE("ETicketServiceImpl::GetEncryptedTicketData()\n");

    // 内部鍵に相当する RightsId チケットは チケット DB に存在しない
    NN_RESULT_THROW_UNLESS(rightsId.IsExternalKey(), ResultRightsIdNotFound());

    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    // RightsId に対応した TicketId を探す
    TicketId ticketId;
    NN_RESULT_TRY(m_CommonTicketDatabase.FindTicket(&ticketId, rightsId))
        NN_RESULT_CATCH(ResultRightsIdNotFound)
        {
            NN_RESULT_DO(m_PersonalizedTicketDatabase.FindTicket(&ticketId, rightsId));

            size_t ticketSize;
            NN_RESULT_DO(m_PersonalizedTicketDatabase.GetEncryptedTicketData(&ticketSize, pOutBuffer.GetPointerUnsafe(), pOutBuffer.GetSize(), pOutKey.GetPointerUnsafe(), pOutKey.GetSize(), rightsId, ticketId));
            *pOutSize = ticketSize;

            NN_RESULT_SUCCESS;
        }
    NN_RESULT_END_TRY

    size_t ticketSize;
    NN_RESULT_DO(m_CommonTicketDatabase.GetEncryptedTicketData(&ticketSize, pOutBuffer.GetPointerUnsafe(), pOutBuffer.GetSize(), pOutKey.GetPointerUnsafe(), pOutKey.GetSize(), rightsId, ticketId));
    *pOutSize = ticketSize;
    *pOutTicketId = ticketId;

    NN_RESULT_SUCCESS;
}

nn::Result ETicketServiceImpl::CheckRightsStatusIncludingKeyId(const nn::sf::OutArray<nn::es::RightsStatus>& outStatusList, const nn::sf::InArray<nn::es::RightsIdIncludingKeyId>& rightsIdList, const nn::sf::InArray<nn::es::ELicenseUserId>& userIdList, bool onlyAllAccountRights) NN_NOEXCEPT
{
    NN_ES_SERVICE_LOG_TRACE("ETicketServiceImpl::CheckRightsStatusIncludingKeyId()\n");

    NN_RESULT_THROW_UNLESS(outStatusList.GetLength() == rightsIdList.GetLength(), ResultInvalidParameter());

    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    NN_RESULT_DO(GetRightsAvailabilityMaster().CheckRightsStatusIncludingKeyId(outStatusList.GetData(), outStatusList.GetLength(), rightsIdList.GetData(), rightsIdList.GetLength(), userIdList.GetData(), userIdList.GetLength(), onlyAllAccountRights));

    NN_RESULT_SUCCESS;
}

nn::Result ETicketServiceImpl::CheckRightsStatus(const nn::sf::OutArray<nn::es::RightsStatus>& outStatusList, const nn::sf::InArray<nn::es::RightsId>& rightsIdList, const nn::sf::InArray<nn::es::ELicenseUserId>& userIdList, bool onlyAllAccountRights) NN_NOEXCEPT
{
    NN_ES_SERVICE_LOG_TRACE("ETicketServiceImpl::CheckRightsStatus()\n");

    NN_RESULT_THROW_UNLESS(outStatusList.GetLength() == rightsIdList.GetLength(), ResultInvalidParameter());

    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    NN_RESULT_DO(GetRightsAvailabilityMaster().CheckRightsStatus(outStatusList.GetData(), outStatusList.GetLength(), rightsIdList.GetData(), rightsIdList.GetLength(), userIdList.GetData(), userIdList.GetLength(), onlyAllAccountRights));

    NN_RESULT_SUCCESS;
}

nn::Result ETicketServiceImpl::RegisterTitleKey(const nn::sf::InArray<nn::es::RightsIdIncludingKeyId>& rightsIdList, const nn::sf::InArray<std::int32_t>& keyGenerationList) NN_NOEXCEPT
{
    NN_ES_SERVICE_LOG_TRACE("ETicketServiceImpl::RegisterTitleKey() keyCount=%lld rightsId[0]=0x%016llx\n", rightsIdList.GetLength(), rightsIdList[0].GetRightsId());

    NN_RESULT_THROW_UNLESS(rightsIdList.GetLength() == keyGenerationList.GetLength(), ResultInvalidParameter());

    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    NN_RESULT_DO(GetRightsAvailabilityMaster().RegisterTitleKeys(rightsIdList, keyGenerationList, util::MakeSpan(m_RestrictedRightsUserList.data(), m_RestrictedRightsUserList.size())));
    NN_RESULT_SUCCESS;
}

nn::Result ETicketServiceImpl::UnregisterAllTitleKey() NN_NOEXCEPT
{
    NN_ES_SERVICE_LOG_TRACE("ETicketServiceImpl::UnregisterAllTitleKey()\n");

    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    NN_RESULT_DO(GetRightsAvailabilityMaster().UnregisterAllTitleKeys());
    NN_RESULT_SUCCESS;
}

nn::Result ETicketServiceImpl::RegisterAccountRestrictedRightsUser(nn::es::ELicenseUserId userId) NN_NOEXCEPT
{
    NN_ES_SERVICE_LOG_TRACE("ETicketServiceImpl::RegisterAccountRestrictedRightsUser()\n");

    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    if (m_RestrictedRightsUserList.size() >= m_RestrictedRightsUserList.max_size())
    {
        NN_RESULT_THROW(ResultUserListFull());
    }

    if (std::find(m_RestrictedRightsUserList.begin(), m_RestrictedRightsUserList.end(), userId) != m_RestrictedRightsUserList.end())
    {
        NN_RESULT_THROW(ResultUserAlreadyRegistered());
    }

    m_RestrictedRightsUserList.push_back(userId);

    NN_RESULT_SUCCESS;
}

nn::Result ETicketServiceImpl::UnregisterAllAccountRestrictedRightsUser() NN_NOEXCEPT
{
    NN_ES_SERVICE_LOG_TRACE("ETicketServiceImpl::UnregisterAllAccountRestrictedRightsUser()\n");

    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    m_RestrictedRightsUserList.clear();

    NN_RESULT_SUCCESS;
}

nn::Result ETicketServiceImpl::ListAccountRestrictedRightsUser(nn::sf::Out<bool> outHasAccount, nn::sf::Out<std::int32_t> outCount, const nn::sf::OutArray<nn::es::ELicenseUserId>& outUserIdList) NN_NOEXCEPT
{
    NN_ES_SERVICE_LOG_TRACE("ETicketServiceImpl::ListAccountRestrictedRightsUser()\n");

    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    if (m_RestrictedRightsUserList.size() == 0)
    {
        *outHasAccount = false;
        *outCount = 0;

        NN_RESULT_SUCCESS;
    }

    size_t count = 0;
    for (const auto& entry : m_RestrictedRightsUserList)
    {
        outUserIdList.GetData()[count] = entry;
        count++;

        if (count >= outUserIdList.GetLength())
        {
            break;
        }
    }

    *outHasAccount = true;
    *outCount = static_cast<int32_t>(count);

    NN_RESULT_SUCCESS;
}

nn::Result ETicketServiceImpl::BeginImportELicenseArchive(nn::sf::Out<nn::es::ELicenseImportContext> pOutContext, nn::es::ELicenseOwnerId ownerId) NN_NOEXCEPT
{
    NN_ES_SERVICE_LOG_TRACE("ETicketServiceImpl::BeginImportELicenseArchive()\n");

    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    NN_RESULT_DO(m_ELicenseManager.BeginImportELicenseArchive(pOutContext.GetPointer(), ownerId));

    NN_RESULT_SUCCESS;
}

nn::Result ETicketServiceImpl::ImportELicenseArchive(const nn::es::ELicenseImportContext& context, const nn::sf::InBuffer& eLicenseArchive) NN_NOEXCEPT
{
    NN_ES_SERVICE_LOG_TRACE("ETicketServiceImpl::ImportELicenseArchive()\n");

    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    NN_RESULT_DO(m_ELicenseManager.ImportELicenseArchive(context, eLicenseArchive.GetPointerUnsafe(), eLicenseArchive.GetSize()));

    NN_RESULT_SUCCESS;
}

nn::Result ETicketServiceImpl::EndImportELicenseArchive(nn::sf::Out<nn::es::ELicenseArchiveId> pOutELicenseArchiveId, const nn::es::ELicenseImportContext& context) NN_NOEXCEPT
{
    NN_ES_SERVICE_LOG_TRACE("ETicketServiceImpl::EndImportELicenseArchive()\n");

    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    NN_RESULT_DO(m_ELicenseManager.EndImportELicenseArchive(pOutELicenseArchiveId.GetPointer(), context));

    NN_ABORT_UNLESS_RESULT_SUCCESS(fs::CommitSaveData(ELicenseArchiveStoreMountName));

    // 全権利コンテキストの eLicense による権利有無を更新
    NN_ABORT_UNLESS_RESULT_SUCCESS(GetRightsAvailabilityMaster().UpdateAllRightsIdListsValidity());
    NN_RESULT_SUCCESS;
}

nn::Result ETicketServiceImpl::EndImportELicenseArchiveForDebug(nn::sf::Out<nn::es::ELicenseArchiveId> pOutELicenseArchiveId, const nn::es::ELicenseImportContext& context) NN_NOEXCEPT
{
    NN_ES_SERVICE_LOG_TRACE("ETicketServiceImpl::EndImportELicenseArchiveForDebug()\n");

    NN_RESULT_THROW_UNLESS(IsDebugApiEnabled(), ResultDevelopmentFunctionCalled());

    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    NN_RESULT_DO(m_ELicenseManager.EndImportELicenseArchiveForDebug(pOutELicenseArchiveId.GetPointer(), context));

    NN_ABORT_UNLESS_RESULT_SUCCESS(fs::CommitSaveData(ELicenseArchiveStoreMountName));

    // 全権利コンテキストの eLicense による権利有無を更新
    NN_ABORT_UNLESS_RESULT_SUCCESS(GetRightsAvailabilityMaster().UpdateAllRightsIdListsValidity());
    NN_RESULT_SUCCESS;
}

nn::Result ETicketServiceImpl::CountELicense(nn::sf::Out<std::int32_t> pOutCount) NN_NOEXCEPT
{
    NN_ES_SERVICE_LOG_TRACE("ETicketServiceImpl::CountELicense()\n");

    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    *pOutCount = m_ELicenseList.Count();

    NN_RESULT_SUCCESS;
}

nn::Result ETicketServiceImpl::ListELicenseIds(nn::sf::Out<std::int32_t> pOutCount, const nn::sf::OutArray<nn::es::ELicenseId>& outELicenseIdList) NN_NOEXCEPT
{
    NN_ES_SERVICE_LOG_TRACE("ETicketServiceImpl::ListELicenseIds()\n");

    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    *pOutCount = m_ELicenseList.List(outELicenseIdList.GetData(), static_cast<int>(outELicenseIdList.GetLength()));

    NN_RESULT_SUCCESS;
}

nn::Result ETicketServiceImpl::ListELicenseInfo(nn::sf::Out<std::int32_t> pOutCount, const nn::sf::OutArray<nn::es::ELicenseInfoWrapper>& outELicenseInfoList, const nn::sf::InArray<nn::es::ELicenseId>& eLicenseIdList) NN_NOEXCEPT
{
    NN_ES_SERVICE_LOG_TRACE("ETicketServiceImpl::ListELicenseInfo()\n");

    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    NN_RESULT_DO(ListELicenseInfoImpl(pOutCount.GetPointer(), outELicenseInfoList.GetData(), static_cast<int>(outELicenseInfoList.GetLength()), eLicenseIdList.GetData(), static_cast<int>(eLicenseIdList.GetLength()), [](const ELicenseInfoForSystem& info) -> ELicenseInfo
    {
        return GetELicenseInfo(info);
    }));

    NN_RESULT_SUCCESS;
}

nn::Result ETicketServiceImpl::ListELicenseInfoForDebug(nn::sf::Out<std::int32_t> pOutCount, const nn::sf::OutArray<nn::es::ELicenseInfoForSystemWrapper>& outELicenseInfoList, const nn::sf::InArray<nn::es::ELicenseId>& eLicenseIdList) NN_NOEXCEPT
{
    NN_ES_SERVICE_LOG_TRACE("ETicketServiceImpl::ListELicenseInfoForDebug()\n");

    NN_RESULT_THROW_UNLESS(IsDebugApiEnabled(), ResultDevelopmentFunctionCalled());

    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    NN_RESULT_DO(ListELicenseInfoImpl(pOutCount.GetPointer(), outELicenseInfoList.GetData(), static_cast<int>(outELicenseInfoList.GetLength()), eLicenseIdList.GetData(), static_cast<int>(eLicenseIdList.GetLength()), [](const ELicenseInfoForSystem& info) -> ELicenseInfoForSystem
    {
        return info;
    }));

    NN_RESULT_SUCCESS;
}

nn::Result ETicketServiceImpl::GetELicenseArchiveSizeForDebug(nn::sf::Out<std::uint64_t> pOutSize, nn::es::ELicenseOwnerId ownerId) NN_NOEXCEPT
{
    NN_ES_SERVICE_LOG_TRACE("ETicketServiceImpl::GetELicenseArchiveSizeForDebug()\n");

    NN_RESULT_THROW_UNLESS(IsDebugApiEnabled(), ResultDevelopmentFunctionCalled());

    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    return m_ELicenseManager.GetELicenseArchiveSizeForDebug(pOutSize.GetPointer(), ownerId);
}

nn::Result ETicketServiceImpl::GetELicenseArchiveDataForDebug(nn::sf::Out<std::uint64_t> pOutSize, const nn::sf::OutBuffer& pOutBuffer, nn::es::ELicenseOwnerId ownerId) NN_NOEXCEPT
{
    NN_ES_SERVICE_LOG_TRACE("ETicketServiceImpl::GetELicenseArchiveDataForDebug()\n");

    NN_RESULT_THROW_UNLESS(IsDebugApiEnabled(), ResultDevelopmentFunctionCalled());

    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    return m_ELicenseManager.GetELicenseArchiveDataForDebug(pOutSize.GetPointer(), pOutBuffer.GetPointerUnsafe(), pOutBuffer.GetSize(), ownerId);
}

nn::Result ETicketServiceImpl::DeleteAllELicenseArchiveForDebug() NN_NOEXCEPT
{
    NN_ES_SERVICE_LOG_TRACE("ETicketServiceImpl::DeleteAllELicenseArchiveForDebug()\n");

    NN_RESULT_THROW_UNLESS(IsDebugApiEnabled(), ResultDevelopmentFunctionCalled());

    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    NN_RESULT_DO(m_ELicenseManager.DeleteAllELicenseArchiveForDebug());

    NN_ABORT_UNLESS_RESULT_SUCCESS(fs::CommitSaveData(ELicenseArchiveStoreMountName));

    NN_RESULT_SUCCESS;
}

nn::Result ETicketServiceImpl::GetChallengeOfLocalConcurrencyCheck(nn::sf::Out<nn::es::LocalConcurrencyCheckChallenge> outValue) NN_NOEXCEPT
{
    NN_ES_SERVICE_LOG_TRACE("ETicketServiceImpl::GetChallengeOfLocalConcurrencyCheck()\n");

    // TORIAEZU: ダミーのチャレンジを入れる
    // TODO: 正しいチャレンジを生成する
    es::LocalConcurrencyCheckChallenge challenge = {};
    challenge.value = 1;
    *outValue = challenge;

    NN_RESULT_SUCCESS;
}

nn::Result ETicketServiceImpl::AuthorizeLocalConcurrencyCheckData(nn::sf::Out<nn::es::LocalConcurrencyCheckAuthenticationInfo> outInfo, const nn::sf::InArray<nn::es::LocalConcurrencyCheckData>& authenticatedDataList, const nn::es::LocalConcurrencyCheckData& unauthenticatedData) NN_NOEXCEPT
{
    NN_ES_SERVICE_LOG_TRACE("ETicketServiceImpl::AuthorizeLocalConcurrencyCheckData()\n");

    // デバッグ用のローカル同時利用判定が有効ならばデバッグ用の判定を行い結果を返す
    if (IsLocalConcurrencyCheckForDebugEnabled())
    {
        // 認証を行う LocalConcurrencyCheckData の 1byte 目が 0 なら成功とする
        if (unauthenticatedData.data[0] == 0)
        {
            std::memset(outInfo.GetPointer(), 0, sizeof(outInfo.Get()));
            NN_RESULT_SUCCESS;
        }
        // 認証を行う LocalConcurrencyCheckData の 1byte 目が 0 以外なら失敗とする
        else
        {
            NN_RESULT_THROW(ResultLocalConcurrencyCheckDuplicationRightsFound());
        }
    }

    // TODO: 正しいローカル同時利用判定を行う
    std::memset(outInfo.GetPointer(), 0, sizeof(outInfo.Get()));

    NN_RESULT_SUCCESS;
}

nn::Result ETicketServiceImpl::GetLocalConcurrencyCheckData(nn::sf::Out<nn::es::LocalConcurrencyCheckData> outData, nn::es::LocalConcurrencyCheckChallenge challenge) NN_NOEXCEPT
{
    NN_ES_SERVICE_LOG_TRACE("ETicketServiceImpl::GetLocalConcurrencyCheckData()\n");

    // デバッグ用のローカル同時利用判定が有効ならばデバッグ用の判定を行い結果を返す
    if (IsLocalConcurrencyCheckForDebugEnabled())
    {
        std::memset(outData.GetPointer(), 0, sizeof(outData.Get()));

        bool isCheckSucceeded = true;
        size_t size = settings::fwdbg::GetSettingsItemValue(&isCheckSucceeded, sizeof(isCheckSucceeded), "es.debug", "local_concurrency_check_for_debug_check_result");

        if (size == sizeof(isCheckSucceeded) && !isCheckSucceeded)
        {
            // 失敗の場合、LocalConcurrencyCheckData の 1byte 目を 1 としておく
            outData->data[0] = 1;
        }
        else // 値の取得に失敗した場合は成功としておく
        {
            // 成功の場合、LocalConcurrencyCheckData の 1byte 目を 0 としておく
            outData->data[0] = 0;
        }
    }

    // TODO: 正しい権利利用情報を作成する
    std::memset(outData.GetPointer(), 0, sizeof(outData.Get()));

    NN_RESULT_SUCCESS;
}

nn::Result ETicketServiceImpl::VerifyLocalConcurrencyAuthentication(const nn::es::LocalConcurrencyCheckAuthenticationInfo& info) NN_NOEXCEPT
{
    NN_ES_SERVICE_LOG_TRACE("ETicketServiceImpl::VerifyLocalConcurrencyAuthentication()\n");

    // TODO: 正しく認証情報の検証を行う
    NN_RESULT_SUCCESS;
}

nn::Result ETicketServiceImpl::CreateActiveRightsContext(sf::Out<nn::sf::SharedPointer<es::IActiveRightsContext>> pOut) NN_NOEXCEPT
{
    NN_ES_SERVICE_LOG_TRACE("ETicketServiceImpl::CreateActiveRightsContext()\n");

    *pOut = GetRightsAvailabilityMaster().CreateActiveRightsContext(this);
    NN_RESULT_SUCCESS;
}

// 開始処理を行います。
Result ETicketServiceImpl::InitializeFile() NN_NOEXCEPT
{
    NN_ES_SERVICE_LOG_TRACE("ETicketServiceImpl::InitializeFile()\n");

    // escore が製品鍵と開発鍵どちらを使うかの設定
    // 現状、証明書の検証に開発版と製品版どちらのルート公開鍵を使うかのみにこの設定を参照している。
    // チケットの署名鍵に対応したルート公開鍵を使う必要があり、本来はサーバでどちらの鍵で署名されたかで分岐する必要があるが、
    // 実用上、開発機では開発用サーバで署名されたチケットが使われ、製品機では製品用サーバで署名されたチケットが使われるとして問題ないため、開発機かどうか判別する。
    nn::escore::SetUseProdETicketKey(!IsDevelopment());

    // チケットDB の初期化
    SetupTicketDB();

    // デバイス証明書とデバイス秘密鍵の初期化
    SetupDeviceCertificateAndDeviceKey();

    // 保存されている eLicense アーカイブの読み込み
    NN_RESULT_DO(m_ELicenseManager.ReadAllSavedELicenseArchive());

    NN_RESULT_SUCCESS;
}

template <typename T, typename ConvertELicenseInfoFunc>
nn::Result ETicketServiceImpl::ListELicenseInfoImpl(int* pOutCount, T* outELicenseInfoList, int eLicenseInfoCount, const nn::es::ELicenseId eLicenseIdList[], int eLicenseIdCount, ConvertELicenseInfoFunc convertFunc) NN_NOEXCEPT
{
    NN_ES_SERVICE_LOG_TRACE("ETicketServiceImpl::ListELicenseInfoImpl()\n");

    int totalCount = 0;
    const int MaxOneReadCount = 32;
    ELicenseInfoForSystem infoList[MaxOneReadCount];
    bool hasTicketList[MaxOneReadCount];
    TicketId ticketIdList[MaxOneReadCount];

    for (int i = 0; i < eLicenseIdCount; i += MaxOneReadCount)
    {
        int readELicenseIdCount = std::min(eLicenseIdCount - i, MaxOneReadCount);

        // eLicense のリストから情報を取得する
        int count = m_ELicenseList.Find(infoList, MaxOneReadCount, eLicenseIdList + i, readELicenseIdCount);

        for (int j = 0; j < count; j++)
        {
            ticketIdList[j] = infoList[j].ticketId;
        }

        std::fill(hasTicketList, hasTicketList + MaxOneReadCount, false);

        // eLicense に対応するチケットを保有しているか確認する
        m_PersonalizedTicketDatabase.OwnTicketList(hasTicketList, ticketIdList, count);

        for (int j = 0; j < count; j++)
        {
            outELicenseInfoList[totalCount + j].info = convertFunc(infoList[j]);
            outELicenseInfoList[totalCount + j]._flags.Reset();

            if (hasTicketList[j])
            {
                outELicenseInfoList[totalCount + j]._flags.Set(ELicenseStatusFlag_HasTicket::Index);
            }
        }

        totalCount += count;
    }

    *pOutCount = totalCount;

    NN_RESULT_SUCCESS;
}

void ETicketServiceImpl::SetupTicketDB() NN_NOEXCEPT
{
    NN_ES_SERVICE_LOG_TRACE("ETicketServiceImpl::SetupTicketDB()\n");

    m_CertificateDatabase.Initialize();
    m_CommonTicketDatabase.Initialize();
    m_PersonalizedTicketDatabase.Initialize();
    m_TicketMetaRecordDatabase.Initialize();
    m_ELicenseArchiveStore.Initialize();

    m_CertificateDatabase.SetVersion(CertificateDbVersion);
    m_CommonTicketDatabase.SetVersion(CommonTicketDbVersion);
    m_PersonalizedTicketDatabase.SetVersion(PersonalizedTicketDbVersion);
    m_TicketMetaRecordDatabase.SetVersion(TicketMetaRecordDbVersion);
    m_ELicenseArchiveStore.SetVersion(ELicenseArchiveStoreVersion);

    NN_ABORT_UNLESS_RESULT_SUCCESS(fs::CommitSaveData(CertificateDbMountName));
    NN_ABORT_UNLESS_RESULT_SUCCESS(fs::CommitSaveData(CommonTicketDbMountName));
    NN_ABORT_UNLESS_RESULT_SUCCESS(fs::CommitSaveData(PersonalizedTicketDbMountName));
    NN_ABORT_UNLESS_RESULT_SUCCESS(fs::CommitSaveData(TicketMetaRecordDbMountName));
    NN_ABORT_UNLESS_RESULT_SUCCESS(fs::CommitSaveData(ELicenseArchiveStoreMountName));
}

void ETicketServiceImpl::SetupDeviceCertificateAndDeviceKey() NN_NOEXCEPT
{
    NN_ES_SERVICE_LOG_TRACE("ETicketServiceImpl::SetupDeviceCertificateAndDeviceKey()\n");

    nn::escore::ETicketService signer;

#ifdef NN_BUILD_CONFIG_OS_WIN
    Result result;

    const char* MountName = "sdcard";
    const char* DeviceCertificateName = "sdcard:/device_cert_for_device_register.cert";
    const char* DevicePrivateKeyName = "sdcard:/device_cert_for_device_register.priv";

    // sdcard フォルダのマウント
    result = nn::fs::MountSdCardForDebug(MountName);
    if (result.IsSuccess())
    {
        NN_UTIL_SCOPE_EXIT{ nn::fs::Unmount(MountName); };

        // 仮想デバイス証明書の読込みと設定
        {
            nn::fs::FileHandle deviceCertificateFileHandle;
            result = nn::fs::OpenFile(&deviceCertificateFileHandle, DeviceCertificateName, nn::fs::OpenMode_Read);
            if (result.IsSuccess())
            {
                NN_UTIL_SCOPE_EXIT{ nn::fs::CloseFile(deviceCertificateFileHandle); };

                int64_t deviceCertificateSize;
                NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::GetFileSize(&deviceCertificateSize, deviceCertificateFileHandle));

                std::unique_ptr<char[]> deviceCertificate(new char[static_cast<unsigned int>(deviceCertificateSize)]);
                NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::ReadFile(deviceCertificateFileHandle, 0, deviceCertificate.get(), static_cast<size_t>(deviceCertificateSize)));

                signer.SetDeviceCert(deviceCertificate.get(), static_cast<int>(deviceCertificateSize));
            }
        }

        // 仮想デバイス秘密鍵の読込みと設定
        {
            nn::fs::FileHandle devicePrivateKeyFileHandle;
            result = nn::fs::OpenFile(&devicePrivateKeyFileHandle, DevicePrivateKeyName, nn::fs::OpenMode_Read);
            if (result.IsSuccess())
            {
                NN_UTIL_SCOPE_EXIT{ nn::fs::CloseFile(devicePrivateKeyFileHandle); };

                int64_t devicePrivateKeySize;
                NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::GetFileSize(&devicePrivateKeySize, devicePrivateKeyFileHandle));

                // 最初の2バイトがパッディングのため、除く
                int64_t devicePrivateKeyOffset = 2;

                std::unique_ptr<char[]> devicePrivateKey(new char[static_cast<unsigned int>(devicePrivateKeySize)]);
                NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::ReadFile(devicePrivateKeyFileHandle, devicePrivateKeyOffset, devicePrivateKey.get(), static_cast<size_t>(devicePrivateKeySize - devicePrivateKeyOffset)));

                signer.SetDevicePrivateKey(devicePrivateKey.get(), static_cast<int>(devicePrivateKeySize - devicePrivateKeyOffset));
            }
        }
    }

#else
    nn::escore::ESError esError;

    // デバイス証明書の取得
    nn::settings::factory::EccB233DeviceCertificate deviceCertificate;
    nn::settings::factory::GetEciDeviceCertificate(&deviceCertificate);

    esError = signer.SetDeviceCert(deviceCertificate.data, 384);
    NN_ABORT_UNLESS(esError == ES_ERR_OK, "An invalid device certificate was established.");

    // デバイス秘密鍵の取得
    nn::settings::factory::EccB233DeviceKey encryptedDevicePrivateKey;
    nn::settings::factory::GetEciDeviceKey(&encryptedDevicePrivateKey);

    char devicePrivateKey[32];
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::spl::ExtractDrmDeviceCertEccKey(devicePrivateKey, sizeof(devicePrivateKey), encryptedDevicePrivateKey.data, encryptedDevicePrivateKey.size));

    // 最初の2バイトがパッディングのため、除く
    int64_t devicePrivateKeyOffset = 2;

    esError = signer.SetDevicePrivateKey(devicePrivateKey + devicePrivateKeyOffset, sizeof(devicePrivateKey) - devicePrivateKeyOffset);
    NN_ABORT_UNLESS(esError == ES_ERR_OK, "An invalid device private key was established.");

    // Eticket デバイス鍵を設定
    nn::settings::factory::Rsa2048DeviceKey rsa2048DeviceKey;
    Result result = nn::settings::factory::GetEticketDeviceKey(&rsa2048DeviceKey);

    if (result.IsSuccess())
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::spl::LoadEsDeviceKey(rsa2048DeviceKey.data, rsa2048DeviceKey.size));
    }

#endif // NN_BUILD_CONFIG_OS_WIN
}

}} // namespace nn::es
