﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <new>
#include <type_traits>
#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/sf/sf_HipcServer.h>
#include <nn/es/es_ETicketServiceHipcServer.h>
#include "es_CreateETicketService.h"
#include "es_ServiceName.h"

namespace nn { namespace es {

namespace {

// nn::sf::HipcSimpleAllInOneServerManager を継承したクラスを作成します。通常は内部を空として構いません。
// nn::sf::HipcSimpleAllInOneServerManager は HIPC のサーバのポートとセッションを一元管理するクラステンプレートです。
// クラステンプレートの第 1 引数は管理する最大セッション数、第 2 引数は管理する最大ポート数です。
class MyServerManager : public nn::sf::HipcSimpleAllInOneServerManager<8, 1>
{
};

// 繰り返しのサーバの起動と終了が可能となるように placement new で初期化を行います。
// 繰り返しの起動と終了が必要ない場合には MyServerManager は直接配置しても構いません。
std::aligned_storage<sizeof(MyServerManager), NN_ALIGNOF(MyServerManager)>::type g_MyServerManagerStorage;
MyServerManager* g_pMyServerManager;

}

// サーバの初期化処理を行います。
// サーバプロセスを開始する際に必ず呼び出す必要があります。
void InitializeETicketServer() NN_NOEXCEPT
{
    NN_SDK_ASSERT(!g_pMyServerManager);

    // MyServerManager オブジェクトのコンストラクト
    g_pMyServerManager = new (&g_MyServerManagerStorage) MyServerManager;

    // CreateETicketServiceByDfc() で得られるオブジェクトを、サービス名で関連付けて、システムに登録します。
    // これによって、ここで指定したサービス名に対して nn::sf::CreateHipcProxyByName() を呼ぶことで、
    // 指定したオブジェクトへの HIPC プロキシ参照を得ることができます。
    // sessionCountMax は、現時点では十分に大きな値を指定しておけば良いです。
    auto sessionCountMax = 30;
    auto result = g_pMyServerManager->RegisterObjectForPort(CreateETicketServiceByDfc(), sessionCountMax, ETicketServiceName);
    NN_ABORT_UNLESS(result.IsSuccess());

    // サーバマネージャを開始します。
    // ただし、実際のサーバ動作は、LoopAuto 関数等を呼び出すことで行う必要があります。
    g_pMyServerManager->Start();
}

// サーバの処理ループを開始します。
// 1 つの処理ループで扱えるリクエストは同時に 1 つのみです。
// 複数リクエストを同時に扱いたい場合は、処理ループをプロセス内で複数用意してください。
void LoopETicketServer() NN_NOEXCEPT
{
    NN_SDK_ASSERT(g_pMyServerManager);

    // サーバの処理ループ実行を実行するには、先ほど定義した MyServerManager クラスから LoopAuto() 関数にアクセスします。
    // - ポートへの接続要求
    // - セッションへの処理要求
    g_pMyServerManager->LoopAuto();
}

}}  // namespace nn::es
