﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/es/es_Configuration.h>
#include <nn/es/es_ServiceLog.h>
#include <nn/sf/sf_Types.h>
#include <nn/result/result_HandlingUtility.h>
#include "es_ELicenseArchiveAdaptor.h"
#include "es_ELicenseManager.h"
#include "json/es_RapidJson.h"

namespace nn { namespace es {

ELicenseImportContext::Challenge ELicenseChallengeManager::GenerateChallenge(ELicenseOwnerId id) NN_NOEXCEPT
{
    // チャレンジを生成してから有効な秒数
    const int ValidSecond = 600;

#if defined (NN_BUILD_CONFIG_OS_HORIZON)
    NN_ABORT_UNLESS_RESULT_SUCCESS(spl::GenerateRandomBytes(&m_Challenge.value, sizeof(m_Challenge.value)));
#else
    os::GenerateRandomBytes(&m_Challenge.value, sizeof(m_Challenge.value));
#endif

    m_OwnerId = id;
    m_ExpireDate = os::GetSystemTick() + os::ConvertToTick(TimeSpan::FromSeconds(ValidSecond));

    return m_Challenge;
}

Result ELicenseChallengeManager::VerifyChallenge(ELicenseOwnerId ownerId, ELicenseImportContext::Challenge challenge) const NN_NOEXCEPT
{
    // チャレンジが一致しているか確認
    NN_RESULT_THROW_UNLESS(m_Challenge == challenge && m_OwnerId == ownerId, ResultELicenseArchiveChallengeIncorrect());

    // チャレンジが有効な時間内か確認
    NN_RESULT_THROW_UNLESS(IsStillValid(), ResultELicenseArchiveChallengeExpired());

    NN_RESULT_SUCCESS;
}

bool  ELicenseChallengeManager::IsStillValid() const NN_NOEXCEPT
{
    return os::GetSystemTick() <= m_ExpireDate;
}

nn::Result ELicenseManager::BeginImportELicenseArchive(ELicenseImportContext* outContext, ELicenseOwnerId ownerId) NN_NOEXCEPT
{
    ELicenseImportContext context = {};
    ELicenseImportContextAccessor accessor(&context);

    accessor.SetELicenseOwnerId(ownerId);

    // チャレンジの生成と設定
    accessor.SetChallenge(m_ChallengeManager.GenerateChallenge(ownerId));

    *outContext = context;

    NN_RESULT_SUCCESS;
}

nn::Result ELicenseManager::ImportELicenseArchive(const ELicenseImportContext& context, const void* eLicenseArchiveBuffer, size_t eLicenseArchiveSize) NN_NOEXCEPT
{
    // eLicense アーカイブを保存する
    NN_RESULT_DO(m_pStore->Import(eLicenseArchiveBuffer, eLicenseArchiveSize, ELicenseImportContextAccessor::GetELicenseOwnerId(context)));

    NN_RESULT_SUCCESS;
}

nn::Result ELicenseManager::EndImportELicenseArchive(ELicenseArchiveId* outELicenseArchiveId, const ELicenseImportContext& context) NN_NOEXCEPT
{
    // eLicense アーカイブの保存を完了させる
    NN_RESULT_DO(m_pStore->EndImporting(ELicenseImportContextAccessor::GetELicenseOwnerId(context)));
    bool readAfterReboot = false;
    bool isDebug = false;
    NN_RESULT_DO(ReadELicenseArchive(outELicenseArchiveId, ELicenseImportContextAccessor::GetELicenseOwnerId(context), readAfterReboot, isDebug));

    NN_RESULT_SUCCESS;
}

nn::Result ELicenseManager::EndImportELicenseArchiveForDebug(ELicenseArchiveId* outELicenseArchiveId, const ELicenseImportContext& context) NN_NOEXCEPT
{
    // eLicense アーカイブの保存を完了させる
    NN_RESULT_DO(m_pStore->EndImporting(ELicenseImportContextAccessor::GetELicenseOwnerId(context)));
    bool readAfterReboot = false;
    bool isDebug = true;
    NN_RESULT_DO(ReadELicenseArchive(outELicenseArchiveId, ELicenseImportContextAccessor::GetELicenseOwnerId(context), readAfterReboot, isDebug));

    // デバッグ用の場合は eLicense アーカイブを削除しておく
    m_pStore->Delete(ELicenseImportContextAccessor::GetELicenseOwnerId(context));

    NN_RESULT_SUCCESS;
}

nn::Result ELicenseManager::GetELicenseArchiveSizeForDebug(uint64_t* outSize, nn::es::ELicenseOwnerId ownerId) NN_NOEXCEPT
{
    Path archivePath;
    bool isArchiveFound = m_pStore->GetArchivePathForParse(&archivePath, ownerId);
    NN_RESULT_THROW_UNLESS(isArchiveFound, ResultELicenseArchiveNotFound());

    fs::FileHandle handle;
    NN_RESULT_DO(fs::OpenFile(&handle, archivePath.Get(), fs::OpenMode_Read));
    NN_UTIL_SCOPE_EXIT{ fs::CloseFile(handle); };

    {
        // スタックに置けないのでヒープに確保する
        std::unique_ptr<ELicenseArchiveFileReader> archiveReader(new(ELicenseArchiveFileReader));
        NN_RESULT_DO(archiveReader->Initailize(handle));
        *outSize = static_cast<uint64_t>(archiveReader->GetSize());
    }

    NN_RESULT_SUCCESS;
}

nn::Result ELicenseManager::GetELicenseArchiveDataForDebug(uint64_t* outSize, void* outELicenseArchiveBuffer, size_t eLicenseArchiveBufferSize, nn::es::ELicenseOwnerId ownerId) NN_NOEXCEPT
{
    Path archivePath;
    bool isArchiveFound = m_pStore->GetArchivePathForParse(&archivePath, ownerId);
    NN_RESULT_THROW_UNLESS(isArchiveFound, ResultELicenseArchiveNotFound());

    fs::FileHandle handle;
    NN_RESULT_DO(fs::OpenFile(&handle, archivePath.Get(), fs::OpenMode_Read));
    NN_UTIL_SCOPE_EXIT{ fs::CloseFile(handle); };

    {
        // スタックに置けないのでヒープに確保する
        std::unique_ptr<ELicenseArchiveFileReader> archiveReader(new(ELicenseArchiveFileReader));
        NN_RESULT_DO(archiveReader->Initailize(handle));

        int64_t size = archiveReader->GetSize();
        NN_RESULT_THROW_UNLESS(static_cast<size_t>(size) <= eLicenseArchiveBufferSize, ResultBufferNotEnough());

        NN_RESULT_DO(archiveReader->Read(0, outELicenseArchiveBuffer, static_cast<size_t>(size)));

        *outSize = static_cast<uint64_t>(size);
    }

    NN_RESULT_SUCCESS;
}

nn::Result ELicenseManager::DeleteAllELicenseArchiveForDebug() NN_NOEXCEPT
{
    // eLicense アーカイブを全削除する
    m_pStore->DeleteAll();

    // 有効な eLicense を全削除する
    m_pList->DeleteAll();

    // eLicense 配下であるフラグを降ろす
    m_pList->DisableELicense();

    NN_RESULT_SUCCESS;
}

nn::Result ELicenseManager::ReadAllSavedELicenseArchive() NN_NOEXCEPT
{
    ELicenseOwnerId ownerIdList[account::UserCountMax];
    int count = m_pStore->List(ownerIdList, account::UserCountMax);

    for (const auto& ownerId : util::MakeSpan(ownerIdList, count))
    {
        ELicenseArchiveId tmpELicenseArchiveId;
        bool readAfterReboot = true;
        bool isDebug = false;
        NN_RESULT_TRY(ReadELicenseArchive(&tmpELicenseArchiveId, ownerId, readAfterReboot, isDebug))
            NN_RESULT_CATCH_ALL
            {
                NN_ES_SERVICE_LOG_INFO("failed to read eLicense archive. naId=%016llx, result=0x%08llx", ownerId, NN_RESULT_CURRENT_RESULT.GetInnerValueForDebug());
                m_pStore->Delete(ownerId);
            }
        NN_RESULT_END_TRY
    }

    NN_RESULT_SUCCESS;
}

nn::Result ELicenseManager::ReadELicenseArchive(ELicenseArchiveId* outELicenseArchiveId, ELicenseOwnerId ownerId, bool readAfterReboot, bool isDebug) NN_NOEXCEPT
{
    // インポートする NA の eLicense を現在のリストから削除する
    m_pList->Delete(ownerId);

    Path archivePath;
    bool isArchiveFound = m_pStore->GetArchivePathForParse(&archivePath, ownerId);
    NN_RESULT_THROW_UNLESS(isArchiveFound, ResultELicenseArchiveNotFound());

    ELicenseArchiveAdaptor adaptor(ownerId, m_pList, readAfterReboot);

    // 再起動後の読み込みの場合またはデバッグ用の eLicense のチャレンジ検証の場合以外はチャレンジの検証を行う
    bool verifyChallenge = !(readAfterReboot || IsELicenseChallengeVerificationForDebugEnabled());
    if (isDebug)
    {
        m_ELicenseArchiveFileStreamForDebug.SetStringBuffer(m_StringBuffer, StringBufferSize);
        NN_RESULT_DO(m_ELicenseArchiveFileStreamForDebug.Initialize(archivePath.Get()));
        NN_UTIL_SCOPE_EXIT
        {
            m_ELicenseArchiveFileStreamForDebug.Finalize();
        };

        NN_RESULT_TRY(ParseELicenseArchive(outELicenseArchiveId, adaptor, m_ELicenseArchiveFileStreamForDebug, ownerId, verifyChallenge))
            NN_RESULT_CATCH_ALL
            {
                // パースに失敗、またはパースした要素に不整合があった場合は途中までインポートした eLicense を取り消す
                m_pList->Revert();
                NN_RESULT_RETHROW;
            }
        NN_RESULT_END_TRY
    }
    else
    {
        m_ELicenseArchiveFileStream.SetStringBuffer(m_StringBuffer, StringBufferSize);
        NN_RESULT_DO(m_ELicenseArchiveFileStream.Initialize(archivePath.Get()));
        NN_UTIL_SCOPE_EXIT
        {
            m_ELicenseArchiveFileStream.Finalize();
        };

        NN_RESULT_TRY(ParseELicenseArchive(outELicenseArchiveId, adaptor, m_ELicenseArchiveFileStream, ownerId, verifyChallenge))
            NN_RESULT_CATCH_ALL
            {
                // パースに失敗、またはパースした要素に不整合があった場合は途中までインポートした eLicense を取り消す
                m_pList->Revert();
                NN_RESULT_RETHROW;
            }
        NN_RESULT_END_TRY
    }

    // eLicense 配下であるフラグを立てる
    m_pList->EnableELicense();

    // eLicense を有効化する
    m_pList->Commit(*outELicenseArchiveId);

    NN_RESULT_SUCCESS;
}

template <typename StreamType>
Result ELicenseManager::ParseELicenseArchive(ELicenseArchiveId* outELicenseArchiveId, ELicenseArchiveAdaptor& adaptor, StreamType& stream, ELicenseOwnerId ownerId, bool verifyChallenge) NN_NOEXCEPT
{
    NN_RESULT_DO(ImportJsonByRapidJson(adaptor, stream));
    NN_RESULT_DO(adaptor.GetLastResult());

    // Challenge 要素が存在しているか確認
    if (!adaptor.GetChallenge())
    {
        NN_RESULT_THROW(ResultELicenseArchiveChallengeNotFound());
    }

    // Challenge の検証
    if (verifyChallenge)
    {
        ELicenseImportContext::Challenge challenge = *adaptor.GetChallenge();
        NN_RESULT_DO(m_ChallengeManager.VerifyChallenge(ownerId, challenge));
    }

    // ELicenseOwnerNaId 要素が存在しているか確認
    if (!adaptor.GetELicenseOwnerId())
    {
        NN_RESULT_THROW(ResultELicenseArchiveELicenseOwnerNaIdNotFound());;
    }

    // 入力された ownerId と ELicenseOwnerNaId 要素が一致しているか確認
    if (ownerId != *adaptor.GetELicenseOwnerId())
    {
        NN_RESULT_THROW(ResultELicenseArchiveELicenseOwnerNaIdIncorrect());
    }

    // ELicenseArchiveId 要素が存在しているか確認
    if (!adaptor.GetELicenseArchiveId())
    {
        NN_RESULT_THROW(ResultELicenseArchiveELicenseArchiveIdNotFound());
    }

    *outELicenseArchiveId = *adaptor.GetELicenseArchiveId();

    NN_RESULT_SUCCESS;
}

}} // namespace nn::es
