﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/es/es_TypesForInner.h>
#include <nn/util/util_StringUtil.h>
#include "es_TicketDatabase.h"

namespace nn { namespace es {

class ELicenseArchiveStore : public ETicketServiceDatabase
{
public:
    explicit ELicenseArchiveStore(const char* eLicenseArchiveStorePath) NN_NOEXCEPT : ETicketServiceDatabase(eLicenseArchiveStorePath)
    {
    }

    void Initialize() NN_NOEXCEPT;

    /**
    *   @brief  指定したユーザの eLicense アーカイブを分割インポートします。
    */
    Result Import(const void* archive, size_t size, ELicenseOwnerId ownerId) NN_NOEXCEPT;

    /**
    *   @brief  指定したユーザの eLicense アーカイブのインポート完了します。
    */
    Result EndImporting(ELicenseOwnerId ownerId) NN_NOEXCEPT;

    /**
    *   @brief  指定したユーザの eLicense アーカイブを削除します。
    *           分割インポート中とインポートが完了した eLicense アーカイブのどちらも削除します。
    */
    void Delete(ELicenseOwnerId ownerId) NN_NOEXCEPT;

    /**
    *   @brief  全ての eLicense アーカイブを削除します。
    *           分割インポート中とインポートが完了した eLicense アーカイブのどちらも削除します。
    */
    void DeleteAll() NN_NOEXCEPT;

    /**
    *   @brief  インポートが完了した eLicense アーカイブの ELicenseOwnerId のリストを取得します。
    *   @return ELicenseOwnerId のリストを取得した数
    */
    int List(ELicenseOwnerId* outELicenseOwnerIdList, int listCount) NN_NOEXCEPT;

    /**
    *   @brief  指定したユーザの eLicense アーカイブのファイルパスを取得します。
    *   @return インポートが完了した eLicense アーカイブが見つかったかどうか
    */
    bool GetArchivePathForParse(Path* outPath, ELicenseOwnerId ownerId) NN_NOEXCEPT;

private:
    enum class ArchiveState : uint8_t
    {
        Constructing = 0x1 << 0,    //!< 分割インポート中
        Saved = 0x1 << 1,           //!< インポートが完了
    };

    /**
    *   以下のファイル構成でアーカイブを配置する
    *   eLicenseArchiveStorePath
    *   - construsting (分割インポート中のアーカイブ)
    *     - (ownerId ごとのアーカイブ)
    *   - saved (インポートが完了したアーカイブ)
    *     - (ownerId ごとのアーカイブ)
    */

    void DeleteImpl(ArchiveState state, ELicenseOwnerId ownerId) NN_NOEXCEPT;
    void DeleteAllImpl(bool onlyConstructing) NN_NOEXCEPT;

    Path GetArchiveDirectoryPath(ArchiveState state) NN_NOEXCEPT
    {
        switch (state)
        {
        case ArchiveState::Constructing:
            return GetDatabasePath().Append("/constructing");
        case ArchiveState::Saved:
            return GetDatabasePath().Append("/saved");
        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }

    Path GetArchiveFilePath(ArchiveState state, ELicenseOwnerId ownerId) NN_NOEXCEPT
    {
        return GetArchiveDirectoryPath(state).AppendFormat("/%016llx", ownerId);
    }
};

}} // namespace nn::es
