﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <array>
#include <nn/fs.h>
#include <nn/crypto/crypto_Sha256Generator.h>
#include <nn/spl/spl_Api.h>
#include <nn/util/util_Optional.h>

namespace nn { namespace es {

struct Sha256HashValue
{
    uint8_t value[crypto::Sha256Generator::HashSize];
};

enum class ELicenseArchivePayloadDataFormatType : uint8_t
{
    Json        = 0,                //!< json
    Msgpack     = 1,                //!< msgpack
};

enum class ELicenseArchiveSignatureType : uint32_t
{
    Rsa4096     = 0x00010000,       //!< RSA 4096 bit signature
    Rsa2048     = 0x00010001,       //!< RSA 2048 bit signature
    Ecc         = 0x00010002,       //!< ECC signature 512 bits
    Rsa4096H256 = 0x00010003,       //!< RSA 4096 bit sig using SHA-256
    Rsa2048H256 = 0x00010004,       //!< RSA 2048 bit sig using SHA-256
    EccH256     = 0x00010005,       //!< ECC sig 512 bits using SHA-256
    HmacSha1    = 0x00010006,       //!< HMAC-SHA1 160 bit signature
};

enum class ELicenseArchiveKeyType : uint8_t
{
    Aes128 = 0,                     //!< AES-128
    RsaOaep2048 = 1,                //!< RSA-OAEP 2048 (メッセージダイジェスト=SHA-256、 マスク生成関数=MGF1)
    Ecc256 = 2,                     //!< ECC 256
};

struct ELicenseArchiveSignature
{
    ELicenseArchiveSignatureType            signatureType;                          //<! 署名タイプ
    uint8_t                                 signature[256];                         //<! 署名データ
    uint8_t                                 padding[60];
};
NN_STATIC_ASSERT(sizeof(ELicenseArchiveSignature) == 320);

struct ELicenseArchiveHeader
{
    uint8_t                                 issuer[64];                             //<! 署名した公開鍵に対応する発行者名
    uint8_t                                 key[256];                               //<! payload を暗号化した鍵を暗号化した鍵
    uint8_t                                 formatVersion;                          //<! フォーマットバージョン
    ELicenseArchiveKeyType                  keyType;                                //<! 鍵の暗号化タイプ
    ELicenseArchivePayloadDataFormatType    payloadDataFormat;                      //<! payload のデータフォーマット
    uint8_t                                 commonKeyGeneration;                    //<! 鍵を暗号化したコモン鍵の鍵世代
    uint8_t                                 keyInitialVector[16];                   //<! 鍵の初期化ベクトル
    uint32_t                                hashTableOffset;                        //<! ハッシュテーブルの先頭のオフセット値(固定値:0x300)
    uint32_t                                numberOfPayloadBlockEntries;            //<! payload block のエントリ数
    uint32_t                                payloadBlockOffset;                     //<! payload block の先頭のオフセット値
    uint32_t                                payloadBlockUnitSize;                   //<! payload block 一つのブロックサイズ(固定値:16KByte)
    uint32_t                                payloadTotalSize;                       //<! payload のデータのサイズ
    Sha256HashValue                         hashOfHashTable;                        //<! ハッシュテーブルのハッシュ値
    uint8_t                                 reserved[56];
};
NN_STATIC_ASSERT(sizeof(ELicenseArchiveHeader) == 448);

class ELicenseArchiveFileReader
{
public:
    Result Initailize(fs::FileHandle handle) NN_NOEXCEPT;

    int64_t GetSize() const NN_NOEXCEPT;
    Result Read(int64_t offset, void* buffer, size_t size) NN_NOEXCEPT;

    ELicenseArchivePayloadDataFormatType GetDataType() NN_NOEXCEPT;

private:
    static const int BlockUnitSize = 16 * 1024;
    static const std::array<uint32_t, 1> SupportedFormatVersionList;
    static const std::array<uint8_t, 1> SupportedKeyGenerationList;

    Result VerifySign() NN_NOEXCEPT;
    Result VerifyHash() NN_NOEXCEPT;
    Result ReadFile(int64_t offset, void* buffer, size_t size) NN_NOEXCEPT;
    Result ReadBlock(int blockOffset) NN_NOEXCEPT;
    Result GetAccessKey(spl::AccessKey* outKey) NN_NOEXCEPT;

    ELicenseArchiveHeader m_Header{};

    struct ELicenseArchiveHashTable
    {
        static const int MaxHashTableCount = 512;

        Sha256HashValue hashes[MaxHashTableCount]{};
    }m_HashTable{};

    struct BlockReadBuffer
    {
        uint8_t encrypted[BlockUnitSize];
        uint8_t decrypted[BlockUnitSize];
        util::optional<int> currentCacheBlockOffset;
    }m_BlockReadBuffer;

    util::optional<spl::AccessKey> m_AccessKey;

    int m_EntryCount = 0;

    util::optional<fs::FileHandle> m_Handle;
    int64_t m_FileSize = 0;
};
}} // namespace nn::es
