﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <nn/es/es_Result.h>
#include <nn/es/es_ServiceLog.h>
#include <nn/crypto/crypto_Aes128CtrDecryptor.h>
#include <nn/crypto/crypto_Sha256Generator.h>
#include <nn/crypto/crypto_RsaPkcs1Sha256Verifier.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/spl/spl_Api.h>
#include <nn/util/util_ScopedTransaction.h>
#include "es_DevicePublicKey.h"
#include "es_ELicenseArchiveReader.h"

namespace nn { namespace es {

namespace
{
    class VerifyKey
    {
    public:
        const uint8_t* GetModulus() const NN_NOEXCEPT
        {
            return Modulus;
        }

        size_t GetModulusSize() const NN_NOEXCEPT
        {
            return sizeof(Modulus);
        }

        const uint8_t* GetPublicExponent() const NN_NOEXCEPT
        {
            return PublicExponent;
        }

        size_t GetPublicExponentSize() const NN_NOEXCEPT
        {
            return sizeof(PublicExponent);
        }

    private:
        static const uint8_t Modulus[256];
        static const uint8_t PublicExponent[4];
    };

    const uint8_t VerifyKey::Modulus[] =
    {
        0xc9, 0xfd, 0xd4, 0x52, 0x39, 0x21, 0x8c, 0x8e, 0x45, 0xaf, 0xd2, 0x17, 0xb3, 0x72, 0x52,
        0x72, 0x59, 0xce, 0x77, 0x63, 0x59, 0x6b, 0x5a, 0xd4, 0xc1, 0xc9, 0xc2, 0xf9, 0xf5, 0x81,
        0xc1, 0xd5, 0x18, 0x20, 0x7f, 0x81, 0xd0, 0x37, 0xf5, 0x3d, 0x58, 0x89, 0xb0, 0x70, 0x3f,
        0xf3, 0x35, 0x2b, 0x16, 0x9e, 0x9f, 0xaf, 0x66, 0xd0, 0x4d, 0xa0, 0xf7, 0x7c, 0x26, 0xc0,
        0xa6, 0x70, 0xe5, 0xce, 0xee, 0x60, 0x13, 0x95, 0x0c, 0x0a, 0x6b, 0x35, 0xe6, 0xf5, 0x1e,
        0x13, 0xb1, 0xe8, 0x27, 0xab, 0xe6, 0x1b, 0x38, 0x7a, 0x28, 0xab, 0x2a, 0xef, 0xee, 0xd6,
        0x3f, 0xa9, 0xcb, 0x86, 0x4c, 0xe7, 0xbf, 0x3f, 0x71, 0x85, 0xfa, 0x5c, 0x1e, 0xc4, 0x8c,
        0xe1, 0x75, 0x3f, 0x6a, 0x6d, 0xa4, 0x66, 0x71, 0x91, 0x19, 0x44, 0x2f, 0xac, 0x6c, 0x35,
        0x35, 0x29, 0xcc, 0x61, 0x30, 0x51, 0xd5, 0x34, 0x87, 0xa9, 0x91, 0xe6, 0x9a, 0xb3, 0x64,
        0x7d, 0x91, 0x98, 0xe0, 0xfe, 0x6a, 0xaf, 0x7b, 0xc9, 0xd7, 0xd1, 0x50, 0xb2, 0x08, 0xad,
        0x3a, 0xc8, 0xb8, 0x3c, 0x9c, 0xce, 0xd7, 0x3f, 0x4a, 0x3b, 0x7c, 0xa0, 0xd5, 0xd9, 0xea,
        0x74, 0x58, 0x58, 0x5e, 0x4d, 0x75, 0x39, 0xdf, 0x56, 0xff, 0x86, 0xbb, 0x08, 0xdf, 0x48,
        0x58, 0x4d, 0x7b, 0x9b, 0x85, 0xc5, 0xb7, 0x46, 0x8e, 0x01, 0x9a, 0x15, 0x2f, 0x7c, 0x01,
        0xfd, 0xc5, 0x49, 0xdd, 0xfd, 0x6f, 0xbb, 0x0f, 0x0b, 0x5f, 0x29, 0xdf, 0x02, 0xcf, 0x3d,
        0x6a, 0xa0, 0x3e, 0xa7, 0xd3, 0x3a, 0x26, 0x24, 0x2a, 0xce, 0x7c, 0x28, 0xc6, 0x36, 0x2b,
        0x6b, 0xea, 0x9f, 0x56, 0x95, 0x2f, 0x73, 0x54, 0x87, 0x72, 0xe6, 0x7a, 0xe0, 0x58, 0x2d,
        0xdd, 0x1c, 0x12, 0x6c, 0xda, 0xa6, 0x66, 0x72, 0x16, 0x15, 0xc4, 0xf5, 0x54, 0xba, 0xea,
        0x33
    };

    const uint8_t VerifyKey::PublicExponent[] =
    {
        0x00, 0x01, 0x00, 0x01
    };

    void AddCounter(void* pCounter, size_t counterSize, uint64_t additionalValue) NN_NOEXCEPT
    {
        uint64_t restAdditionalValue = additionalValue;
        auto pCounterU8 = static_cast<uint8_t*>(pCounter);
        uint8_t carry = 0;

        for (int i = 0; i < static_cast<int>(counterSize); ++i)
        {
            auto sum = pCounterU8[counterSize - i - 1] + (restAdditionalValue & 0xFF) + carry;
            carry = static_cast<uint8_t>(sum >> 8);
            auto sumU8 = static_cast<uint8_t>(sum & 0xFF);

            pCounterU8[counterSize - i - 1] = sumU8;

            restAdditionalValue >>= 8;
            if (carry == 0 && restAdditionalValue == 0)
            {
                return;
            }
        }
    }
}

const std::array<uint32_t, 1> ELicenseArchiveFileReader::SupportedFormatVersionList =
{
    0
};

const std::array<uint8_t, 1> ELicenseArchiveFileReader::SupportedKeyGenerationList =
{
    5
};

Result ELicenseArchiveFileReader::Initailize(fs::FileHandle handle) NN_NOEXCEPT
{
    m_BlockReadBuffer.currentCacheBlockOffset = util::nullopt;
    m_AccessKey = util::nullopt;

    // 全て処理が完了する前に失敗した場合、状態を巻き戻す
    util::ScopedTransaction transaction;

    m_Handle = handle;
    NN_UTIL_RESERVE_SCOPED_ROLLBACK(transaction)
    {
        m_Handle = util::nullopt;
    };

    // ファイルサイズを取得
    NN_RESULT_DO(fs::GetFileSize(&m_FileSize, *m_Handle));

    // まず、ヘッダ領域を読み込む
    NN_RESULT_DO(ReadFile(sizeof(ELicenseArchiveSignature), &m_Header, sizeof(m_Header)));

    // ヘッダの署名検証を行う
    // TORIAEZU: サーバの対応が済むまで署名検証を無効化する
#if 0
    NN_RESULT_DO(VerifySign());
#endif

    // ヘッダの値が期待通りか確認する
    // サポートできるバージョンかを確認する
    NN_RESULT_THROW_UNLESS(std::any_of(SupportedFormatVersionList.begin(), SupportedFormatVersionList.end(),
        [&](uint32_t version)
        {
            return m_Header.formatVersion == version;
        }
    ), ResultELicenseArchiveFormatVersionNotSupported());

    // サポートできる鍵世代か確認する
    NN_RESULT_THROW_UNLESS(std::any_of(SupportedKeyGenerationList.begin(), SupportedKeyGenerationList.end(),
        [&](uint8_t keyGeneration)
        {
            return m_Header.commonKeyGeneration == keyGeneration;
        }
    ), ResultELicenseArchiveKeyGenerationNotSupported());

    // payload ブロックのエントリ数がサポートできるエントリ数であるか確認する
    NN_RESULT_THROW_UNLESS(m_Header.numberOfPayloadBlockEntries <= m_HashTable.MaxHashTableCount, ResultELicenseArchiveELicenseBlockEntriesTooMany());
    m_EntryCount = static_cast<int>(m_Header.numberOfPayloadBlockEntries);

    // ハッシュテーブルの先頭へのオフセットの値が正しいか確認する
    NN_RESULT_THROW_UNLESS(m_Header.hashTableOffset == sizeof(ELicenseArchiveSignature) + sizeof(ELicenseArchiveHeader), ResultELicenseArchiveHashTableOffsetUnexpected());

    // payload ブロックの先頭へのオフセットの値が正しいかを確認する
    NN_RESULT_THROW_UNLESS(m_Header.payloadBlockOffset == m_Header.hashTableOffset + sizeof(Sha256HashValue) * m_EntryCount, ResultELicenseArchiveELicenseBlockOffsetUnexpected());

    // payload ブロックのユニットサイズが正しいかを確認する
    NN_RESULT_THROW_UNLESS(m_Header.payloadBlockUnitSize == BlockUnitSize, ResultELicenseArchiveBlockUnitSizeUnexpected());

    // 次に、ハッシュテーブルを読み込む
    size_t hashTableSize = static_cast<size_t>(m_Header.payloadBlockOffset - m_Header.hashTableOffset);
    NN_RESULT_DO(ReadFile(m_Header.hashTableOffset, &m_HashTable, hashTableSize));

    // ハッシュテーブルのハッシュ検証を行う
    NN_RESULT_DO(VerifyHash());

    transaction.Commit();

    NN_RESULT_SUCCESS;
}

int64_t ELicenseArchiveFileReader::GetSize() const NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_Handle);
    return m_Header.payloadTotalSize;
}

Result ELicenseArchiveFileReader::Read(int64_t offset, void* buffer, size_t size) NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_Handle);

    NN_RESULT_THROW_UNLESS(offset >= 0, ResultELicenseArchivePayloadOutOfRangeAccess());
    NN_RESULT_THROW_UNLESS(size <= m_Header.payloadTotalSize, ResultELicenseArchivePayloadOutOfRangeAccess());
    NN_RESULT_THROW_UNLESS(offset <= static_cast<int64_t>(m_Header.payloadTotalSize - size), ResultELicenseArchivePayloadOutOfRangeAccess());

    size_t copiedSize = 0;

    while (size - copiedSize > 0)
    {
        int64_t currentOffset = offset + copiedSize;

        // 読み込む必要がある offset が何番目のブロックかを計算する
        int readBlockOffset = static_cast<int>(currentOffset / BlockUnitSize);

        // 必要があるブロックがバッファにない場合は読み込む
        if (!(m_BlockReadBuffer.currentCacheBlockOffset == readBlockOffset))
        {
            NN_RESULT_DO(ReadBlock(readBlockOffset));
        }

        // バッファから必要な領域をコピーする
        size_t begin = currentOffset % BlockUnitSize;
        size_t end = std::min(begin + (size - copiedSize), static_cast<size_t>(BlockUnitSize));
        size_t copySize = end - begin;
        memcpy(static_cast<uint8_t*>(buffer) + copiedSize, &m_BlockReadBuffer.decrypted[begin], copySize);

        copiedSize += copySize;
    }

    NN_RESULT_SUCCESS;
}

ELicenseArchivePayloadDataFormatType ELicenseArchiveFileReader::GetDataType() NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_Handle);
    return m_Header.payloadDataFormat;
}

Result ELicenseArchiveFileReader::VerifySign() NN_NOEXCEPT
{
    ELicenseArchiveSignature signature;
    NN_RESULT_DO(ReadFile(0, &signature, sizeof(signature)));

    // サポートできる署名タイプかを確認する
    NN_RESULT_THROW_UNLESS(signature.signatureType == ELicenseArchiveSignatureType::Rsa2048H256, ResultELicenseArchiveSignatureTypeNotSupported());

    // 署名検証を行う
    // TORIAEZU: issuer はチェックせずに常に固定の検証鍵で署名検証をする
    // TODO: issuer に応じた検証鍵を使う
    VerifyKey key;
    bool isVerifySucceeded = crypto::VerifyRsa2048Pkcs1Sha256(
        signature.signature, sizeof(signature.signature),
        key.GetModulus(), key.GetModulusSize(),
        key.GetPublicExponent(), key.GetPublicExponentSize(),
        &m_Header, sizeof(m_Header),
        m_BlockReadBuffer.encrypted, sizeof(m_BlockReadBuffer.encrypted)    // TORIAEZU: リードバッファを署名検証のワークバッファとして使う
    );
    NN_RESULT_THROW_UNLESS(isVerifySucceeded, ResultELicenseArchiveSignatureVerificationFailed());

    NN_RESULT_SUCCESS;
}

Result ELicenseArchiveFileReader::VerifyHash() NN_NOEXCEPT
{
    // ハッシュテーブル全体のハッシュ値を計算する(SHA256)
    Sha256HashValue hash;
    size_t hashTableSize = static_cast<size_t>(m_Header.payloadBlockOffset - m_Header.hashTableOffset);
    crypto::GenerateSha256Hash(&hash, sizeof(hash), &m_HashTable, hashTableSize);

    // ヘッダに書かれているのハッシュ値と一致しているか検証する
    NN_RESULT_THROW_UNLESS(memcmp(&hash, &m_Header.hashOfHashTable, sizeof(hash)) == 0, ResultELicenseArchiveHashTableVerificationFailed());

    NN_RESULT_SUCCESS;
}

Result ELicenseArchiveFileReader::ReadFile(int64_t offset, void* buffer, size_t size) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(static_cast<int64_t>(offset + size) <= m_FileSize, ResultELicenseArchiveIncomplete());

    NN_RESULT_DO(fs::ReadFile(*m_Handle, offset, buffer, size));

    NN_RESULT_SUCCESS;
}

Result ELicenseArchiveFileReader::ReadBlock(int blockOffset) NN_NOEXCEPT
{
    NN_SDK_ASSERT(blockOffset < m_EntryCount);

    m_BlockReadBuffer.currentCacheBlockOffset = util::nullopt;
    int64_t readOffset = m_Header.payloadBlockOffset + BlockUnitSize * blockOffset;

    // リードバッファにブロックを読み込む
    NN_RESULT_DO(ReadFile(readOffset, m_BlockReadBuffer.encrypted, BlockUnitSize));

    // ブロックのハッシュ値を計算する(SHA256)
    Sha256HashValue hash;
    crypto::GenerateSha256Hash(&hash, sizeof(hash), m_BlockReadBuffer.encrypted, sizeof(m_BlockReadBuffer.encrypted));

    // ハッシュテーブルに書かれているハッシュ値と一致しているか検証する
    NN_RESULT_THROW_UNLESS(memcmp(&hash, &m_HashTable.hashes[blockOffset], sizeof(hash)) == 0, ResultELicenseArchiveELicenseBlockVerificationFailed());

    // spl を使った復号処理
#ifdef NN_BUILD_CONFIG_OS_HORIZON
    // AccessKey を取得していない場合は読み込む
    if (!m_AccessKey)
    {
        spl::AccessKey key;
        NN_RESULT_DO(GetAccessKey(&key));
        m_AccessKey = key;
    }

    // 鍵スロットを割り当てる
    int slotIndex;
    NN_RESULT_DO(spl::AllocateAesKeySlot(&slotIndex));
    NN_UTIL_SCOPE_EXIT{ NN_ABORT_UNLESS_RESULT_SUCCESS(spl::DeallocateAesKeySlot(slotIndex)); };

    // 鍵スロットにアクセスキーを割り当てる
    NN_ABORT_UNLESS_RESULT_SUCCESS(spl::LoadPreparedAesKey(slotIndex, *m_AccessKey));

    // 復号時に渡す初期カウンターをインクリメントする
    uint8_t currentCounter[sizeof(ELicenseArchiveHeader::keyInitialVector)];
    memcpy(currentCounter, m_Header.keyInitialVector, sizeof(currentCounter));
    AddCounter(currentCounter, sizeof(currentCounter), BlockUnitSize * blockOffset / crypto::Aes128CtrDecryptor::BlockSize);

    // 復号を行う
    NN_RESULT_DO(spl::ComputeCtr(
        m_BlockReadBuffer.decrypted, sizeof(m_BlockReadBuffer.decrypted),
        slotIndex,
        m_BlockReadBuffer.encrypted, sizeof(m_BlockReadBuffer.encrypted),
        currentCounter, sizeof(currentCounter)
    ));

    // 復号した内容をログ出力
    auto size = (blockOffset == m_EntryCount - 1) ? (m_Header.payloadTotalSize % BlockUnitSize) : BlockUnitSize;
    NN_UNUSED(size);
    NN_ES_SERVICE_LOG_INFO("Decrypted(block: %d) %.*s\n", blockOffset, size, m_BlockReadBuffer.decrypted);
#endif

    m_BlockReadBuffer.currentCacheBlockOffset = blockOffset;

    NN_RESULT_SUCCESS;
}

Result ELicenseArchiveFileReader::GetAccessKey(spl::AccessKey* outKey) NN_NOEXCEPT
{
#ifdef NN_BUILD_CONFIG_OS_HORIZON
    // 0バイトに SHA256 をかけたバイト列を使う
    const uint8_t LabelDigest[32] =
    {
        0xE3, 0xB0, 0xC4, 0x42, 0x98, 0xFC, 0x1C, 0x14, 0x9A, 0xFB, 0xF4, 0xC8, 0x99, 0x6F, 0xB9, 0x24,
        0x27, 0xAE, 0x41, 0xE4, 0x64, 0x9B, 0x93, 0x4C, 0xA4, 0x95, 0x99, 0x1B, 0x78, 0x52, 0xB8, 0x55,
    };

    Rsa2048DevicePublicKey publicKey;

    // ペイロードを復号するアクセスキーを取得する
    NN_RESULT_DO(spl::PrepareEsArchiveKey(
        outKey,
        &m_Header.key, sizeof(m_Header.key),
        publicKey.GetModulus(), publicKey.GetModulusSize(),
        LabelDigest, sizeof(LabelDigest),
        m_Header.commonKeyGeneration
    ));

    NN_RESULT_SUCCESS;
#else
    NN_UNUSED(outKey);
    NN_RESULT_SUCCESS;
#endif
}

}} // namespace nn::es
