﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/es/es_ELicenseApi.h>
#include <nn/es/es_ELicenseApiForSystem.h>
#include <nn/es/es_Result.h>
#include "es_ELicenseImportContext.h"
#include "es_ETicketServiceObject.h"

namespace nn { namespace es {

ELicenseImportContext::Challenge ELicenseImportContext::GetChallenge() NN_NOEXCEPT
{
    return ELicenseImportContext::Challenge{ ELicenseImportContextAccessor::GetChallenge(*this) };
}

Result BeginImportELicenseArchive(ELicenseImportContext* outContext, account::NintendoAccountId naId) NN_NOEXCEPT
{
    return GetETicketServiceObject()->BeginImportELicenseArchive(outContext, GetELicenseOwnerId(naId));
}

Result ImportELicenseArchive(const ELicenseImportContext& context, void* eLicenseArchive, size_t eLicenseArchiveSize) NN_NOEXCEPT
{
    sf::InBuffer pInELicenseBuffer(reinterpret_cast<const char*>(eLicenseArchive), eLicenseArchiveSize);
    return GetETicketServiceObject()->ImportELicenseArchive(context, pInELicenseBuffer);
}

Result EndImportELicenseArchive(ELicenseArchiveId* outELicenseArchiveId, const ELicenseImportContext& context) NN_NOEXCEPT
{
    return GetETicketServiceObject()->EndImportELicenseArchive(outELicenseArchiveId, context);
}

Result EndImportELicenseArchiveForDebug(ELicenseArchiveId* outELicenseArchiveId, const ELicenseImportContext& context) NN_NOEXCEPT
{
    nn::Result result = GetETicketServiceObject()->EndImportELicenseArchiveForDebug(outELicenseArchiveId, context);

    if (ResultDevelopmentFunctionCalled::Includes(result))
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    }

    return result;
}

Result GetELicenseArchiveSizeForDebug(size_t* pOutSize, account::NintendoAccountId naId) NN_NOEXCEPT
{
    uint64_t outSize;

    Result result = GetETicketServiceObject()->GetELicenseArchiveSizeForDebug(&outSize, GetELicenseOwnerId(naId));

    if (ResultDevelopmentFunctionCalled::Includes(result))
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    }

    NN_RESULT_DO(result);

    *pOutSize = static_cast<size_t>(outSize);
    NN_RESULT_SUCCESS;
}

Result GetELicenseArchiveDataForDebug(size_t* pOutSize, void* pOutBuffer, size_t size, account::NintendoAccountId naId) NN_NOEXCEPT
{
    uint64_t outSize;

    sf::OutBuffer outBuffer(reinterpret_cast<char*>(pOutBuffer), size);

    Result result = GetETicketServiceObject()->GetELicenseArchiveDataForDebug(&outSize, outBuffer, GetELicenseOwnerId(naId));

    if (ResultDevelopmentFunctionCalled::Includes(result))
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    }

    NN_RESULT_DO(result);

    *pOutSize = static_cast<size_t>(outSize);
    NN_RESULT_SUCCESS;
}

int CountELicense() NN_NOEXCEPT
{
    int32_t outCount;

    NN_ABORT_UNLESS_RESULT_SUCCESS(GetETicketServiceObject()->CountELicense(&outCount));

    return outCount;
}

int ListELicenseIds(ELicenseId outELicenseIdList[], int count) NN_NOEXCEPT
{
    int32_t outCount;
    sf::OutArray<ELicenseId> pOutELicenseIdList(outELicenseIdList, static_cast<size_t>(count));

    NN_ABORT_UNLESS_RESULT_SUCCESS(GetETicketServiceObject()->ListELicenseIds(&outCount, pOutELicenseIdList));

    return outCount;
}

int ListELicenseInfo(ELicenseInfoWrapper outELicenseInfoList[], int eLicenseInfoCount, const ELicenseId eLicenseIdList[], int eLicenseIdCount) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_GREATER_EQUAL(eLicenseInfoCount, eLicenseIdCount);

    int32_t outCount;
    sf::OutArray<ELicenseInfoWrapper> pOutELicenseInfoList(outELicenseInfoList, static_cast<size_t>(eLicenseInfoCount));
    sf::InArray<ELicenseId> pInELicenseIdList(eLicenseIdList, static_cast<size_t>(eLicenseIdCount));

    NN_ABORT_UNLESS_RESULT_SUCCESS(GetETicketServiceObject()->ListELicenseInfo(&outCount, pOutELicenseInfoList, pInELicenseIdList));

    return outCount;
}

int ListELicenseInfoForDebug(ELicenseInfoForSystemWrapper outELicenseInfoList[], int eLicenseInfoCount, const ELicenseId eLicenseIdList[], int eLicenseIdCount) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_GREATER_EQUAL(eLicenseInfoCount, eLicenseIdCount);

    int32_t outCount;
    sf::OutArray<ELicenseInfoForSystemWrapper> pOutELicenseInfoList(outELicenseInfoList, static_cast<size_t>(eLicenseInfoCount));
    sf::InArray<ELicenseId> pInELicenseIdList(eLicenseIdList, static_cast<size_t>(eLicenseIdCount));

    NN_ABORT_UNLESS_RESULT_SUCCESS(GetETicketServiceObject()->ListELicenseInfoForDebug(&outCount, pOutELicenseInfoList, pInELicenseIdList));

    return outCount;
}

void DeleteAllELicenseArchiveForDebug() NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(GetETicketServiceObject()->DeleteAllELicenseArchiveForDebug());
}

}} // namespace nn::es
