﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Abort.h>
#include <nn/nn_SdkAssert.h>
#include <nn/es/es_Api.h>
#include <nn/result/result_HandlingUtility.h>
#include "es_ETicketServiceObject.h"

namespace nn { namespace es {

Result ImportTicket(const void* ticket, size_t ticketSize, const void* certificate, size_t certificateSize) NN_NOEXCEPT
{
    sf::InBuffer pInTicketBuffer(reinterpret_cast<const char*>(ticket), ticketSize);
    sf::InBuffer pInCertificateBuffer(reinterpret_cast<const char*>(certificate), certificateSize);

    return GetETicketServiceObject()->ImportTicket(pInTicketBuffer, pInCertificateBuffer);
}

Result ImportTicketCertificateSet(const void* ticketCertificateSet, size_t ticketCertificateSetSize) NN_NOEXCEPT
{
    sf::InBuffer pInTicketCertificateSetBuffer(reinterpret_cast<const char*>(ticketCertificateSet), ticketCertificateSetSize);

    return GetETicketServiceObject()->ImportTicketCertificateSet(pInTicketCertificateSetBuffer);
}

void DeleteTicket(const es::RightsIdIncludingKeyId rightsIdList[], int count) NN_NOEXCEPT
{
    sf::InArray<RightsIdIncludingKeyId> pInRightsIdList(rightsIdList, static_cast<size_t>(count));

    NN_ABORT_UNLESS_RESULT_SUCCESS(GetETicketServiceObject()->DeleteTicket(pInRightsIdList));
}

void DeletePersonalizedTicket(es::AccountId accountId) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(GetETicketServiceObject()->DeletePersonalizedTicket(accountId));
}

void DeleteAllCommonTicket() NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(GetETicketServiceObject()->DeleteAllCommonTicket());
}

void DeleteAllPersonalizedTicket() NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(GetETicketServiceObject()->DeleteAllPersonalizedTicket());
}

void DeleteAllPersonalizedTicket(const es::TicketId ticketIdExclusionList[], int count) NN_NOEXCEPT
{
    sf::InArray<TicketId> pInTicketIdExclusionList(ticketIdExclusionList, static_cast<size_t>(count));

    NN_ABORT_UNLESS_RESULT_SUCCESS(GetETicketServiceObject()->DeleteAllPersonalizedTicketExcludingList(pInTicketIdExclusionList));
}

int CountCommonTicket() NN_NOEXCEPT
{
    int32_t outCount;

    NN_ABORT_UNLESS_RESULT_SUCCESS(GetETicketServiceObject()->CountCommonTicket(&outCount));

    return outCount;
}

int CountPersonalizedTicket() NN_NOEXCEPT
{
    int32_t outCount;

    NN_ABORT_UNLESS_RESULT_SUCCESS(GetETicketServiceObject()->CountPersonalizedTicket(&outCount));

    return outCount;
}

int ListCommonTicketRightsIds(es::RightsIdIncludingKeyId outRightsIdList[], int count) NN_NOEXCEPT
{
    int32_t outCount;
    sf::OutArray<RightsIdIncludingKeyId> pOutRightsIdList(outRightsIdList, static_cast<size_t>(count));

    NN_ABORT_UNLESS_RESULT_SUCCESS(GetETicketServiceObject()->ListCommonTicket(&outCount, pOutRightsIdList));

    return outCount;
}

int ListPersonalizedTicketRightsIds(es::RightsIdIncludingKeyId outRightsIdList[], int count) NN_NOEXCEPT
{
    int32_t outCount;
    sf::OutArray<RightsIdIncludingKeyId> pOutRightsIdList(outRightsIdList, static_cast<size_t>(count));

    NN_ABORT_UNLESS_RESULT_SUCCESS(GetETicketServiceObject()->ListPersonalizedTicket(&outCount, pOutRightsIdList));

    return outCount;
}

int ListMissingPersonalizedTicket(es::TicketId outTicketIdList[], int outTicketIdListCount, const es::TicketId ticketIdList[], int ticketIdListCount) NN_NOEXCEPT
{
    int32_t outCount;
    sf::OutArray<es::TicketId> pOutTicketIdList(outTicketIdList, static_cast<size_t>(outTicketIdListCount));
    sf::InArray<es::TicketId> pInTicketIdList(ticketIdList, static_cast<size_t>(ticketIdListCount));

    NN_ABORT_UNLESS_RESULT_SUCCESS(GetETicketServiceObject()->ListMissingPersonalizedTicket(&outCount, pOutTicketIdList, pInTicketIdList));

    return outCount;
}

Result GetCommonTicketSize(size_t* pOutSize, es::RightsIdIncludingKeyId rightsId) NN_NOEXCEPT
{
    uint64_t outSize;

    NN_RESULT_DO(GetETicketServiceObject()->GetCommonTicketSize(&outSize, rightsId));

    *pOutSize = static_cast<size_t>(outSize);
    NN_RESULT_SUCCESS;
}

Result GetCommonTicketData(size_t* pOutSize, void* pOutBuffer, size_t size, es::RightsIdIncludingKeyId rightsId) NN_NOEXCEPT
{
    uint64_t outSize;

    sf::OutBuffer outBuffer(reinterpret_cast<char*>(pOutBuffer), size);

    NN_RESULT_DO(GetETicketServiceObject()->GetCommonTicketData(&outSize, outBuffer, rightsId));

    *pOutSize = static_cast<size_t>(outSize);
    NN_RESULT_SUCCESS;
}

void OwnTicket(bool outOwnTicketList[], const es::RightsIdIncludingKeyId rightsIdList[], int count) NN_NOEXCEPT
{
    sf::OutArray<bool> pOutOwnTicketList(outOwnTicketList, static_cast<size_t>(count));
    sf::InArray<es::RightsIdIncludingKeyId> pInRightsIdList(rightsIdList, static_cast<size_t>(count));

    NN_ABORT_UNLESS_RESULT_SUCCESS(GetETicketServiceObject()->OwnTicket(pOutOwnTicketList, pInRightsIdList));
}

int GetTicketInfo(es::TicketInfo outTicketInfoList[], int ticketInfoListCount, const es::RightsIdIncludingKeyId rightsIdList[], int rightsIdListCount) NN_NOEXCEPT
{
    int32_t outCount;
    sf::OutArray<TicketInfo> pOutTicketInfoList(outTicketInfoList, static_cast<size_t>(ticketInfoListCount));
    sf::InArray<RightsIdIncludingKeyId> pInRightsIdList(rightsIdList, static_cast<size_t>(rightsIdListCount));

    NN_ABORT_UNLESS_RESULT_SUCCESS(GetETicketServiceObject()->GetTicketInfo(&outCount, pOutTicketInfoList, pInRightsIdList));

    return outCount;
}

int ListLightTicketInfo(nn::es::LightTicketInfo outTicketInfoList[], int ticketInfoListCount, nn::es::RightsIdIncludingKeyId rightsId) NN_NOEXCEPT
{
    int32_t outCount;
    sf::OutArray<LightTicketInfo> pOutTicketInfoList(outTicketInfoList, static_cast<size_t>(ticketInfoListCount));

    NN_ABORT_UNLESS_RESULT_SUCCESS(GetETicketServiceObject()->ListLightTicketInfo(&outCount, pOutTicketInfoList, rightsId));

    return outCount;
}

void SignData(es::Sign* pOutSign, es::Certificate* pOutCertificate, const void* pData, size_t size) NN_NOEXCEPT
{
    sf::InBuffer pInData(reinterpret_cast<const char*>(pData), size);

    NN_ABORT_UNLESS_RESULT_SUCCESS(GetETicketServiceObject()->SignData(pOutSign, pOutCertificate, pInData));
}

Result GetCommonTicketAndCertificateSize(size_t* pOutTicketSize, size_t* pOutCertificateSize, nn::es::RightsIdIncludingKeyId rightsId) NN_NOEXCEPT
{
    uint64_t outTicketSize;
    uint64_t outCertificateSize;
    NN_RESULT_DO(GetETicketServiceObject()->GetCommonTicketAndCertificateSize(&outTicketSize, &outCertificateSize, rightsId));

    *pOutTicketSize = static_cast<size_t>(outTicketSize);
    *pOutCertificateSize = static_cast<size_t>(outCertificateSize);
    NN_RESULT_SUCCESS;
}

Result GetCommonTicketAndCertificateData(size_t* pOutTicketSize, size_t* pOutCertificateSize, void* pOutTicketBuffer, void* pOutCertificateBuffer, size_t ticketBufferSize, size_t certificateBufferSize, nn::es::RightsIdIncludingKeyId rightsId) NN_NOEXCEPT
{
    sf::OutBuffer outTicketBuffer(reinterpret_cast<char*>(pOutTicketBuffer), ticketBufferSize);
    sf::OutBuffer outCertificateBuffer(reinterpret_cast<char*>(pOutCertificateBuffer), certificateBufferSize);

    uint64_t outTicketSize;
    uint64_t outCertificateSize;
    NN_RESULT_DO(GetETicketServiceObject()->GetCommonTicketAndCertificateData(&outTicketSize, &outCertificateSize, outTicketBuffer, outCertificateBuffer, rightsId));

    *pOutTicketSize = static_cast<size_t>(outTicketSize);
    *pOutCertificateSize = static_cast<size_t>(outCertificateSize);
    NN_RESULT_SUCCESS;
}

Result ImportPrepurchaseRecord(const es::PrepurchaseRecord& record) NN_NOEXCEPT
{
    NN_RESULT_DO(GetETicketServiceObject()->ImportPrepurchaseRecord(record));

    NN_RESULT_SUCCESS;
}

void DeletePrepurchaseRecord(const es::PrepurchaseRecord& record) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(GetETicketServiceObject()->DeletePrepurchaseRecord(record));
}

void DeleteAllPrepurchaseRecord() NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(GetETicketServiceObject()->DeleteAllPrepurchaseRecord());
}

int CountPrepurchaseRecord() NN_NOEXCEPT
{
    int32_t outCount;

    NN_ABORT_UNLESS_RESULT_SUCCESS(GetETicketServiceObject()->CountPrepurchaseRecord(&outCount));

    return outCount;
}

int ListPrepurchaseRecordRightsIds(nn::es::RightsIdIncludingKeyId outRightsIdList[], int count) NN_NOEXCEPT
{
    int32_t outCount;
    sf::OutArray<RightsIdIncludingKeyId> pOutRightsIdList(outRightsIdList, static_cast<size_t>(count));

    NN_ABORT_UNLESS_RESULT_SUCCESS(GetETicketServiceObject()->ListPrepurchaseRecord(&outCount, pOutRightsIdList));

    return outCount;
}

int ListPrepurchaseRecordInfo(nn::es::PrepurchaseRecord outPrepurchaseRecordList[], int count, nn::es::RightsIdIncludingKeyId rightsId) NN_NOEXCEPT
{
    int32_t outCount;
    sf::OutArray<PrepurchaseRecord> pOutPrepurchaseRecordList(outPrepurchaseRecordList, static_cast<size_t>(count));

    NN_ABORT_UNLESS_RESULT_SUCCESS(GetETicketServiceObject()->ListPrepurchaseRecordInfo(&outCount, pOutPrepurchaseRecordList, rightsId));

    return outCount;
}

int CountTicket(nn::es::RightsId rightsId) NN_NOEXCEPT
{
    int outCount;

    NN_ABORT_UNLESS_RESULT_SUCCESS(GetETicketServiceObject()->CountDesignatedTicket(&outCount, rightsId));

    return outCount;
}

int ListTicketRightsIds(nn::es::RightsIdIncludingKeyId outRightsIdList[], int count, nn::es::RightsId rightsId) NN_NOEXCEPT
{
    int32_t outCount;
    sf::OutArray<RightsIdIncludingKeyId> pOutRightsIdList(outRightsIdList, static_cast<size_t>(count));

    NN_ABORT_UNLESS_RESULT_SUCCESS(GetETicketServiceObject()->ListDesignatedTicket(&outCount, pOutRightsIdList, rightsId));

    return outCount;
}

int CountPrepurchaseRecord(nn::es::RightsId rightsId) NN_NOEXCEPT
{
    int outCount;

    NN_ABORT_UNLESS_RESULT_SUCCESS(GetETicketServiceObject()->CountDesignatedPrepurchaseRecord(&outCount, rightsId));

    return outCount;
}

int ListPrepurchaseRecordRightsIds(nn::es::RightsIdIncludingKeyId outRightsIdList[], int count, nn::es::RightsId rightsId) NN_NOEXCEPT
{
    int32_t outCount;
    sf::OutArray<RightsIdIncludingKeyId> pOutRightsIdList(outRightsIdList, static_cast<size_t>(count));

    NN_ABORT_UNLESS_RESULT_SUCCESS(GetETicketServiceObject()->ListDesignatedPrepurchaseRecord(&outCount, pOutRightsIdList, rightsId));

    return outCount;
}

Result GetEncryptedTicketSize(size_t* pOutSize, nn::es::RightsIdIncludingKeyId rightsId) NN_NOEXCEPT
{
    uint64_t outSize;
    NN_RESULT_DO(GetETicketServiceObject()->GetEncryptedTicketSize(&outSize, rightsId));

    *pOutSize = static_cast<size_t>(outSize);
    NN_RESULT_SUCCESS;
}

Result GetEncryptedTicketData(TicketId* pOutTicketId, size_t* pOutSize, void* pOutBuffer, size_t outBufferSize, void* pOutKey, size_t outKeySize, nn::es::RightsIdIncludingKeyId rightsId) NN_NOEXCEPT
{
    uint64_t outSize;
    sf::OutBuffer outBuffer(reinterpret_cast<char*>(pOutBuffer), outBufferSize);
    sf::OutBuffer outKey(reinterpret_cast<char*>(pOutKey), outKeySize);

    NN_RESULT_DO(GetETicketServiceObject()->GetEncryptedTicketData(pOutTicketId, &outSize, outBuffer, outKey, rightsId));

    *pOutSize = static_cast<size_t>(outSize);
    NN_RESULT_SUCCESS;
}

}} // namespace nn::es
