﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cmath>
#include <nn/crypto/crypto_Aes128CbcEncryptor.h>
#include <nn/crypto/crypto_Aes128CtrEncryptor.h>
#include <nn/crypto/crypto_RsaOaepEncryptor.h>
#include <nn/crypto/crypto_RsaPkcs1Sha256Signer.h>
#include <nn/crypto/crypto_Sha256Generator.h>
#include <nn/es/debug/es_ELicenseArchiveBuilder.h>
#include <nn/os/os_Random.h>
#include <nn/result/result_HandlingUtility.h>

namespace nn { namespace es { namespace debug {

namespace
{
    // eLicense アーカイブ用コモンキー
    const uint8_t CommonKey[16] =
    {
        0x43, 0x9e, 0x3e, 0xb9, 0x19, 0xa2, 0xf5, 0xf5, 0xb1, 0xde, 0xa3, 0x4a, 0x21, 0xaf, 0xee, 0x05
    };

    const uint8_t CommonKeyGeneration = 5;
}

void ELicenseArchiveBuilder::SetPayloadAndEncryptRsa2048OaepSha256(const void* payload, size_t payloadSize, ELicenseArchivePayloadDataFormatType payloadDataType, const void* modulus, size_t modulusSize, const void* publicExponent, size_t publicExponentSize) NN_NOEXCEPT
{
    const int KeySize = 256;

    m_Data.header.payloadDataFormat = payloadDataType;

    m_Data.header.payloadTotalSize = static_cast<uint32_t>(payloadSize);
    m_Data.header.numberOfPayloadBlockEntries = static_cast<uint32_t>(((payloadSize + Data::Payload::UnitSize - 1) / Data::Payload::UnitSize));
    m_Data.header.payloadBlockOffset = m_Data.header.hashTableOffset + m_Data.header.numberOfPayloadBlockEntries * Data::HashTable::HashSize;
    m_Data.payload.blocks.reset(new Data::Payload::Block[m_Data.header.numberOfPayloadBlockEntries]);

    // eLicense データを暗号化する鍵を作成
    uint8_t rawKey[16];
    os::GenerateRandomBytes(rawKey, sizeof(rawKey));

    // eLicense データの暗号化に使う IV を作成
    os::GenerateRandomBytes(m_Data.header.keyInitialVector, sizeof(m_Data.header.keyInitialVector));

    // payload block を乱数埋めする
    os::GenerateRandomBytes(m_Data.payload.blocks.get(), Data::Payload::UnitSize * m_Data.header.numberOfPayloadBlockEntries);

    // eLicense データを AES-CTR で暗号化して、eLicense block に格納する
    size_t aesEncryptedSize = crypto::EncryptAes128Ctr(m_Data.payload.blocks.get(), payloadSize, rawKey, sizeof(rawKey), m_Data.header.keyInitialVector, sizeof(m_Data.header.keyInitialVector), payload, payloadSize);
    NN_ABORT_UNLESS_EQUAL(aesEncryptedSize, payloadSize);

    // 暗号化鍵をコモン鍵で AES-EBC 暗号化する
    // crypto に AES-EBC の API はないので、Iv = 0 で AES-CBC の暗号化を行う(鍵は16バイト=ブロック長なので Iv = 0 として暗号化すると AES-EBC と同じ結果となる)
    const uint8_t Iv[] =
    {
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    };
    uint8_t commonEncryptedKey[16];
    aesEncryptedSize = crypto::EncryptAes128Cbc(commonEncryptedKey, sizeof(commonEncryptedKey), CommonKey, sizeof(CommonKey), Iv, sizeof(Iv), rawKey, sizeof(rawKey));
    NN_ABORT_UNLESS_EQUAL(aesEncryptedSize, sizeof(commonEncryptedKey));
    m_Data.header.commonKeyGeneration = CommonKeyGeneration;

    // コモン鍵で暗号化された暗号化鍵を RSA で暗号化する
    uint8_t seed[Data::HashTable::HashSize];
    os::GenerateRandomBytes(seed, sizeof(seed));

    bool isRsaEncryptionSucceeded = crypto::RsaOaepEncryptor<KeySize, crypto::Sha256Generator>::Encrypt(
        m_Data.header.key, sizeof(m_Data.header.key),
        modulus, modulusSize,
        publicExponent, publicExponentSize,
        &commonEncryptedKey, sizeof(commonEncryptedKey),
        seed, sizeof(seed),
        "", 0
    );
    NN_ABORT_UNLESS(isRsaEncryptionSucceeded);
    m_Data.header.keyType = ELicenseArchiveKeyType::RsaOaep2048;

    // eLicense データのハッシュ値を Block ごとに計算して、ハッシュテーブルに代入する
    m_Data.hashTable.hashes.reset(new Sha256HashValue[m_Data.header.numberOfPayloadBlockEntries]);
    for (uint32_t i = 0; i < m_Data.header.numberOfPayloadBlockEntries; i++)
    {
        crypto::GenerateSha256Hash(m_Data.hashTable.hashes[i].value, Data::HashTable::HashSize, m_Data.payload.blocks[i].value, Data::Payload::UnitSize);
    }

    // ハッシュテーブルのハッシュ値を計算してヘッダに代入する
    crypto::GenerateSha256Hash(&m_Data.header.hashOfHashTable, Data::HashTable::HashSize, m_Data.hashTable.hashes.get(), m_Data.header.numberOfPayloadBlockEntries * Data::HashTable::HashSize);
}

void ELicenseArchiveBuilder::SignRsa2048Pkcs1Sha256(const void* modulus, size_t modulusSize, const void* privateExponent, size_t privateExponentSize) NN_NOEXCEPT
{
    // header を署名する
    bool isSignSucceeded = crypto::SignRsa2048Pkcs1Sha256(
        m_Data.signature.signature, sizeof(m_Data.signature.signature),
        modulus, modulusSize,
        privateExponent, privateExponentSize,
        &m_Data.header, sizeof(m_Data.header)
    );
    NN_ABORT_UNLESS(isSignSucceeded);
    m_Data.signature.signatureType = ELicenseArchiveSignatureType::Rsa2048H256;
}

size_t ELicenseArchiveBuilder::GetSize() const NN_NOEXCEPT
{
    return sizeof(m_Data.signature) + sizeof(m_Data.header) + (Data::HashTable::HashSize + Data::Payload::UnitSize) * m_Data.header.numberOfPayloadBlockEntries;
}

size_t ELicenseArchiveBuilder::Get(void* outBuffer, size_t bufferSize) const NN_NOEXCEPT
{
    if (GetSize() > bufferSize)
    {
        return 0;
    }

    auto pCurrent = static_cast<uint8_t*>(outBuffer);

    std::memcpy(pCurrent, &m_Data.signature, sizeof(m_Data.signature));
    pCurrent += sizeof(m_Data.signature);

    std::memcpy(pCurrent, &m_Data.header, sizeof(m_Data.header));
    pCurrent += sizeof(m_Data.header);

    std::memcpy(pCurrent, m_Data.hashTable.hashes.get(), Data::HashTable::HashSize * m_Data.header.numberOfPayloadBlockEntries);
    pCurrent += Data::HashTable::HashSize * m_Data.header.numberOfPayloadBlockEntries;

    std::memcpy(pCurrent, m_Data.payload.blocks.get(), Data::Payload::UnitSize * m_Data.header.numberOfPayloadBlockEntries);
    pCurrent += Data::Payload::UnitSize * m_Data.header.numberOfPayloadBlockEntries;

    return GetSize();
}
}}}
