﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/err/err_Api.h>
#include <nn/err/err_ApiForErrorViewer.h>
#include <nn/err/err_ErrorViewerAppletParam.h>
#include <nn/err/detail/err_ErrorCodeConvert.h>
#include <nn/err/detail/err_SystemData.h>
#include <nn/err/detail/err_Log.h>
#include <nn/ae.h>
#include <nn/erpt.h>
#include <nn/nn_SdkAssert.h>
#include <nn/kvdb/kvdb_Result.h>
#include <nn/ns/ns_Result.h>
#include <nn/ns/ns_ApplicationRecordSystemApi.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_StringUtil.h>
#include <nn/util/util_CharacterEncoding.h>
#include <algorithm>

#include "detail/err_ErptUtil.h"
#include "detail/err_ErptUtilForErrorViewer.h"
#include "detail/err_StringUtil.h"

namespace nn { namespace err {

namespace
{
    void RecordErrorDescription(erpt::Context& context, const char* dialogViewMessage, const char* fullScreenViewMessage) NN_NOEXCEPT
    {
        // 全画面用メッセージを優先して記録する。
        auto fullScreenMessageLength = (fullScreenViewMessage == nullptr) ? 0 : util::Strnlen(fullScreenViewMessage, ErrorViewerStartupParamMessageBufferSizeMax);
        if( fullScreenMessageLength > 0 )
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(context.Add(erpt::FieldId::ErrorDescription, fullScreenViewMessage, fullScreenMessageLength));
        }
        else
        {
            auto dialogMessageLength = (dialogViewMessage == nullptr) ? 0 : util::Strnlen(dialogViewMessage, ErrorViewerStartupParamMessageBufferSizeMax);
            if( dialogMessageLength > 0 )
            {
                NN_ABORT_UNLESS_RESULT_SUCCESS(context.Add(erpt::FieldId::ErrorDescription, dialogViewMessage, dialogMessageLength));
            }
            else
            {
                NN_SDK_ASSERT(false, "Neither 'fullScreenViewMessage' nor 'dialogViewMessage' has a valid message.\n");
            }
        }
    }
}

ErrorType GetErrorType(const void* startupParam) NN_NOEXCEPT
{
    return reinterpret_cast<const ErrorViewerStartupParamCommon*>(startupParam)->errorType;
}

nn::settings::LanguageCode GetMessageLanguageCode(const void* startupParam) NN_NOEXCEPT
{
    switch( GetErrorType(startupParam) )
    {
    case ErrorType::SystemError:
        return reinterpret_cast<const ErrorViewerStartupParamForSystemError*>(startupParam)->languageCode;
    case ErrorType::ApplicationError:
        return reinterpret_cast<const ErrorViewerStartupParamForApplicationError*>(startupParam)->languageCode;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

nn::settings::system::RegionCode GetEulaRegionCode(const void* startupParam) NN_NOEXCEPT
{
    switch( GetErrorType(startupParam) )
    {
    case ErrorType::Eula:
        return reinterpret_cast<const ErrorViewerStartupParamForEula*>(startupParam)->regionCode;
    case ErrorType::SystemUpdateEula:
        return reinterpret_cast<const ErrorViewerStartupParamForSystemUpdateEula*>(startupParam)->regionCode;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

bool GetEulaData(size_t* pOutActualSize, void* outBuffer, size_t outBufferSize, const void* eulaParamBuffer, settings::Language language) NN_NOEXCEPT
{
    auto pEulaData = reinterpret_cast<const EulaData*>(eulaParamBuffer);
    for( int i = 0; i < pEulaData->dataCount; i++ )
    {
        if( pEulaData->data[i].language == language )
        {
            *pOutActualSize = static_cast<size_t>(pEulaData->data[i].size);
            memcpy(outBuffer, pEulaData->data[i].body, std::min(outBufferSize, sizeof(pEulaData->data[i].body)));
            return true;
        }
    }
    return false;
}

nn::Result GetParentalControlResultRestricted(const void* startupParam) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(GetErrorType(startupParam) == ErrorType::ParentalControl);
    return reinterpret_cast<const ErrorViewerStartupParamForParentalControl*>(startupParam)->pctlResultRestricted;
}

bool IsErrorRecordDisplayRequested(const void* startupParam) NN_NOEXCEPT
{
    auto type = GetErrorType(startupParam);
    return (type == ErrorType::RecordedSystemData) || (type == ErrorType::RecordedSystemError) || (type == ErrorType::RecordedApplicationError);
}

nn::time::PosixTime GetTimeOfOccurrence(const void* startupParam) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsErrorRecordDisplayRequested(startupParam));
    switch( GetErrorType(startupParam) )
    {
    case ErrorType::RecordedSystemData:
        {
            return reinterpret_cast<const ErrorViewerStartupParamForRecordedSystemData*>(startupParam)->timeOfOccurrence;
        }
        break;

    case ErrorType::RecordedSystemError:
        {
            return reinterpret_cast<const ErrorViewerStartupParamForRecordedSystemError*>(startupParam)->timeOfOccurrence;
        }
        break;

    case ErrorType::RecordedApplicationError:
        {
            return reinterpret_cast<const ErrorViewerStartupParamForRecordedApplicationError*>(startupParam)->timeOfOccurrence;
        }
        break;

    default:
        NN_DETAIL_ERR_WARN("GetTimeOfOccurrence works only for recorded errors.\n");
        break;
    }
    return nn::time::PosixTime();
}

bool IsJumpEnabled(const void* startupParam) NN_NOEXCEPT
{
    return reinterpret_cast<const ErrorViewerStartupParamCommon*>(startupParam)->isJumpEnabled;
}

void SetErrorViewerResult(ErrorViewerJumpDestination destination) NN_NOEXCEPT
{
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
    ErrorViewerReturnValue returnValue;
    returnValue.version = 0;
    returnValue.destination = destination;

    nn::applet::StorageHandle handle;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::applet::CreateStorage(&handle, sizeof(ErrorViewerReturnValue)));
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::applet::WriteToStorage(handle, 0, &returnValue, sizeof(ErrorViewerReturnValue)));
    nn::ae::PushToOutChannel(handle);
#else
    NN_DETAIL_ERR_INFO("SetErrorViewerResult is not supported on this platform.\n");
    NN_UNUSED(destination);
#endif
}

namespace {

    void CreateErrorReportForSystemData(const void* startupParam, void* workBuffer, size_t workBufferSize) NN_NOEXCEPT
    {
        NN_UNUSED(workBufferSize);
        auto param = reinterpret_cast<const ErrorViewerStartupParamForSystemData*>(startupParam);
        // 先頭 2KiB を erpt::Context 作成用のバッファに使用する。
        const size_t WorkBufferSizeForContext = 2 * 1024;
        erpt::Context context(erpt::CategoryId::ErrorInfo, static_cast<uint8_t*>(workBuffer), static_cast<uint32_t>(WorkBufferSizeForContext));

        if( param->hasResultBacktrace )
        {
            // erpt::Context 作成用バッファの後ろの 1KiB を ResultBacktrace の追加に使用する。
            const size_t WorkBufferForAddResultBacktraceSize = 1024;
            NN_STATIC_ASSERT(WorkBufferForAddResultBacktraceSize < WorkBufferSizeForContext);
            auto workBufferForAddResultBacktrace = reinterpret_cast<Bit8*>(workBuffer) + WorkBufferSizeForContext;
            detail::AddResultBacktrace(&context, workBufferForAddResultBacktrace, WorkBufferForAddResultBacktraceSize);
        }
        if( param->hasErrorContext )
        {
            // erpt::Context 作成用バッファの後ろの 512byte(sizeof(ErrorContext)) を ErrorContext の追加に使用する。
            const size_t WorkBufferSizeForErrorContext = sizeof(ErrorContext);
            auto workBufferForErrorContext = reinterpret_cast<Bit8*>(workBuffer) + WorkBufferSizeForContext;
            detail::AddErrorContext(&context, workBufferForErrorContext, WorkBufferSizeForErrorContext);
        }

        char errorCodeString[ErrorCode::StringLengthMax];
        detail::MakeErrorCodeString(errorCodeString, ErrorCode::StringLengthMax, param->GetErrorCode());
        NN_ABORT_UNLESS_RESULT_SUCCESS(context.Add(erpt::FieldId::ErrorCode, errorCodeString, util::Strnlen(errorCodeString, ErrorCode::StringLengthMax)));
        NN_ABORT_UNLESS_RESULT_SUCCESS(context.CreateReport(erpt::ReportType_Visible));
    }

    void CreateErrorReportForSystemError(const void* startupParam, void* workBuffer, size_t workBufferSize) NN_NOEXCEPT
    {
        // 先頭 3KiB を erpt::Context 用のバッファに使用する（3KiB >= メッセージ最大長2KiB + ErrorContext 512 byte + ErrorCode 15 bytes）
        const size_t WorkBufferSizeForErptContext = 3 * 1024;
        NN_STATIC_ASSERT(WorkBufferSizeForErptContext >= sizeof(ErrorViewerStartupParamForSystemError::fullScreenMessage) + sizeof(ErrorContext) + ErrorCode::StringLengthMax);
        NN_SDK_ASSERT_GREATER_EQUAL(workBufferSize, WorkBufferSizeForErptContext + sizeof(ErrorContext));
        NN_UNUSED(workBufferSize);

        auto param = reinterpret_cast<const ErrorViewerStartupParamForSystemError*>(startupParam);
        auto context = erpt::Context(erpt::CategoryId::ErrorInfo, static_cast<uint8_t*>(workBuffer), static_cast<uint32_t>(WorkBufferSizeForErptContext));

        if( param->hasErrorContext )
        {
            // erpt::Context 用バッファの後ろの 512byte(sizeof(ErrorContext)) を ErrorContext の追加に使用する。
            const size_t WorkBufferSizeForErrorContext = sizeof(ErrorContext);
            auto workBufferForErrorContext = reinterpret_cast<Bit8*>(workBuffer) + WorkBufferSizeForErptContext;
            detail::AddErrorContext(&context, workBufferForErrorContext, WorkBufferSizeForErrorContext);
        }
        char errorCodeString[ErrorCode::StringLengthMax];
        detail::MakeErrorCodeString(errorCodeString, ErrorCode::StringLengthMax, param->errorCode);
        NN_ABORT_UNLESS_RESULT_SUCCESS(context.Add(erpt::FieldId::ErrorCode, errorCodeString, util::Strnlen(errorCodeString, ErrorCode::StringLengthMax)));
        RecordErrorDescription(context, param->dialogMessage, param->fullScreenMessage);
        NN_ABORT_UNLESS_RESULT_SUCCESS(context.CreateReport(erpt::ReportType_Visible));
    }

    void CreateErrorReportForApplicationError(const void* startupParam, void* workBuffer, size_t workBufferSize) NN_NOEXCEPT
    {
        ns::ApplicationErrorCodeCategory category;
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        auto pApplicationControlProperty = reinterpret_cast<ns::ApplicationControlProperty*>(workBuffer);
        NN_ABORT_UNLESS_RESULT_SUCCESS(ae::GetMainAppletApplicationControlProperty(pApplicationControlProperty));
        util::Strlcpy(category.value, pApplicationControlProperty->applicationErrorCodeCategory.value, sizeof(category.value));
#elif( NN_BUILD_CONFIG_OS_WIN )
        // テスト用に適当な固定値を入れる。
        util::Strlcpy(category.value, "DUMMY", sizeof(category.value));
#endif
        auto context = erpt::Context(erpt::CategoryId::ErrorInfo, static_cast<uint8_t*>(workBuffer), static_cast<uint32_t>(workBufferSize));
        auto param = reinterpret_cast<const ErrorViewerStartupParamForApplicationError*>(startupParam);
        char errorCodeString[ErrorCode::StringLengthMax];
        detail::MakeApplicationErrorCodeString(errorCodeString, ErrorCode::StringLengthMax, category, param->applicationErrorCodeNumber);
        NN_ABORT_UNLESS_RESULT_SUCCESS(context.Add(erpt::FieldId::ErrorCode, errorCodeString, util::Strnlen(errorCodeString, ErrorCode::StringLengthMax)));
        RecordErrorDescription(context, param->dialogMessage, param->fullScreenMessage);
        NN_ABORT_UNLESS_RESULT_SUCCESS(context.CreateReport(erpt::ReportType_Visible));
    }
}

ErrorCode ErrorViewerStartupParamForSystemData::GetErrorCode() const NN_NOEXCEPT
{
    if( isErrorCode )
    {
        return errorCode;
    }
    else
    {
        return detail::ConvertResultToErrorCode(result);
    }
}

void RecordError(const void* startupParam, void* workBuffer, size_t workBufferSize) NN_NOEXCEPT
{
    NN_STATIC_ASSERT(RequiredBufferSizeForRecordError >= sizeof(ns::ApplicationControlProperty));
    NN_SDK_REQUIRES_GREATER_EQUAL(workBufferSize, RequiredBufferSizeForRecordError);

    detail::SubmitFileSystemInfo();
    detail::SubmitApplicationInfo(workBuffer, workBufferSize);

    NN_UTIL_SCOPE_EXIT
    {
        detail::ClearFileSystemInfo();
        detail::ClearApplicationInfo();
    };

    switch( GetErrorType(startupParam) )
    {
    case ErrorType::SystemData:
        {

            CreateErrorReportForSystemData(startupParam, workBuffer, workBufferSize);
        }
        break;

    case ErrorType::SystemError:
        {
            CreateErrorReportForSystemError(startupParam, workBuffer, workBufferSize);
        }
        break;

    case ErrorType::ApplicationError:
        {
            CreateErrorReportForApplicationError(startupParam, workBuffer, workBufferSize);
        }
        break;

    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

void ProcessErrorInfoAttributes(const ErrorInfo& errorInfo) NN_NOEXCEPT
{
    detail::AttributeFlagSet attrFlagSet{ { errorInfo.GetAttributeFlagSetValue() } };
    if( attrFlagSet.Test<detail::AttributeFlag::UnacceptableApplicationVersion>() )
    {
        NN_DETAIL_ERR_TRACE("[ProcessErrorInfoAttributes] UnacceptableApplicationVersion flag is set.\n");
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        auto mainAppletIdentity = ae::GetMainAppletIdentityInfo();
        if( mainAppletIdentity.IsApplication() )
        {
            auto result = ns::RequestApplicationUpdate(mainAppletIdentity.applicationId, errorInfo.GetResult());
            if( nn::ns::ResultApplicationRecordNotFound::Includes(result) )
            {
                NN_DETAIL_ERR_TRACE("[ProcessErrorInfoAttributes] Failed to request application update. Application 0x%016x is not found.\n", mainAppletIdentity.applicationId);
            }
            else
            {
                NN_ABORT_UNLESS_RESULT_SUCCESS(result);
            }
        }
        else
        {
            NN_DETAIL_ERR_TRACE("[ProcessErrorInfoAttributes] Unable to request application update. Main caller is not an application.\n");
        }
#endif
    }
}

}}
