﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/err/err_ApiForApplet.h>
#include <nn/err/err_SystemApi.h>
#include <nn/err/err_SystemTypes.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_StringUtil.h>
#include "detail/err_ErptUtil.h"
#include "detail/err_ErptUtilForErrorViewer.h"
#include "detail/err_StringUtil.h"

namespace nn { namespace err {

namespace
{
    void RecordErrorImpl(ErrorCode errorCode, const char* message, erpt::ReportType reportType, void* workBuffer, size_t workBufferSize) NN_NOEXCEPT
    {
        NN_STATIC_ASSERT(RequiredBufferSizeForRecordError >= sizeof(ns::ApplicationControlProperty));
        NN_SDK_REQUIRES_GREATER_EQUAL(workBufferSize, RequiredBufferSizeForRecordError);

        detail::SubmitFileSystemInfo();
        detail::SubmitApplicationInfo(workBuffer, workBufferSize);

        NN_UTIL_SCOPE_EXIT
        {
            detail::ClearFileSystemInfo();
            detail::ClearApplicationInfo();
        };

        auto context = erpt::Context(erpt::CategoryId::ErrorInfo, static_cast<uint8_t*>(workBuffer), static_cast<uint32_t>(workBufferSize));
        char errorCodeString[ErrorCode::StringLengthMax];
        detail::MakeErrorCodeString(errorCodeString, ErrorCode::StringLengthMax, errorCode);
        NN_ABORT_UNLESS_RESULT_SUCCESS(context.Add(erpt::FieldId::ErrorCode, errorCodeString, util::Strnlen(errorCodeString, ErrorCode::StringLengthMax)));
        if( message != nullptr && strlen(message) > 0 )
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(context.Add(erpt::FieldId::ErrorDescription, message, static_cast<uint32_t>(strlen(message))));
        }
        NN_ABORT_UNLESS_RESULT_SUCCESS(context.CreateReport(reportType));
    }
}

void RecordError(ErrorCode errorCode, const char* message, void* workBuffer, size_t workBufferSize) NN_NOEXCEPT
{
    RecordErrorImpl(errorCode, message, erpt::ReportType_Visible, workBuffer, workBufferSize);
}

void RecordErrorForTelemetry(ErrorCode errorCode, const char* message, void* workBuffer, size_t workBufferSize) NN_NOEXCEPT
{
    RecordErrorImpl(errorCode, message, erpt::ReportType_Invisible, workBuffer, workBufferSize);
}

void RecordErrorForTelemetry(Result result, ncm::ApplicationId applicationId, const util::optional<ResultBacktrace>& resultBacktrace, void* workBuffer, size_t workBufferSize) NN_NOEXCEPT
{
    NN_UNUSED(workBufferSize);

    detail::SubmitFileSystemInfo();
    {
        // アプリケーションIDのみ記録する。
        auto context = erpt::Context(erpt::CategoryId::ApplicationInfo);
        char idStr[17];
        util::SNPrintf(idStr, 17, "%016llx", applicationId.value);
        NN_ABORT_UNLESS_RESULT_SUCCESS(context.Add(erpt::ApplicationID, idStr, static_cast<uint32_t>(util::Strnlen(idStr, 17))));
        NN_ABORT_UNLESS_RESULT_SUCCESS(context.SubmitContext());
    }

    NN_UTIL_SCOPE_EXIT
    {
        detail::ClearFileSystemInfo();
        detail::ClearApplicationInfo();
    };

    auto context = erpt::Context(erpt::CategoryId::ErrorInfo);
    char errorCodeString[ErrorCode::StringLengthMax];
    auto errorCode = ConvertResultToErrorCode(result);
    GetErrorCodeString(errorCodeString, sizeof(errorCodeString), errorCode);
    NN_ABORT_UNLESS_RESULT_SUCCESS(context.Add(erpt::FieldId::ErrorCode, errorCodeString, util::Strnlen(errorCodeString, ErrorCode::StringLengthMax)));

    if( resultBacktrace && resultBacktrace->count > 0 )
    {
        NN_SDK_REQUIRES_ALIGNED(workBuffer, sizeof(uint32_t));
        NN_SDK_REQUIRES_GREATER_EQUAL(workBufferSize, sizeof(uint32_t) * resultBacktrace->count);
        uint32_t* resultBacktraceArray = reinterpret_cast<uint32_t*>(workBuffer);
        for( int i = 0; i < resultBacktrace->count; i++ )
        {
            resultBacktraceArray[i] = resultBacktrace->results[i].GetInnerValueForDebug();
        }
        NN_ABORT_UNLESS_RESULT_SUCCESS(context.Add(erpt::FieldId::ResultBacktrace, resultBacktraceArray, static_cast<uint32_t>(resultBacktrace->count)));
    }
    NN_ABORT_UNLESS_RESULT_SUCCESS(context.CreateReport(erpt::ReportType_Invisible));
}

}}
