﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/err/err_Types.h>
#include <nn/nn_SdkAssert.h>
#include <nn/ns/ns_ApplicationManagerApi.h>
#include <nn/util/util_StringUtil.h>
#include <nn/util/util_TFormatString.h>
#include <cstdlib>
#include <cctype>
#include "err_StringUtil.h"

namespace nn { namespace err { namespace detail {

namespace
{
    bool AreDigits(const char* string) NN_NOEXCEPT
    {
        size_t length = strlen(string);
        for( size_t i = 0; i < length; i++ )
        {
            if( !std::isdigit(string[i]) )
            {
                return false;
            }
        }
        return true;
    }
}

void MakeErrorCodeString(char* outValue, size_t outValueSize, ErrorCode errorCode) NN_NOEXCEPT
{
    auto len = util::TSNPrintf(outValue, outValueSize, "%04d-%04d", errorCode.category, errorCode.number);
    NN_SDK_ASSERT_LESS(static_cast<size_t>(len), outValueSize);
    NN_UNUSED(len);
}

void MakeApplicationErrorCodeString(char* outValue, size_t outValueSize, const ns::ApplicationErrorCodeCategory& category, ApplicationErrorCodeNumber number) NN_NOEXCEPT
{
    auto len = util::TSNPrintf(outValue, outValueSize, "2-%s-%04d", category.value, number);
    NN_SDK_ASSERT_LESS(static_cast<size_t>(len), outValueSize);
    NN_UNUSED(len);
}

void ParseErrorCodeString(ErrorCode* outValue, const char* errorCodeStr) NN_NOEXCEPT
{
    NN_SDK_ASSERT_LESS(strlen(errorCodeStr), 15u);
    char categoryStr[15];
    auto pHyphen = strstr(errorCodeStr, "-");
    NN_SDK_ASSERT_NOT_NULL(pHyphen);
    util::Strlcpy(categoryStr, errorCodeStr, static_cast<int>(pHyphen - errorCodeStr + 1));
    outValue->category = static_cast<ErrorCodeCategory>(atoi(categoryStr));
    outValue->number = static_cast<ErrorCodeNumber>(atoi(pHyphen + 1));
}

void ParseApplicationErrorCodeString(ns::ApplicationErrorCodeCategory* outCategory, ApplicationErrorCodeNumber* outNumber, const char* applicationErrorCodeStr) NN_NOEXCEPT
{
    NN_SDK_ASSERT_LESS(strlen(applicationErrorCodeStr), 15u);
    auto pFirstHyphen = strstr(applicationErrorCodeStr, "-");
    NN_SDK_ASSERT_NOT_NULL(pFirstHyphen);
    auto pSecondHyphen = strstr(pFirstHyphen + 1, "-");
    NN_SDK_ASSERT_NOT_NULL(pSecondHyphen);
    util::Strlcpy(outCategory->value, pFirstHyphen + 1, static_cast<int>(pSecondHyphen - pFirstHyphen));
    *outNumber = static_cast<ApplicationErrorCodeNumber>(atoi(pSecondHyphen + 1));
}

bool IsApplicationErrorCodeString(const char* errorCodeStr) NN_NOEXCEPT
{
    return util::Strncmp(errorCodeStr, "2-", 2) == 0;
}

bool TryParseErrorCodeString(ErrorCode* outValue, const char* errorCodeStr) NN_NOEXCEPT
{
    // 正しいエラーコードのフォーマットは XXXX-YYYY。XXXX, YYYY ともに数字。
    if( strlen(errorCodeStr) >= ErrorCode::StringLengthMax )
    {
        return false;
    }
    char categoryStr[ErrorCode::StringLengthMax];
    auto pHyphen = strstr(errorCodeStr, "-");
    if( pHyphen == nullptr )
    {
        return false;
    }
    util::Strlcpy(categoryStr, errorCodeStr, static_cast<int>(pHyphen - errorCodeStr + 1));
    const char* numberStr = pHyphen + 1;
    // TORIAEZU : それぞれ長さが 1 以上あればよいことにする。
    if( strlen(categoryStr) >= 1 && strlen(numberStr) >= 1 && AreDigits(categoryStr) && AreDigits(numberStr) )
    {
        outValue->category = static_cast<ErrorCodeCategory>(atoi(categoryStr));
        outValue->number = static_cast<ErrorCodeNumber>(atoi(numberStr));
        return true;
    }
    else
    {
        return false;
    }
}

bool TryParseApplicationErrorCodeString(ns::ApplicationErrorCodeCategory* outCategory, ApplicationErrorCodeNumber* outNumber, const char* applicationErrorCodeStr) NN_NOEXCEPT
{
    // 正しいアプリケーションエラーコードのフォーマットは 2-AAAAA-BBBB。AAAAA は英数字、BBBB は数字。
    if( strlen(applicationErrorCodeStr) >= ErrorCode::StringLengthMax )
    {
        return false;
    }
    if( util::Strncmp(applicationErrorCodeStr, "2-", 2) != 0 )
    {
        return false;
    }
    auto pFirstHyphen = strstr(applicationErrorCodeStr, "-");
    if( pFirstHyphen == nullptr )
    {
        return false;
    }
    auto pSecondHyphen = strstr(pFirstHyphen + 1, "-");
    if( pSecondHyphen == nullptr )
    {
        return false;
    }
    const char* numberStr = pSecondHyphen + 1;
    // TORIAEZU : それぞれ長さが 1 以上あればよいことにする。
    if( strlen(numberStr) >= 1 && AreDigits(numberStr) )
    {
        util::Strlcpy(outCategory->value, pFirstHyphen + 1, static_cast<int>(pSecondHyphen - pFirstHyphen));
        *outNumber = static_cast<ApplicationErrorCodeNumber>(atoi(numberStr));
        return strlen(outCategory->value) >= 1;
    }
    else
    {
        return false;
    }
}


}}}
