﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/os.h>
#include <nn/nn_Result.h>
#include <nn/util/util_FormatString.h>
#include <nn/fs.h>
#include <nn/time.h>
#include <nn/psc.h>
#include <nn/nn_SystemThreadDefinition.h>

#include <nn/sf/sf_HipcServer.h>
#include <nn/sf/sf_ExpHeapAllocator.h>
#include <nn/sf/impl/sf_StaticOneAllocator.h>
#include <nn/sf/sf_ObjectFactory.h>

#include <nn/erpt/sfdl/erpt.sfdl.h>
#include <nn/erpt/erpt_Result.h>
#include <nn/erpt/server/erpt_Server.h>
#include <nn/erpt/server/erpt_ServerTypes.h>

#include "erptsrv_ISession.h"
#include "erptsrv_IContext.h"

#include "erptsrv_Context.h"
#include "erptsrv_Journal.h"
#include "erptsrv_Stream.h"

namespace nn   {
namespace erpt {
namespace srv  {

extern nn::sf::ExpHeapAllocator g_ServiceFrameworkAllocator;

namespace     {

const int SessionThreadStackSize = (16 * 1024);
const int ContextSessionCount    = 5;
const int ReportSessionCount     = 5;
const int SessionMaxCount        = ReportSessionCount + ContextSessionCount;

struct ServerOptions
{
    static const int SubDomainCountMax         = 64;
    static const int ObjectInSubDomainCountMax = 2 * NumberOfReports + 5;
};

enum ipcSessionPort
{
    sessionPort_Report,
    sessionPort_Context,
    sessionPort_Highest
};

class Service
    : public nn::sf::HipcSimpleAllInOneServerManager<SessionMaxCount, sessionPort_Highest, ServerOptions>
{
private:
    nn::os::ThreadType m_SessionThread;
    uint8_t            m_SessionThreadStack[SessionThreadStackSize] NN_ALIGNAS(os::StackRegionAlignment);

    nn::sf::UnmanagedServiceObject<nn::erpt::sf::IContext, nn::erpt::srv::IContext>
                       m_ContextSession;

    void SessionThread()
    NN_NOEXCEPT
    {
        nn::Result                  result;
        nn::psc::PmModule           module;
        nn::psc::PmModuleId         dependencyList[] = {nn::psc::PmModuleId_Fs};
        nn::psc::PmState            state;
        nn::psc::PmFlagSet          flags;
        nn::os::MultiWaitHolderType moduleEventHolder;

        result = module.Initialize(
            nn::psc::PmModuleId_Erpt,
            dependencyList,
            sizeof(dependencyList) / sizeof(dependencyList[0]),
            nn::os::EventClearMode_ManualClear);

        NN_ABORT_UNLESS_RESULT_SUCCESS(result);

        nn::os::InitializeMultiWaitHolder(
            &moduleEventHolder,
            module.GetEventPointer()->GetBase());

        nn::os::SetMultiWaitHolderUserData(
            &moduleEventHolder,
            nn::psc::PmModuleId_Erpt);

        HipcSimpleAllInOneServerManager::AddUserWaitHolder(&moduleEventHolder);

        while (NN_STATIC_CONDITION(true))
        {
            nn::os::MultiWaitHolderType* p = HipcSimpleAllInOneServerManager::Wait();

            switch (nn::os::GetMultiWaitHolderUserData(p))
            {
            case HipcSimpleAllInOneServerManager::AcceptTag:
            case HipcSimpleAllInOneServerManager::InvokeTag:
                ProcessAuto(p);
                break;
            case nn::psc::PmModuleId_Erpt:
                module.GetEventPointer()->Clear();
                if ((result = module.GetRequest(&state, &flags)).IsSuccess())
                {
                    switch (state)
                    {
                    case nn::psc::PmState_FullAwake:
                    case nn::psc::PmState_MinimumAwake:
                        Stream::EnableFsAccess(true);
                        break;
                    case nn::psc::PmState_SleepReady:
                    case nn::psc::PmState_ShutdownReady:
                        Stream::EnableFsAccess(false);
                        break;
                    default:
                        break;
                    }
                    module.Acknowledge(state, ResultSuccess());
                }
                else
                {
                    NN_SDK_ASSERT(
                        false,
                        "[erpt]: failed to get PM state (%d:%d)\n",
                        result.GetModule(), result.GetDescription()
                    );
                }
                HipcSimpleAllInOneServerManager::AddUserWaitHolder(&moduleEventHolder);
                break;
            default:
                break;
            }
        }
    }

    static void SessionThreadEntry(void* arg)
    NN_NOEXCEPT
    {
        Service* p = reinterpret_cast<Service*>(arg);
        p->SessionThread();
    }

    nn::Result OnNeedsToAccept(int portIndex, PortObjectImpl* pPort)
    NN_NOEXCEPT
    {
        if (portIndex == sessionPort_Context)
        {
            return this->AcceptImpl(pPort, m_ContextSession.GetShared());
        }
        else if (portIndex == sessionPort_Report)
        {
            auto interface = nn::sf::ObjectFactory<
                    nn::sf::ExpHeapAllocator::Policy
                    >::CreateSharedEmplaced<
                    nn::erpt::sf::ISession,
                    nn::erpt::srv::ISession
                    >(&g_ServiceFrameworkAllocator);

            NN_ABORT_UNLESS(interface != nullptr);

            return this->AcceptImpl(pPort, interface);
        }

        return nn::erpt::ResultNotSupported();
    }

public:
    void Wait()
    NN_NOEXCEPT
    {
        nn::os::WaitThread(&m_SessionThread);
    }

    nn::Result Initialize()
    NN_NOEXCEPT
    {
        nn::Result result;

        NN_ABORT_UNLESS_RESULT_SUCCESS(result);

        result = this->InitializePort(
                    sessionPort_Context,
                    ContextSessionCount,
                    ContextPortName);

        NN_ABORT_UNLESS_RESULT_SUCCESS(result);

        result = this->InitializePort(
                    sessionPort_Report,
                    ReportSessionCount,
                    ReportPortName);

        NN_ABORT_UNLESS_RESULT_SUCCESS(result);

        this->Start();

        result = nn::os::CreateThread(
                    &m_SessionThread,
                    SessionThreadEntry,
                    this,
                    m_SessionThreadStack,
                    SessionThreadStackSize,
                    NN_SYSTEM_THREAD_PRIORITY(erpt, IpcServer));

        NN_ABORT_UNLESS_RESULT_SUCCESS(result);

        nn::os::SetThreadNamePointer(&m_SessionThread, NN_SYSTEM_THREAD_NAME(erpt, IpcServer));

        nn::os::StartThread(&m_SessionThread);

        return ResultSuccess();
    }

} g_Service;

} // namespace

nn::Result InitializeService()
NN_NOEXCEPT
{
    return g_Service.Initialize();
}

void WaitService()
{
    g_Service.Wait();
}

}}}
