﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/erpt/erpt_Result.h>
#include <nn/erpt/server/erpt_ServerTypes.h>

#include "erptsrv_Report.h"

namespace nn   {
namespace erpt {
namespace srv  {

nn::Result Report::SetFlags(ReportFlagSet flagsIn)
NN_NOEXCEPT
{
    if ((~(m_pRecord->m_Info.reportFlags) & flagsIn).IsAnyOn())
    {
        m_pRecord->m_Info.reportFlags |= flagsIn;
        return Journal::Commit();
    }
    return ResultSuccess();
}

nn::Result Report::GetFlags(ReportFlagSet* pFlagsOut)
NN_NOEXCEPT
{
    *pFlagsOut = m_pRecord->m_Info.reportFlags;
    return ResultSuccess();
}

nn::Result Report::GetSize(int64_t* pSizeOut)
NN_NOEXCEPT
{
    return GetStreamSize(pSizeOut);
}

nn::Result Report::Open(ReportOpenType openType)
NN_NOEXCEPT
{
    nn::Result result;

    switch (openType)
    {
    case ReportOpenType_Create:
        result = OpenStream(FileName().name, StreamMode_Write, ReportStreamBufferSize);
        break;
    case ReportOpenType_Read:
        result = OpenStream(FileName().name, StreamMode_Read, ReportStreamBufferSize);
        break;
    default:
        result = nn::erpt::ResultInvalidArgument();
    }

    return result;
}

nn::Result Report::Read(uint32_t* pBytesRead, uint8_t* pBufferIn, uint32_t bufferLength)
NN_NOEXCEPT
{
    return ReadStream(pBytesRead, pBufferIn, bufferLength);
}

void Report::Close()
NN_NOEXCEPT
{
    CloseStream();
}

ReportFileName Report::FileName(ReportId reportId)
NN_NOEXCEPT
{
    ReportFileName reportName;

    // Keep the FileName format as it is so that error reports created in Pre-Install system can be retrieved after NUP.
    util::SNPrintf(
        reportName.name,
        sizeof(reportName.name),
        "%s:/%08x-%04x-%04x-%02x%02x-%04x%08x",
        ReportStoragePath,
        reportId.u.uuid.timeLow,
        reportId.u.uuid.timeMid,
        reportId.u.uuid.hiVersion,
        reportId.u.uuid.clockHi,
        reportId.u.uuid.clockLow,
        (reportId.u.uuid.node >> 32) & 0xffff,
        (reportId.u.uuid.node & 0xffffffff));

    return reportName;
}

ReportFileName Report::FileName()
NN_NOEXCEPT
{
    return FileName(m_pRecord->m_Info.reportId);
}

nn::Result Report::Delete()
NN_NOEXCEPT
{
    return DeleteStream(FileName().name);
}

Report::Report(JournalRecord* pRecord)
NN_NOEXCEPT :
    m_pRecord(pRecord)
{
    m_pRecord->AddReference();
}

Report::~Report()
NN_NOEXCEPT
{
    CloseStream();

    if (m_pRecord->RemoveReference())
    {
        DeleteStream(FileName().name);
        delete m_pRecord;
    }
}

}}}
