﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/erpt/server/erpt_ServerTypes.h>
#include <nn/erpt/erpt_Result.h>

#include "erptsrv_IManager.h"
#include "erptsrv_Journal.h"

namespace nn   {
namespace erpt {
namespace srv  {

nn::util::IntrusiveList<
      IManager, nn::util::IntrusiveListBaseNodeTraits<IManager>
    > IManager::m_ManagerList;

nn::Result IManager::NotifyAll()
NN_NOEXCEPT
{
    for (auto manager  = m_ManagerList.begin();
              manager != m_ManagerList.end();
              manager++)
    {
        manager->NotifyOne();
    }

    return ResultSuccess();
}

void IManager::NotifyOne()
NN_NOEXCEPT
{
    m_SystemEvent.Signal();
}

nn::Result IManager::GetReportList(nn::sf::OutBuffer listBufferOut, ReportType filterType)
NN_NOEXCEPT
{
    return listBufferOut.GetSize() == sizeof(ReportList) ?
           Journal::GetReportList(reinterpret_cast<ReportList*>(listBufferOut.GetPointerUnsafe()), filterType) :
           nn::erpt::ResultInvalidArgument();
}

nn::Result IManager::GetEvent(nn::sf::Out<nn::sf::NativeHandle> eventHandleOut)
NN_NOEXCEPT
{
    eventHandleOut.Set(nn::sf::NativeHandle(m_SystemEvent.GetReadableHandle(), false));
    return ResultSuccess();
}

nn::Result IManager::CleanupReports()
NN_NOEXCEPT
{
    Journal::CleanupReports();
    return Journal::Commit();
}

nn::Result IManager::DeleteReport(ReportId reportId)
NN_NOEXCEPT
{
    NN_RESULT_DO(Journal::Delete(reportId));
    return Journal::Commit();
}

nn::Result IManager::GetStorageUsageStatistics(nn::sf::Out<nn::erpt::StorageUsageStatistics> outUsage)
NN_NOEXCEPT
{
    outUsage->journalUuid = Journal::GetJournalId();
    outUsage->usedStorageSize = Journal::GetUsedStorage();
    outUsage->maxReportSize = Journal::GetMaxReportSize();
    for( int i = ReportType_First; i < ReportType_Last; i++ )
    {
        auto type = static_cast<ReportType>(i);
        outUsage->reportCount[i] = Journal::GetStoredReportCount(type);
        outUsage->transmittedCount[i] = Journal::GetTransmittedCount(type);
        outUsage->untransmittedCount[i] = Journal::GetUnstransmittedCount(type);
    }
    return ResultSuccess();
}

IManager::IManager()
NN_NOEXCEPT :
    m_SystemEvent(nn::os::EventClearMode_AutoClear, true)
{
    m_ManagerList.push_front(*this);
}

IManager::~IManager()
NN_NOEXCEPT
{
    m_ManagerList.erase(m_ManagerList.iterator_to(*this));
}

}}}
