﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "erptsrv_IManager.h"
#include "erptsrv_IContext.h"
#include "erptsrv_Context.h"
#include <nn/result/result_HandlingUtility.h>

namespace nn   {
namespace erpt {
namespace srv  {

nn::Result IContext::SubmitContext(
    const nn::sf::InBuffer& contextBufferIn,
    const nn::sf::InBuffer& dataBufferIn)
NN_NOEXCEPT
{
    const ContextEntry*   pContext      = reinterpret_cast<const ContextEntry*>(contextBufferIn.GetPointerUnsafe());
    const uint8_t*        pData         = reinterpret_cast<const uint8_t*>(dataBufferIn.GetPointerUnsafe());
    uint32_t              contextLength = static_cast<uint32_t>(contextBufferIn.GetSize());
    uint32_t              dataLength    = static_cast<uint32_t>(dataBufferIn.GetSize());

    if (contextLength != sizeof(ContextEntry) || dataLength > MaxArrayBufferLength)
    {
        return nn::erpt::ResultInvalidArgument();
    }

    return Context::SubmitContext(pContext, pData, dataLength);
}

nn::Result IContext::CreateReport(
    ReportType type,
    const nn::sf::InBuffer& contextBufferIn,
    const nn::sf::InBuffer& dataBufferIn,
    const nn::sf::InBuffer& metaBufferIn)
NN_NOEXCEPT
{
    nn::Result            result;
    const ReportMetaData* pMeta         = reinterpret_cast<const ReportMetaData*>(metaBufferIn.GetPointerUnsafe());
    const ContextEntry*   pContext      = reinterpret_cast<const ContextEntry*>(contextBufferIn.GetPointerUnsafe());
    const uint8_t*        pData         = reinterpret_cast<const uint8_t*>(dataBufferIn.GetPointerUnsafe());
    uint32_t              contextLength = static_cast<uint32_t>(contextBufferIn.GetSize());
    uint32_t              dataLength    = static_cast<uint32_t>(dataBufferIn.GetSize());
    uint32_t              metaLength    = static_cast<uint32_t>(metaBufferIn.GetSize());

    if (contextLength != sizeof(ContextEntry) || (metaLength != 0 && metaLength != sizeof(ReportMetaData)))
    {
        return nn::erpt::ResultInvalidArgument();
    }

    result = Context::CreateReport(type, pContext, pData, dataLength, metaLength ? pMeta : nullptr);

    if (result.IsSuccess())
    {
        IManager::NotifyAll();
    }

    return result;
}

nn::Result IContext::SetInitialLaunchSettingsCompletionTime(
    const nn::time::SteadyClockTimePoint& timePoint
)
NN_NOEXCEPT
{
    Context::SetInitialLaunchSettingsCompletionTime(timePoint);
    NN_RESULT_SUCCESS;
}

nn::Result IContext::ClearInitialLaunchSettingsCompletionTime()
NN_NOEXCEPT
{
    Context::ClearInitialLaunchSettingsCompletionTime();
    NN_RESULT_SUCCESS;
}

nn::Result IContext::UpdatePowerOnTime()
NN_NOEXCEPT
{
    Context::UpdatePowerOnTime();
    NN_RESULT_SUCCESS;
}

nn::Result IContext::UpdateAwakeTime()
NN_NOEXCEPT
{
    Context::UpdateAwakeTime();
    NN_RESULT_SUCCESS;
}

nn::Result IContext::SubmitMultipleCategoryContext(
    const MultipleCategoryContextEntry& entry,
    const nn::sf::InBuffer& stringsBufferIn)
NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(entry.categoryCount >= 0 && entry.categoryCount <= erpt::CategoriesPerMultipleCategoryContext, ResultInvalidArgument());

    auto stringBuffer = reinterpret_cast<const uint8_t*>(stringsBufferIn.GetPointerUnsafe());
    auto stringBufferSize = static_cast<uint32_t>(stringsBufferIn.GetSize());

    uint32_t fieldCountSum = 0;
    uint32_t arrayCountSum = 0;
    for( uint32_t i = 0; i < entry.categoryCount; i++ )
    {
        ContextEntry e;
        e.version = entry.version;
        e.category = entry.categories[i];
        e.fieldCount = entry.fieldCounts[i];
        NN_RESULT_THROW_UNLESS(e.fieldCount <= erpt::FieldsPerContext, ResultInvalidArgument());
        NN_RESULT_THROW_UNLESS(e.fieldCount + fieldCountSum <= erpt::FieldsPerMultipleCategoryContext, ResultInvalidArgument());
        NN_RESULT_THROW_UNLESS(entry.arrayBufferCounts[i] <= erpt::MaxArrayBufferLength, ResultInvalidArgument());
        NN_RESULT_THROW_UNLESS(arrayCountSum + entry.arrayBufferCounts[i] <= stringBufferSize, ResultInvalidArgument());
        std::memcpy(e.fields, entry.fields + fieldCountSum, e.fieldCount * sizeof(FieldEntry));
        NN_RESULT_DO(Context::SubmitContext(&e, stringBuffer + arrayCountSum, entry.arrayBufferCounts[i]));
        fieldCountSum += e.fieldCount;
        arrayCountSum += entry.arrayBufferCounts[i];
    }
    NN_RESULT_SUCCESS;
}

IContext::IContext()
NN_NOEXCEPT
{

}

}}}
