﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/os.h>
#include <nn/erpt/erpt_Result.h>
#include <nn/erpt/server/erpt_ServerTypes.h>
#include <nn/time/time_SteadyClockTimePoint.h>
#include <nn/util/util_IntrusiveList.h>
#include <nn/util/util_Optional.h>

#include "erptsrv_Allocator.h"
#include "erptsrv_Cipher.h"

namespace nn   {
namespace erpt {
namespace srv  {

class Report;
class ContextRecord;
class JournalRecord;

class Context :
    public Allocator,
    public nn::util::IntrusiveListBaseNode<Context>
{

private:
    struct ReportDefaultValue
    {
        FieldId   id;
        FieldType type;
        bool      exists;
        union
        {
            uint64_t numericFieldU64;
            uint32_t numericFieldU32;
            uint16_t numericFieldU16;
            uint8_t  numericFieldU8;
            int64_t  numericFieldI64;
            int32_t  numericFieldI32;
            int16_t  numericFieldI16;
            int8_t   numericFieldI8;
            bool     boolField;
        }
        value;
    };

    const CategoryId m_Category;
    const uint32_t   m_MaxRecordCount;
    uint32_t         m_RecordCount;
    static char      s_SerialNumber[24];
    static char      s_OsVersion[24];
    static char      s_PrivateOsVersion[96];
    static util::optional<time::SteadyClockTimePoint> s_InitialLaunchSettingsCompletionTime;
    static util::optional<time::SteadyClockTimePoint> s_PowerOnTime;
    static util::optional<time::SteadyClockTimePoint> s_AwakeTime;

    static nn::util::IntrusiveList<
         Context, nn::util::IntrusiveListBaseNodeTraits<Context>
        > s_CategoryList;

    nn::util::IntrusiveList<
         ContextRecord, nn::util::IntrusiveListBaseNodeTraits<ContextRecord>
        > m_RecordList;

    nn::Result AddContextToCategory(
        const ContextEntry*   pContext,
        const uint8_t*        pData,
        uint32_t              dataLength)
    NN_NOEXCEPT;

    nn::Result AddContextToCategory(
        ContextRecord*        pContextRecord)
    NN_NOEXCEPT;

    nn::Result AddCategoryToReport(
        Report*               pReport)
    NN_NOEXCEPT;

    static nn::Result SubmitReportDefaults(
        const ContextEntry*   pContextIn)
    NN_NOEXCEPT;

    static nn::Result AddRequiredFieldsToReport(
        Report*               pRecord)
    NN_NOEXCEPT;

    static nn::Result ValidateReportContext(
        const ContextEntry*   pContextIn)
    NN_NOEXCEPT;

    static ReportId GenerateReportId()
        NN_NOEXCEPT;

    static nn::Result CreateReportFile(
        JournalRecord* pRecordIn)
    NN_NOEXCEPT;

    static nn::Result CreateJournalRecord(
        JournalRecord**       pRecordOut,
        ReportType            type,
        const ReportMetaData* pMeta)
    NN_NOEXCEPT;

    static nn::Result SubmitErrorInfoAutoContext(
        JournalRecord*        pJournalRecord,
        ContextRecord*        pContextRecord
    )
    NN_NOEXCEPT;

public:
    static nn::Result SetSerialNumberAndOsVersion(
        const char*           pSerialNumber,
        uint32_t              serialNumberLength,
        const char*           pOsVersion,
        uint32_t              osVersionLength,
        const char*           pPrivateOsVersion,
        uint32_t              privateOsVersionLength)
    NN_NOEXCEPT;

    static void SetInitialLaunchSettingsCompletionTime(
        const time::SteadyClockTimePoint& timePoint)
    NN_NOEXCEPT;

    static void ClearInitialLaunchSettingsCompletionTime()
    NN_NOEXCEPT;

    static void UpdatePowerOnTime()
    NN_NOEXCEPT;

    static void UpdateAwakeTime()
    NN_NOEXCEPT;

    static nn::Result SubmitContext(
        const ContextEntry*   pContext,
        const uint8_t*        pData,
        uint32_t              dataLength)
    NN_NOEXCEPT;

    static nn::Result SubmitContext(
        ContextRecord*        pContextRecord)
    NN_NOEXCEPT;

    static nn::Result CreateReport(
        ReportType            type,
        const ContextEntry*   pContext,
        const uint8_t*        pData,
        uint32_t              dataLength,
        const ReportMetaData* pMeta)
    NN_NOEXCEPT;

    Context(
        CategoryId categoryId,
        uint32_t   recordCount)
    NN_NOEXCEPT;

    ~Context()
    NN_NOEXCEPT;
};

}}}
