﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/os.h>
#include <nn/os/os_SdkMutex.h>
#include <nn/sf/sf_ShimLibraryUtility.h>
#include <nn/sf/sf_ExpHeapAllocator.h>
#include <nn/util/util_LockGuard.h>
#include <nn/erpt/server/erpt_Server.h>
#include <nn/erpt/sfdl/erpt.sfdl.h>

namespace nn   {
namespace erpt {

struct HeapTag;
const int32_t SessionHeapSize = (8 * 1024);

class InitExpHeap
{
public:
    InitExpHeap()
    NN_NOEXCEPT
    {
        nn::sf::ExpHeapStaticAllocator
            <SessionHeapSize, HeapTag>::
            Initialize(nn::lmem::CreationOption_ThreadSafe);
    }
} g_InitExpHeap;

class Session
{

private:
    uint32_t         m_RefCount = 0u;
    nn::os::SdkMutex m_Lock;

    nn::sf::ShimLibraryObjectHolder<nn::erpt::sf::ISession> m_Session =
           NN_SF_SHIM_LIBRARY_OBJECT_HOLDER_INITIALIZER;

    nn::sf::SimpleAllInOneHipcSubDomainClientManager<NumberOfReports + 2> m_Domain =
           NN_SF_SIMPLE_ALL_IN_ONE_HIPC_SUB_DOMAIN_CLIENT_MANAGER_INITIALIZER;

public:
    nn::Result GetSessionObject(nn::sf::SharedPointer<nn::erpt::sf::ISession>* pSessionObject)
    NN_NOEXCEPT
    {
        nn::Result result;

        NN_UTIL_LOCK_GUARD(m_Lock);

        if (m_RefCount == 0)
        {
            if ((result = m_Domain.InitializeShimLibraryHolder(
                          &m_Session,
                          ReportPortName)).IsFailure())
            {
                return result;
            }
        }

        *pSessionObject = m_Session.GetObject();

        m_RefCount++;

        return nn::ResultSuccess();
    }

    void Release()
    NN_NOEXCEPT
    {
        NN_UTIL_LOCK_GUARD(m_Lock);

        if (m_RefCount == 1)
        {
            m_Session.FinalizeHolder();
        }

        m_RefCount--;

        NN_ABORT_UNLESS(m_RefCount >= 0);
    }

    Session()
    NN_NOEXCEPT = default;
}
g_Session;

nn::Result GetSessionObject(nn::sf::SharedPointer<sf::ISession> *pSessionObjectOut)
NN_NOEXCEPT
{
    return g_Session.GetSessionObject(pSessionObjectOut);
}

void ReleaseSessionObject()
{
    g_Session.Release();
}

nn::Result GetContextObject(nn::sf::SharedPointer<sf::IContext> *pContextObjectOut)
NN_NOEXCEPT
{
    return nn::sf::CreateHipcProxyByName
           <sf::IContext, nn::sf::ExpHeapStaticAllocator
           <SessionHeapSize, HeapTag>::Policy>
           (pContextObjectOut, ContextPortName);
}

}}
