﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/erpt.h>
#include <nn/vi.h>
#include <nn/vi/vi_LibMinimum.h>
#include <nn/vi/vi_DisplayError.h>
#include <nn/vi/vi_DisplayEvents.h>
#include <nn/vi/vi_DisplayModeInfo.h>
#include <nn/vi/vi_DisplayMode.private.h>
#include <nn/util/util_StringUtil.h>
#include "eclct_Util.h"
#include "eclct_Display.h"

namespace nn    {
namespace eclct {

namespace {

bool g_Initialized = false;
nn::vi::Display* g_pDisplay = nullptr;
nn::vi::Display* g_pExternal = nullptr;

void Initialize() NN_NOEXCEPT
{
    if ( !g_Initialized )
    {
        nn::vi::InitializeMinimum();
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::vi::OpenDefaultDisplay(&g_pDisplay));
        // Certain hardware configurations may lack an external display.  nn::vi::OpenDisplay will fail
        // in these scenarios.
        nn::vi::OpenDisplay(&g_pExternal, "External");

        g_Initialized = true;
    }
}

}

void GetHotplugEvent(nn::os::SystemEventType* pOutSystemEvent) NN_NOEXCEPT
{
    Initialize();

    if( g_pExternal != nullptr )
    {
        nn::vi::GetDisplayHotplugEvent(pOutSystemEvent, g_pExternal);
    }
    else
    {
        // Create a dummy system event if external display isn't available.
        nn::os::CreateSystemEvent(pOutSystemEvent, nn::os::EventClearMode_AutoClear, false);
    }
}

void GetCompositorInfoEvent(nn::os::SystemEventType* pOutSystemEvent) NN_NOEXCEPT
{
    Initialize();
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::vi::GetDisplayErrorEvent(pOutSystemEvent, g_pDisplay));
}

void GetResolutionChangeEvent(nn::os::SystemEvent* pOutSystemEvent) NN_NOEXCEPT
{
    Initialize();

    if( g_pExternal != nullptr )
    {
        nn::vi::GetDisplayModeChangedEvent(pOutSystemEvent->GetBase(), g_pExternal);
    }
    else
    {
        // Create a dummy system event if external display isn't available.
        nn::os::CreateSystemEvent(pOutSystemEvent->GetBase(), nn::os::EventClearMode_AutoClear, false);
    }
}

void UpdateCompositorInfo() NN_NOEXCEPT
{
    Initialize();
    nn::vi::CompositorError info;
    int length;
    uint8_t buffer[nn::vi::ErrorLengthMax];


    //CompositorStateInfo
    {
        length = nn::vi::GetCompositorErrorInfo(&info, nn::erpt::CompositorState, g_pDisplay);
        if(length > 0)
        {
            //NN_DETAIL_ECLCT_INFO("CompositorState %d\n", length);
            nn::erpt::Context context(nn::erpt::CompositorStateInfo, buffer, nn::vi::ErrorLengthMax);
            NN_DETAIL_ECLCT_WARN_UNLESS_RESULT_SUCCESS(context.Add(nn::erpt::CompositorState, info.buffer, length));
            NN_DETAIL_ECLCT_WARN_UNLESS_RESULT_SUCCESS(context.SubmitContext());
        }
    }

    //CompositorDisplayState
    {
        length = nn::vi::GetCompositorErrorInfo(&info, nn::erpt::CompositorDisplayState, g_pDisplay);
        if(length > 0)
        {
            //NN_DETAIL_ECLCT_INFO("compositorDisplay %d\n", length);
            nn::erpt::Context context(nn::erpt::CompositorDisplayInfo, buffer, nn::vi::ErrorLengthMax);
            NN_DETAIL_ECLCT_WARN_UNLESS_RESULT_SUCCESS(context.Add(nn::erpt::CompositorDisplayState, info.buffer, length));
            NN_DETAIL_ECLCT_WARN_UNLESS_RESULT_SUCCESS(context.SubmitContext());

        }
    }

    //CompositorHWCState
    {
        length = nn::vi::GetCompositorErrorInfo(&info, nn::erpt::CompositorHWCState, g_pDisplay);
        if(length > 0)
        {
            //NN_DETAIL_ECLCT_INFO("compositorHWCState %d\n", length);
            nn::erpt::Context context(nn::erpt::CompositorHWCInfo, buffer, nn::vi::ErrorLengthMax);
            NN_DETAIL_ECLCT_WARN_UNLESS_RESULT_SUCCESS(context.Add(nn::erpt::CompositorHWCState, info.buffer, length));
            NN_DETAIL_ECLCT_WARN_UNLESS_RESULT_SUCCESS(context.SubmitContext());
        }
    }

    //CompositorLayerState
    {
        length = nn::vi::GetCompositorErrorInfo(&info, nn::erpt::CompositorLayerState, g_pDisplay);
        if(length > 0)
        {
            //NN_DETAIL_ECLCT_INFO("compositorLayerState %d\n", length);
            nn::erpt::Context context(nn::erpt::CompositorLayerInfo, buffer, nn::vi::ErrorLengthMax);
            NN_DETAIL_ECLCT_WARN_UNLESS_RESULT_SUCCESS(context.Add(nn::erpt::CompositorLayerState, info.buffer, length));
            NN_DETAIL_ECLCT_WARN_UNLESS_RESULT_SUCCESS(context.SubmitContext());
        }
    }

    //Report
    nn::erpt::Context contextReport(nn::erpt::ErrorInfo);
    NN_DETAIL_ECLCT_WARN_UNLESS_RESULT_SUCCESS(contextReport.Add(nn::erpt::ErrorCode, "2520-0100", 9)); //Error code has been decided on NSBG-6968(nvhos_errorcodes.h)
    NN_DETAIL_ECLCT_WARN_UNLESS_RESULT_SUCCESS(contextReport.CreateReport(nn::erpt::ReportType_Invisible));

    //Clear data by submitting an empty context
    {
        nn::erpt::Context context(nn::erpt::CompositorStateInfo);
        NN_DETAIL_ECLCT_WARN_UNLESS_RESULT_SUCCESS(context.SubmitContext());
    }

    {
        nn::erpt::Context context(nn::erpt::CompositorDisplayInfo);
        NN_DETAIL_ECLCT_WARN_UNLESS_RESULT_SUCCESS(context.SubmitContext());
    }

    {
        nn::erpt::Context context(nn::erpt::CompositorHWCInfo);
        NN_DETAIL_ECLCT_WARN_UNLESS_RESULT_SUCCESS(context.SubmitContext());
    }

    {
        nn::erpt::Context context(nn::erpt::CompositorLayerInfo);
        NN_DETAIL_ECLCT_WARN_UNLESS_RESULT_SUCCESS(context.SubmitContext());
    }
}

void UpdateMonitorSettings() NN_NOEXCEPT
{
    Initialize();

    if( g_pExternal != nullptr )
    {
        nn::vi::DisplayModeInfo mode{ };
        // It's possible for this to fail, but the error will be ignored in case
        // the system is in handheld mode.
        nn::vi::GetDisplayMode(&mode, g_pExternal);

        auto context = nn::erpt::Context(nn::erpt::CategoryId::MonitorSettings);

        NN_DETAIL_ECLCT_WARN_UNLESS_RESULT_SUCCESS(context.Add(nn::erpt::MonitorCurrentWidth, static_cast<std::uint16_t>(mode.width)));
        NN_DETAIL_ECLCT_WARN_UNLESS_RESULT_SUCCESS(context.Add(nn::erpt::MonitorCurrentHeight, static_cast<std::uint16_t>(mode.height)));

        char refreshRateStr[6];
        nn::util::SNPrintf(refreshRateStr, sizeof(refreshRateStr), "%02.2f", mode.refreshRate);
        NN_DETAIL_ECLCT_WARN_UNLESS_RESULT_SUCCESS(context.Add(nn::erpt::MonitorCurrentRefreshRate,
                                                               refreshRateStr,
                                                               nn::util::Strnlen(refreshRateStr, sizeof(refreshRateStr))));

        NN_DETAIL_ECLCT_WARN_UNLESS_RESULT_SUCCESS(context.SubmitContext());
    }
}

}}
