﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/erpt.h>
#include <nn/err.h>
#include <nn/err/detail/err_ErrorCodeConvert.h>
#include <nn/psm/psm_Api.h>
#include <nn/psm/psm_ErrorReport.h>
#include <nn/psm/psm_ResultPrivate.h>

#include "eclct_BatteryCharge.h"
#include "eclct_Util.h"

namespace nn    {
namespace eclct {

namespace {

bool g_Initialized = false;
bool g_OnceReported = false;

void Initialize() NN_NOEXCEPT
{
    if ( !g_Initialized )
    {
        nn::psm::Initialize();
        g_Initialized = true;
    }
}

}

void GetBatteryChargeInfoEvent(nn::os::SystemEventType* pOutSystemEvent, nn::os::EventClearMode eventClearMode) NN_NOEXCEPT
{
    Initialize();

    nn::psm::GetBatteryChargeInfoEvent(pOutSystemEvent, eventClearMode);
}

void GetBatteryChargeCalibratedEvent(nn::os::SystemEventType* pOutSystemEvent, nn::os::EventClearMode eventClearMode) NN_NOEXCEPT
{
    Initialize();

    nn::psm::GetBatteryChargeCalibratedEvent(pOutSystemEvent, eventClearMode);
}

void UpdateBatteryChargeInfo() NN_NOEXCEPT
{
    Initialize();

    nn::erpt::Context context(nn::erpt::BatteryChargeInfo);

    nn::psm::BatteryChargeInfoFields fields;

    fields.Clear();

    nn::psm::GetBatteryChargeInfoFields(&fields);

    // 充電 IC の設定
    NN_DETAIL_ECLCT_WARN_UNLESS_RESULT_SUCCESS(context.Add(nn::erpt::InputCurrentLimit,         fields.inputCurrentLimit));
    NN_DETAIL_ECLCT_WARN_UNLESS_RESULT_SUCCESS(context.Add(nn::erpt::FastChargeCurrentLimit,    fields.fastChargeCurrentLimit));
    NN_DETAIL_ECLCT_WARN_UNLESS_RESULT_SUCCESS(context.Add(nn::erpt::BoostModeCurrentLimit,     fields.boostModeCurrentLimit));
    NN_DETAIL_ECLCT_WARN_UNLESS_RESULT_SUCCESS(context.Add(nn::erpt::ChargeVoltageLimit,        fields.chargeVoltageLimit));
    NN_DETAIL_ECLCT_WARN_UNLESS_RESULT_SUCCESS(context.Add(nn::erpt::ChargeConfiguration,       fields.chargeConfiguration));
    NN_DETAIL_ECLCT_WARN_UNLESS_RESULT_SUCCESS(context.Add(nn::erpt::HizMode,                   fields.hizMode));

    // GPIO の設定
    NN_DETAIL_ECLCT_WARN_UNLESS_RESULT_SUCCESS(context.Add(nn::erpt::ChargeEnabled,             fields.chargeEnabled));
    NN_DETAIL_ECLCT_WARN_UNLESS_RESULT_SUCCESS(context.Add(nn::erpt::PowerSupplyPath,           fields.powerSupplyPath));

    // 電池状態
    NN_DETAIL_ECLCT_WARN_UNLESS_RESULT_SUCCESS(context.Add(nn::erpt::BatteryTemperature,        fields.batteryTemperature));
    NN_DETAIL_ECLCT_WARN_UNLESS_RESULT_SUCCESS(context.Add(nn::erpt::BatteryChargePercent,      fields.batteryChargePercent));
    NN_DETAIL_ECLCT_WARN_UNLESS_RESULT_SUCCESS(context.Add(nn::erpt::BatteryChargeVoltage,      fields.batteryChargeVoltage));
    NN_DETAIL_ECLCT_WARN_UNLESS_RESULT_SUCCESS(context.Add(nn::erpt::BatteryAge,                fields.batteryAge));

    // 給電状態
    NN_DETAIL_ECLCT_WARN_UNLESS_RESULT_SUCCESS(context.Add(nn::erpt::PowerRole,                 fields.powerRole));
    NN_DETAIL_ECLCT_WARN_UNLESS_RESULT_SUCCESS(context.Add(nn::erpt::PowerSupplyType,           fields.powerSupplyType));
    NN_DETAIL_ECLCT_WARN_UNLESS_RESULT_SUCCESS(context.Add(nn::erpt::PowerSupplyVoltage,        fields.powerSupplyVoltage));
    NN_DETAIL_ECLCT_WARN_UNLESS_RESULT_SUCCESS(context.Add(nn::erpt::PowerSupplyCurrent,        fields.powerSupplyCurrent));

    // 外部条件
    NN_DETAIL_ECLCT_WARN_UNLESS_RESULT_SUCCESS(context.Add(nn::erpt::FastBatteryChargingEnabled,    fields.fastBatteryChargingEnabled));
    NN_DETAIL_ECLCT_WARN_UNLESS_RESULT_SUCCESS(context.Add(nn::erpt::ControllerPowerSupplyAcquired, fields.controllerPowerSupplyAcquired));
    NN_DETAIL_ECLCT_WARN_UNLESS_RESULT_SUCCESS(context.Add(nn::erpt::OtgRequested,                  fields.otgRequested));

    NN_DETAIL_ECLCT_WARN_UNLESS_RESULT_SUCCESS(context.SubmitContext());
}

void ReportBatteryChargeCalibrated() NN_NOEXCEPT
{
    Initialize();

    // BatteryChargeCalibrated のエラーレポート送信は、起動中に一度しか行わない
    if (g_OnceReported)
    {
        return;
    }
    g_OnceReported = true;

    nn::erpt::Context context(nn::erpt::ErrorInfo);

    // エラーレポートの作成にエラーコードの指定が必要で、エラーコードは文字列で指定する必要がある
    auto errorCode = nn::err::detail::ConvertResultToErrorCode(nn::psm::ResultBatteryChargeCalibrated());
    char errorCodeString[nn::err::ErrorCode::StringLengthMax];
    util::SNPrintf(errorCodeString, sizeof(errorCodeString), "%04d-%04d", errorCode.category, errorCode.number);
    NN_DETAIL_ECLCT_WARN_UNLESS_RESULT_SUCCESS(context.Add(nn::erpt::ErrorCode, errorCodeString, static_cast<uint32_t>(sizeof(errorCodeString))));

    NN_DETAIL_ECLCT_WARN_UNLESS_RESULT_SUCCESS(context.CreateReport(nn::erpt::ReportType_Invisible));
}

}}
