﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/ec/system/ec_DeviceAccountApi.h>
#include <nn/ec/system/ec_DeviceAccountSystemApi.h>
#include <nn/ec/ec_Result.h>
#include <nn/nifm/nifm_ApiClientManagement.h>

#include "ec_ResultUtil.h"
#include "ec_ShopServiceObject.h"

namespace nn { namespace ec { namespace system {
    Result GetDeviceAccountInfo(DeviceAccountInfo* outValue) NN_NOEXCEPT
    {
        NN_EC_RESULT_DO(GetShopServiceObject().GetDeviceAccountInfo(outValue));
        NN_RESULT_SUCCESS;
    }

    Result RequestRegisterDeviceAccount(AsyncResult* outValue) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(nifm::IsAnyInternetRequestAccepted(nifm::GetClientId()), ResultInternetRequestNotAccepted());
        NN_RESULT_DO(RequestRegisterDeviceAccountPrivate(outValue));

        NN_RESULT_SUCCESS;
    }

    Result RequestUnregisterDeviceAccount(AsyncResult* outValue) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(nifm::IsAnyInternetRequestAccepted(nifm::GetClientId()), ResultInternetRequestNotAccepted());
        NN_RESULT_DO(RequestUnregisterDeviceAccountPrivate(outValue));

        NN_RESULT_SUCCESS;
    }


    Result RequestDeviceRegistrationInfo(AsyncDeviceRegistrationInfo* outValue) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(nifm::IsAnyInternetRequestAccepted(nifm::GetClientId()), ResultInternetRequestNotAccepted());
        NN_RESULT_DO(RequestDeviceRegistrationInfoPrivate(outValue));

        NN_RESULT_SUCCESS;
    }

    Result RequestDeviceAccountStatus(AsyncDeviceAccountStatus* outValue) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(nifm::IsAnyInternetRequestAccepted(nifm::GetClientId()), ResultInternetRequestNotAccepted());

        sf::NativeHandle nativeHandle;
        sf::SharedPointer<nim::detail::IAsyncValue> sp;
        NN_EC_RESULT_DO(GetShopServiceObject().RequestDeviceAccountStatus(&nativeHandle, &sp));
        outValue->Initialize(sp, nativeHandle);
        nativeHandle.Detach();

        NN_RESULT_SUCCESS;
    }

    Result RequestTransferDeviceAccount(AsyncResult* outValue) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(nifm::IsAnyInternetRequestAccepted(nifm::GetClientId()), ResultInternetRequestNotAccepted());

        sf::NativeHandle nativeHandle;
        sf::SharedPointer<nim::detail::IAsyncResult> sp;
        NN_EC_RESULT_DO(GetShopServiceObject().RequestTransferDeviceAccount(&nativeHandle, &sp));
        outValue->Initialize(sp, nativeHandle);
        nativeHandle.Detach();

        NN_RESULT_SUCCESS;
    }

    Result RequestSyncRegistration(AsyncResult* outValue) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(nifm::IsAnyInternetRequestAccepted(nifm::GetClientId()), ResultInternetRequestNotAccepted());

        sf::NativeHandle nativeHandle;
        sf::SharedPointer<nim::detail::IAsyncResult> sp;
        NN_EC_RESULT_DO(GetShopServiceObject().RequestSyncRegistration(&nativeHandle, &sp));
        outValue->Initialize(sp, nativeHandle);
        nativeHandle.Detach();

        NN_RESULT_SUCCESS;
    }

    bool IsOwnDeviceId(uint64_t deviceId) NN_NOEXCEPT
    {
        bool isOwn;
        NN_ABORT_UNLESS_RESULT_SUCCESS(GetShopServiceObject().IsOwnDeviceId(&isOwn, deviceId));

        return isOwn;
    }

    Result RequestRegisterDeviceAccountPrivate(AsyncResult* outValue) NN_NOEXCEPT
    {
        sf::NativeHandle nativeHandle;
        sf::SharedPointer<nim::detail::IAsyncResult> sp;
        NN_EC_RESULT_DO(GetShopServiceObject().RequestRegisterDeviceAccount(&nativeHandle, &sp));
        outValue->Initialize(sp, nativeHandle);
        nativeHandle.Detach();

        NN_RESULT_SUCCESS;
    }

    Result RequestUnregisterDeviceAccountPrivate(AsyncResult* outValue) NN_NOEXCEPT
    {
        sf::NativeHandle nativeHandle;
        sf::SharedPointer<nim::detail::IAsyncResult> sp;
        NN_EC_RESULT_DO(GetShopServiceObject().RequestUnregisterDeviceAccount(&nativeHandle, &sp));
        outValue->Initialize(sp, nativeHandle);
        nativeHandle.Detach();

        NN_RESULT_SUCCESS;
    }

    Result RequestDeviceRegistrationInfoPrivate(AsyncDeviceRegistrationInfo* outValue) NN_NOEXCEPT
    {
        sf::NativeHandle nativeHandle;
        sf::SharedPointer<nim::detail::IAsyncValue> sp;
        NN_EC_RESULT_DO(GetShopServiceObject().RequestDeviceRegistrationInfo(&nativeHandle, &sp));
        outValue->Initialize(sp, nativeHandle);
        nativeHandle.Detach();

        NN_RESULT_SUCCESS;
    }
}}}
