﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>
#include <string>
#include <type_traits>
#include <nn/ec/ec_Result.h>
#include <nn/ec/ec_ResultPrivate.h>
#include <nn/ec/ec_ResultShopServiceAccessor.h>
#include <nn/ec/ec_ShopServiceAccessorForDebug.h>
#include <nn/os/os_Mutex.h>
#include <nn/sf/sf_ProxyObjectAllocator.h>
#include <nn/sf/sf_NativeHandle.h>
#include <nn/sf/sf_HipcClient.h>
#include <nn/sf/sf_ObjectFactory.h>
#include <nn/util/util_Optional.h>
#include <nn/util/util_StringUtil.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/nim/nim_Result.h>
#include <nn/nim/nim_ShopServiceAccessTypes.h>
#include <nn/nim/nim_ShopServiceAccessConfig.h>
#include <nn/nim/detail/nim_ServiceName.h>
#include <nn/nim/detail/nim_IShopServiceAccessServer.sfdl.h>

namespace nn { namespace ec {

namespace {

/**
 * @brief   サービスサーバーIPCプロキシ。
 *
 * @details 本プロキシは、@ref nn::ec::ShopServiceAccessor が利用可能状態 ( @ref nn::ec::InitializeForShopServiceAccessors() ～ @ref nn::ec::FinalizeForShopServiceAccessors() )でもセッションを取得する事ができます。
 *          但し、@ref nn::ec::ShopServiceAccessor と同じポート及びインタフェースを利用するため、デバッグ対象機能である @ref nn::ec::ShopServiceAccessor のサーバー側のセッションリソースを消費します。@n
 *
 *          また、このサーバー側セッションリソース消費に対する本デバッグ機能用のための専用のセッションリソース増加は行っていません。@n
 *          そのため、@ref nn::ec::ShopServiceAccessor が利用可能状態で本機能を利用する場合は、以下の制約がある点注意してください。@n
 *
 *          @li @ref nn::ec::ShopServiceAccessor からの要求、もしくは本機能からの要求において、リソース消費状況に伴ったリソース不足のエラーを返す可能性があります。
 */
class ShopServiceAccessDebugProxy
{
    NN_DISALLOW_COPY(ShopServiceAccessDebugProxy);
    NN_DISALLOW_MOVE(ShopServiceAccessDebugProxy);

private:
    /**
     * @brief   維持想定セッション数。
     */
    static const int IpcSessionLimit = 1;

    typedef ::nn::os::Mutex                                         TMutex;
    typedef ::nn::nim::detail::IShopServiceAccessServerInterface    IService;
    typedef ::nn::sf::ProxyObjectAllocator<IpcSessionLimit>         TAllocator;
    typedef ::nn::sf::SharedPointer<IService>                       TServiceProxy;

    //! @brief  サーバーリソース解放許可通知。
    NN_FORCEINLINE void NotifyFinalizeToServer() NN_NOEXCEPT
    {
        m_ProxyImpl = nullptr;
    }

    //! @brief  デフォルトコンストラクタ。
    NN_IMPLICIT ShopServiceAccessDebugProxy() NN_NOEXCEPT
        : m_ReferLock(true), m_RefCounter(0)
    {
        m_Allocator = NN_SF_PROXY_OBJECT_ALLOCATOR_INITIALIZER;
        m_Allocator.Initialize();
    }

    //! @brief  デストラクタ。
    ~ShopServiceAccessDebugProxy() NN_NOEXCEPT
    {
        std::lock_guard<decltype(m_ReferLock)> guard(m_ReferLock);
        NotifyFinalizeToServer();
        m_Allocator.Finalize();
        m_RefCounter = 0;
    }

public:
    //! @brief  シングルトンインスタンス取得。
    static ShopServiceAccessDebugProxy& GetInstance() NN_NOEXCEPT
    {
        NN_FUNCTION_LOCAL_STATIC(ShopServiceAccessDebugProxy, s_ProxyImpl);
        return s_ProxyImpl;
    }

    //! @brief  インタフェース取得＆初期化。
    Result Initialize() NN_NOEXCEPT;

    //! @brief  インタフェース解放＆終了。
    void Finalize() NN_NOEXCEPT;

    //! @brief  デバッグ支援機能の利用可否状態の更新要求。
    Result RefreshDebugAvailability() NN_NOEXCEPT
    {
        std::lock_guard<decltype(m_ReferLock)> guard(m_ReferLock);
        NN_SDK_ASSERT(static_cast<bool>(m_ProxyImpl), "Has not been initialized the features.");
        return m_ProxyImpl->RefreshDebugAvailability();
    }

    //! @brief  登録済デバッグレスポンスの全消去要求。
    Result ClearDebugResponse() NN_NOEXCEPT
    {
        std::lock_guard<decltype(m_ReferLock)> guard(m_ReferLock);
        NN_SDK_ASSERT(static_cast<bool>(m_ProxyImpl), "Has not been initialized the features.");
        return m_ProxyImpl->ClearDebugResponse();
    }

    //! @brief  デバッグレスポンス登録要求。
    Result RegisterDebugResponse(const ::nn::nim::ShopServiceAccessTypes::Server& inServer, const ::nn::sf::InArray<char>& inKeyPath, const ::nn::sf::InArray<char>& inValueResponse, ::std::uint32_t inExpectResult, ::std::uint32_t inHappenedRate) NN_NOEXCEPT
    {
        std::lock_guard<decltype(m_ReferLock)> guard(m_ReferLock);
        NN_SDK_ASSERT(static_cast<bool>(m_ProxyImpl), "Has not been initialized the features.");
        return m_ProxyImpl->RegisterDebugResponse(inServer, inKeyPath, inValueResponse, inExpectResult, inHappenedRate);
    }

private:
    TMutex          m_ReferLock;
    TAllocator      m_Allocator;
    TServiceProxy   m_ProxyImpl;
    int             m_RefCounter;
};

//-----------------------------------------------------------------------------
Result ShopServiceAccessDebugProxy::Initialize() NN_NOEXCEPT
{
    std::lock_guard<decltype(m_ReferLock)> guard(m_ReferLock);
    const auto count = m_RefCounter + 1;
    if (1 == count)
    {
        // サービスインタフェース取得
        NN_RESULT_DO(::nn::sf::CreateHipcProxyByName<IService>(
            &m_ProxyImpl
            , m_Allocator.GetMemoryResource()
            , ::nn::nim::detail::ServiceNameShopServiceAccessor
            ));
    }
    m_RefCounter = count;
    NN_RESULT_SUCCESS;
}

//-----------------------------------------------------------------------------
void ShopServiceAccessDebugProxy::Finalize() NN_NOEXCEPT
{
    std::lock_guard<decltype(m_ReferLock)> guard(m_ReferLock);
    const auto count = m_RefCounter - 1;
    if (0 == count)
    {
        m_RefCounter = 0;
        NotifyFinalizeToServer();
    }
    else if (0 < count)
    {
        m_RefCounter = count;
    }
}

//-----------------------------------------------------------------------------
// key=value 形式のデータとして用いることができる文字かどうかを返します。
bool IsValidCharForPathAndQuery(const char ch) NN_NOEXCEPT
{
    // 参考: RFC 3986:
    //   query         = *( pchar / "/" / "?" )
    //   pchar         = unreserved / pct-encoded / sub-delims / ":" / "@"
    //   unreserved    = ALPHA / DIGIT / "-" / "." / "_" / "~"
    //   pct-encoded   = "%" HEXDIG HEXDIG
    //   sub-delims    = "!" / "$" / "&" / "'" / "(" / ")"
    //                 / "*" / "+" / "," / ";" / "="
    // ※ RFC 2234:
    //   ALPHA         = %x41-5A / %x61-7A   ; A-Z / a-z
    //   DIGIT         = %x30-39             ; 0-9
    //   HEXDIG        =  DIGIT / "A" / "B" / "C" / "D" / "E" / "F"
    return
        // unreserved
        (ch >= 'A' && ch <= 'Z') || (ch >= 'a' && ch <= 'z') || (ch >= '0' && ch <= '9')
        || ch == '.' || ch == '-' || ch == '_' || ch == '~'
        // pct-encoded ( 「"%" HEXDIG HEXDIG」という書式までは簡潔化のためチェックしない )
        || ch == '%'
        // sub-delims
        || ch == '!' || ch == '$' || ch == '\'' || ch == '(' || ch == ')'
        || ch == '*' || ch == '+' || ch == ',' || ch == ';' || ch == '&' || ch == '='
        // pchar
        || ch == ':' || ch == '@'
        // query
        || ch == '/' || ch == '?'
        // for replace alias symbol.
        || ch == '{' || ch == '}'
        ;
}

//-----------------------------------------------------------------------------
}   // ~::unnamed
//-----------------------------------------------------------------------------


//-----------------------------------------------------------------------------
DebugForShopServiceAccessor::ScopedSession::ScopedSession() NN_NOEXCEPT
{
}

//-----------------------------------------------------------------------------
DebugForShopServiceAccessor::ScopedSession::~ScopedSession() NN_NOEXCEPT
{
    ShopServiceAccessDebugProxy::GetInstance().Finalize();
}

//-----------------------------------------------------------------------------
Result DebugForShopServiceAccessor::Initialize(ScopedSession* pOutSession) NN_NOEXCEPT
{
    NN_UNUSED(pOutSession);
    NN_SDK_REQUIRES_NOT_NULL(pOutSession);
    return ShopServiceAccessDebugProxy::GetInstance().Initialize();
}

//-----------------------------------------------------------------------------
void DebugForShopServiceAccessor::RefreshAvailability() NN_NOEXCEPT
{
    ShopServiceAccessDebugProxy::GetInstance().RefreshDebugAvailability();
}

//-----------------------------------------------------------------------------
Result DebugForShopServiceAccessor::ClearResponse() NN_NOEXCEPT
{
    return ShopServiceAccessDebugProxy::GetInstance().ClearDebugResponse();
}

//-----------------------------------------------------------------------------
Result DebugForShopServiceAccessor::RegisterResponse(const ShopService& keyTarget, const char* pKeyPath, const char *pValue, const size_t valueSize, const Result& expectResult, const uint32_t happenedRate) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(0 <= static_cast<size_t>(keyTarget.type) && static_cast<size_t>(keyTarget.type) < ShopService::Type_EndOfConstants);
    NN_SDK_REQUIRES_NOT_NULL(pKeyPath);

    // URLパスフィールド中の無効文字検出.
    const auto pathLength = std::strlen(pKeyPath);
    const auto pPathEnd = &pKeyPath[pathLength];
    NN_RESULT_THROW_UNLESS(pPathEnd == std::find_if_not(pKeyPath, pPathEnd, IsValidCharForPathAndQuery), ResultShopServiceAccessInvalidCharacter());

    return ShopServiceAccessDebugProxy::GetInstance().RegisterDebugResponse(::nn::nim::ShopServiceAccessTypes::CreateServerFrom(keyTarget.type)
        , ::nn::sf::InArray<char>(pKeyPath, pathLength)
        , (nullptr == pValue || 0 == valueSize) ? ::nn::sf::InArray<char>() : ::nn::sf::InArray<char>(pValue, valueSize)
        , expectResult.GetInnerValueForDebug()
        , happenedRate
    );
}

}}
