﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <mutex>
#include <limits>
#include <string>
#include <type_traits>
#include <nn/ec/ec_Result.h>
#include <nn/ec/ec_ResultPrivate.h>
#include <nn/ec/ec_ShopServiceAccessor.h>
#include <nn/os/os_Mutex.h>
#include <nn/os/os_TransferMemory.h>
#include <nn/sf/sf_ProxyObjectAllocator.h>
#include <nn/sf/sf_NativeHandle.h>
#include <nn/sf/sf_HipcClient.h>
#include <nn/sf/sf_ObjectFactory.h>
#include <nn/util/util_StringUtil.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/nifm/nifm_ApiClientManagement.h>
#include <nn/nim/nim_Result.h>
#include <nn/nim/nim_ShopServiceAccessTypes.h>
#include <nn/nim/nim_ShopServiceAccessConfig.h>
#include <nn/nim/detail/nim_ServiceName.h>
#include <nn/nim/detail/nim_IShopServiceAccessServer.sfdl.h>

namespace nn { namespace ec {

namespace {

/**
 * @brief   サービスサーバーIPCプロキシ。
 */
class ShopServiceAccessServerProxy
{
    NN_DISALLOW_COPY(ShopServiceAccessServerProxy);
    NN_DISALLOW_MOVE(ShopServiceAccessServerProxy);

    /**
     * @brief   維持想定セッション数。
     * @note    IShopServiceAccessServerInterface と IShopServiceAccessServer 用セッションは１つとして考えます。
     *          IShopServiceAccessServerInterface から IShopServiceAccessServer を取得後、即破棄することで除外可能。
     */
    static const int IpcSessionLimit = 1
        + ::nn::nim::ShopServiceAccessConfiguration::AvailableAccessorMax
        + ::nn::nim::ShopServiceAccessConfiguration::AvailableAsyncAccessMax;

public:
    typedef ::nn::nim::detail::IShopServiceAccessServerInterface    IService;
    typedef ::nn::nim::detail::IShopServiceAccessServer             IServer;
    typedef ::nn::nim::detail::IShopServiceAccessor                 IAccessor;
    typedef ::nn::sf::ProxyObjectAllocator<IpcSessionLimit>         TAllocator;
    typedef ::nn::sf::SharedPointer<IServer>                        TServerProxy;
    typedef ::nn::os::TransferMemoryType                            TTransferMemory;

    static ShopServiceAccessServerProxy& GetInstance() NN_NOEXCEPT
    {
        NN_FUNCTION_LOCAL_STATIC(ShopServiceAccessServerProxy, s_ProxyImpl);
        return s_ProxyImpl;
    }

    Result Initialize(void* pTransferMemoryTop, size_t byteSizeOfTotal) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pTransferMemoryTop);
        NN_SDK_REQUIRES(0 == (byteSizeOfTotal % ::nn::os::MemoryPageSize));

        std::lock_guard<decltype(m_ReferLock)> guard(m_ReferLock);
        NN_SDK_REQUIRES(!m_Initialized);

        // サービスインタフェース取得
        ::nn::sf::SharedPointer<IService> iService;
        NN_RESULT_DO(::nn::sf::CreateHipcProxyByName<IService>(&iService
            , m_Allocator.GetMemoryResource()
            , ::nn::nim::detail::ServiceNameShopServiceAccessor
        ));

        // トランスファーメモリ作成
        NN_RESULT_DO(::nn::os::CreateTransferMemory(&m_TransferHandle, pTransferMemoryTop, byteSizeOfTotal, os::MemoryPermission_None));

        // サーバーインタフェース生成要求
        ::nn::sf::NativeHandle transferHandle(::nn::os::DetachTransferMemory(&m_TransferHandle), true);
        NN_RESULT_TRY(iService.Get()->CreateServerInterface(&m_ProxyImpl, 0, std::move(transferHandle), byteSizeOfTotal))
        NN_RESULT_CATCH_ALL
        {
            ::nn::os::DestroyTransferMemory(&m_TransferHandle);
            NN_RESULT_RETHROW;
        }
        NN_RESULT_END_TRY;

        m_Initialized = true;
        NN_RESULT_SUCCESS;
    }

    Result Finalize() NN_NOEXCEPT
    {
        std::lock_guard<decltype(m_ReferLock)> guard(m_ReferLock);
        if (!m_Initialized)
        {
            NN_RESULT_SUCCESS;
        }

        // 発行済 IShopServiceAccessor インタフェースの回収。
        auto& list = m_ListAccessor;
        while (NN_STATIC_CONDITION(true))
        {
            auto it = list.begin();
            if (it == list.end())
            {
                break;
            }
            it->Finalize();
        }

        // サーバープロセスセッション切断。
        m_ProxyImpl = nullptr;

        // トランスファーメモリハンドル解放。
        ::nn::os::DestroyTransferMemory(&m_TransferHandle);
        m_Initialized = false;
        NN_RESULT_SUCCESS;
    }

    Result CreateAccessorInterface(::nn::sf::SharedPointer<IAccessor>* outValue, const ::nn::nim::ShopServiceAccessTypes::Server& target) NN_NOEXCEPT
    {
        std::lock_guard<decltype(m_ReferLock)> guard(m_ReferLock);
        return AcquireHipcProxy().CreateAccessorInterface(outValue, target);
    }

    void Attach(ShopServiceAccessor* pAccessor) NN_NOEXCEPT
    {
        std::lock_guard<decltype(m_ReferLock)> guard(m_ReferLock);
        NN_SDK_REQUIRES_NOT_NULL(pAccessor);
        NN_SDK_REQUIRES(false == pAccessor->IsLinked());
        m_ListAccessor.push_back(*pAccessor);
    }

    void Detach(ShopServiceAccessor* pAccessor) NN_NOEXCEPT
    {
        std::lock_guard<decltype(m_ReferLock)> guard(m_ReferLock);
        NN_SDK_REQUIRES_NOT_NULL(pAccessor);

        if (pAccessor->IsLinked())
        {
            auto& list = m_ListAccessor;
            list.erase(list.iterator_to(*pAccessor));
        }
    }

private:
    typedef ::nn::util::IntrusiveList<ShopServiceAccessor, ::nn::util::IntrusiveListBaseNodeTraits<ShopServiceAccessor>> TAccessorList;

    NN_IMPLICIT ShopServiceAccessServerProxy() NN_NOEXCEPT
        : m_ReferLock(true), m_Initialized(false)
    {
        m_Allocator = NN_SF_PROXY_OBJECT_ALLOCATOR_INITIALIZER;
        m_Allocator.Initialize();
    }

    ~ShopServiceAccessServerProxy() NN_NOEXCEPT
    {
        Finalize();
        m_Allocator.Finalize();
    }

    IServer& AcquireHipcProxy() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(m_Initialized);
        return *m_ProxyImpl.Get();
    }

    ::nn::os::Mutex m_ReferLock;
    TAllocator      m_Allocator;
    TServerProxy    m_ProxyImpl;
    TTransferMemory m_TransferHandle;
    TAccessorList   m_ListAccessor;
    bool            m_Initialized;
};

//-----------------------------------------------------------------------------
// key=value 形式のデータとして用いることができる文字かどうかを返します。
bool IsValidCharForPathAndQuery(const char ch) NN_NOEXCEPT
{
    // 参考: RFC 3986:
    //   query         = *( pchar / "/" / "?" )
    //   pchar         = unreserved / pct-encoded / sub-delims / ":" / "@"
    //   unreserved    = ALPHA / DIGIT / "-" / "." / "_" / "~"
    //   pct-encoded   = "%" HEXDIG HEXDIG
    //   sub-delims    = "!" / "$" / "&" / "'" / "(" / ")"
    //                 / "*" / "+" / "," / ";" / "="
    // ※ RFC 2234:
    //   ALPHA         = %x41-5A / %x61-7A   ; A-Z / a-z
    //   DIGIT         = %x30-39             ; 0-9
    //   HEXDIG        =  DIGIT / "A" / "B" / "C" / "D" / "E" / "F"
    return
        // unreserved
        (ch >= 'A' && ch <= 'Z') || (ch >= 'a' && ch <= 'z') || (ch >= '0' && ch <= '9')
        || ch == '.' || ch == '-' || ch == '_' || ch == '~'
        // pct-encoded ( 「"%" HEXDIG HEXDIG」という書式までは簡潔化のためチェックしない )
        || ch == '%'
        // sub-delims
        || ch == '!' || ch == '$' || ch == '\'' || ch == '(' || ch == ')'
        || ch == '*' || ch == '+' || ch == ',' || ch == ';' || ch == '&' || ch == '='
        // pchar
        || ch == ':' || ch == '@'
        // query
        || ch == '/' || ch == '?'
        // for replace alias symbol.
        || ch == '{' || ch == '}'
        ;
}

//-----------------------------------------------------------------------------
Result ToServiceSpecificationResultAsSugar(const ::nn::err::ErrorCodeNumber code) NN_NOEXCEPT
{
    switch (code)
    {
    case 1542:
        NN_RESULT_THROW(ResultOwnedConsumableServiceItemConsumedRights());
    default:
        NN_RESULT_THROW(ResultShowErrorCodeRequired());
    }
}

//-----------------------------------------------------------------------------
Result ToServiceSpecificationResult(const ::nn::err::ErrorCode& code) NN_NOEXCEPT
{
    switch (code.category)
    {
    case 2309:  // Sugar
        return ToServiceSpecificationResultAsSugar(code.number);
    default:
        NN_RESULT_THROW(ResultShowErrorCodeRequired());
    }
}

//-----------------------------------------------------------------------------
Result ToHttpResult(const Result result) NN_NOEXCEPT
{
    NN_RESULT_TRY(result)

        NN_RESULT_CATCH_CONVERT(::nn::nim::ResultHttpStatus300MultipleChoices, ResultHttpStatus300MultipleChoices())

        NN_RESULT_CATCH_CONVERT(::nn::nim::ResultHttpStatus301MovedPermanently, ResultHttpStatus301MovedPermanently())
        NN_RESULT_CATCH_CONVERT(::nn::nim::ResultHttpStatus302Found, ResultHttpStatus302Found())
        NN_RESULT_CATCH_CONVERT(::nn::nim::ResultHttpStatus303SeeOther, ResultHttpStatus303SeeOther())
        NN_RESULT_CATCH_CONVERT(::nn::nim::ResultHttpStatus304NotModified, ResultHttpStatus304NotModified())
        NN_RESULT_CATCH_CONVERT(::nn::nim::ResultHttpStatus305UseProxy, ResultHttpStatus305UseProxy())
        NN_RESULT_CATCH_CONVERT(::nn::nim::ResultHttpStatus306SwitchProxy, ResultHttpStatus306SwitchProxy())
        NN_RESULT_CATCH_CONVERT(::nn::nim::ResultHttpStatus307TemporaryRedirect, ResultHttpStatus307TemporaryRedirect())
        NN_RESULT_CATCH_CONVERT(::nn::nim::ResultHttpStatus308PermanentRedirect, ResultHttpStatus308PermanentRedirect())

        NN_RESULT_CATCH_CONVERT(::nn::nim::ResultHttpStatus400BadRequest, ResultHttpStatus400BadRequest())
        NN_RESULT_CATCH_CONVERT(::nn::nim::ResultHttpStatus401Unauthorized, ResultHttpStatus401Unauthorized())
        NN_RESULT_CATCH_CONVERT(::nn::nim::ResultHttpStatus403Forbidden, ResultHttpStatus403Forbidden())
        NN_RESULT_CATCH_CONVERT(::nn::nim::ResultHttpStatus404NotFound, ResultHttpStatus404NotFound())
        NN_RESULT_CATCH_CONVERT(::nn::nim::ResultHttpStatus405MethodNotAllowed, ResultHttpStatus405MethodNotAllowed())
        NN_RESULT_CATCH_CONVERT(::nn::nim::ResultHttpStatus406NotAcceptable, ResultHttpStatus406NotAcceptable())
        NN_RESULT_CATCH_CONVERT(::nn::nim::ResultHttpStatus407ProxyAuthenticationRequired, ResultHttpStatus407ProxyAuthenticationRequired())
        NN_RESULT_CATCH_CONVERT(::nn::nim::ResultHttpStatus408RequestTimeout, ResultHttpStatus408RequestTimeout())
        NN_RESULT_CATCH_CONVERT(::nn::nim::ResultHttpStatus409Conflict, ResultHttpStatus409Conflict())
        NN_RESULT_CATCH_CONVERT(::nn::nim::ResultHttpStatus410Gone, ResultHttpStatus410Gone())
        NN_RESULT_CATCH_CONVERT(::nn::nim::ResultHttpStatus411LengthRequired, ResultHttpStatus411LengthRequired())
        NN_RESULT_CATCH_CONVERT(::nn::nim::ResultHttpStatus412PreconditionFailed, ResultHttpStatus412PreconditionFailed())
        NN_RESULT_CATCH_CONVERT(::nn::nim::ResultHttpStatus413PayloadTooLarge, ResultHttpStatus413PayloadTooLarge())
        NN_RESULT_CATCH_CONVERT(::nn::nim::ResultHttpStatus414UriTooLong, ResultHttpStatus414UriTooLong())
        NN_RESULT_CATCH_CONVERT(::nn::nim::ResultHttpStatus415UnsupportedMediaType, ResultHttpStatus415UnsupportedMediaType())
        NN_RESULT_CATCH_CONVERT(::nn::nim::ResultHttpStatus416RangeNotSatisfiable, ResultHttpStatus416RangeNotSatisfiable())
        NN_RESULT_CATCH_CONVERT(::nn::nim::ResultHttpStatus417ExpectationFailed, ResultHttpStatus417ExpectationFailed())

        NN_RESULT_CATCH_CONVERT(::nn::nim::ResultHttpStatus500InternalServerError, ResultHttpStatus500InternalServerError())
        NN_RESULT_CATCH_CONVERT(::nn::nim::ResultHttpStatus501NotImplemented, ResultHttpStatus501NotImplemented())
        NN_RESULT_CATCH_CONVERT(::nn::nim::ResultHttpStatus502BadGateway, ResultHttpStatus502BadGateway())
        NN_RESULT_CATCH_CONVERT(::nn::nim::ResultHttpStatus503ServiceUnavailable, ResultHttpStatus503ServiceUnavailable())
        NN_RESULT_CATCH_CONVERT(::nn::nim::ResultHttpStatus504GatewayTimeout, ResultHttpStatus504GatewayTimeout())
        NN_RESULT_CATCH_CONVERT(::nn::nim::ResultHttpStatus505HttpVersionNotSupported, ResultHttpStatus505HttpVersionNotSupported())
        NN_RESULT_CATCH_CONVERT(::nn::nim::ResultHttpStatus509BandwidthLimitExceeded, ResultHttpStatus509BandwidthLimitExceeded())

        NN_RESULT_CATCH_ALL
        {
            NN_RESULT_MARK_AS_RETHROW;
            NN_RESULT_THROW(ResultHttpStatusUnexpected());
        }

    NN_RESULT_END_TRY;
    NN_RESULT_SUCCESS;
}

//-----------------------------------------------------------------------------
Result ToResult(const Result result) NN_NOEXCEPT
{
    NN_RESULT_TRY(result)

        // 不正値をSDKが提供する列挙型などにキャストして要求した際の Result.
        NN_RESULT_CATCH(::nn::nim::ResultNotSupported)
        {
            NN_ABORT("Invalid operation request.\n");
        }

        // Program errors
        NN_RESULT_CATCH_CONVERT(::nn::nim::ResultBufferNotEnough, ResultShopServiceAccessInsufficientBuffer())
        NN_RESULT_CATCH_CONVERT(::nn::nim::ResultShopServiceAccessInvalidCharacter, ResultShopServiceAccessInvalidCharacter())
        NN_RESULT_CATCH_CONVERT(::nn::nim::ResultShopServiceAccessInsufficientWorkMemory, ResultShopServiceAccessInsufficientWorkMemory())

        NN_RESULT_CATCH_CONVERT(::nn::nim::ResultNetworkTimeUnavailable, ResultNetworkTimeUnavailable())
        NN_RESULT_CATCH_CONVERT(::nn::nim::ResultAgeRestriction, ResultAgeRestriction())

        NN_RESULT_CATCH_CONVERT(::nn::nim::ResultOutOfMaxTask, ResultShopServiceAccessOverRequest())
        NN_RESULT_CATCH_CONVERT(::nn::sf::ResultMemoryAllocationFailed, ResultShopServiceAccessOverRequest())
        NN_RESULT_CATCH_CONVERT(::nn::nim::ResultAllocationMemoryFailed, ResultShopServiceAccessOutOfResource())

        NN_RESULT_CATCH_CONVERT(::nn::nim::ResultHttpConnectionCanceled, ResultShopServiceAccessCanceled())
        NN_RESULT_CATCH_CONVERT(::nn::nim::ResultShopServiceAccessCanceled, ResultShopServiceAccessCanceled())

        NN_RESULT_CATCH_CONVERT(::nn::nim::ResultHttpConnectionTimeout, ResultShopServiceAccessRequestTimeout())

        // A http status code.
        NN_RESULT_CATCH(::nn::nim::ResultHttpStatus)
        {
            NN_RESULT_MARK_AS_RETHROW;
            NN_RESULT_THROW(ToHttpResult(NN_RESULT_CURRENT_RESULT));
        }

    NN_RESULT_END_TRY;
    NN_RESULT_SUCCESS;
}
#define NN_EC_RESULT_DO(result_)    NN_RESULT_DO(ToResult(result_))
#define NN_EC_RESULT_TRY(result_)   NN_RESULT_TRY(ToResult(result_))

//-----------------------------------------------------------------------------
template<typename T>
void Emplace(T& object) NN_NOEXCEPT
{
    object.~T();
    new (&object) T();
}

}   // ~::unnamed

//-----------------------------------------------------------------------------
Result InitializeForShopServiceAccessors(void* pTransferMemoryTop, size_t byteSizeOfTotal) NN_NOEXCEPT
{
    NN_EC_RESULT_DO(ShopServiceAccessServerProxy::GetInstance().Initialize(pTransferMemoryTop, byteSizeOfTotal));
    NN_RESULT_SUCCESS;
}

//-----------------------------------------------------------------------------
Result FinalizeForShopServiceAccessors() NN_NOEXCEPT
{
    NN_EC_RESULT_DO(ShopServiceAccessServerProxy::GetInstance().Finalize());
    NN_RESULT_SUCCESS;
}

//-----------------------------------------------------------------------------
const ::nn::TimeSpan ShopServiceAccessor::DefaultTimeout = ::nn::TimeSpan::FromSeconds(60);

//-----------------------------------------------------------------------------
ShopServiceAccessor::ShopServiceAccessor() NN_NOEXCEPT
    : m_pImpl(nullptr)
    , m_Lock(true)
{
}

//-----------------------------------------------------------------------------
ShopServiceAccessor::~ShopServiceAccessor() NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(Finalize());
}

//-----------------------------------------------------------------------------
Result ShopServiceAccessor::Initialize(const ShopService& target) NN_NOEXCEPT
{
    NN_SDK_ASSERT(0 <= static_cast<size_t>(target.type) && static_cast<size_t>(target.type) < ShopService::Type_EndOfConstants);
    {
        std::lock_guard<decltype(m_Lock)> guard(m_Lock);
        NN_SDK_REQUIRES(nullptr == m_pImpl);

        ::nn::sf::SharedPointer<IAccessor> handle;
        const auto ipcServer = ::nn::nim::ShopServiceAccessTypes::CreateServerFrom(target.type);
        NN_EC_RESULT_DO(ShopServiceAccessServerProxy::GetInstance().CreateAccessorInterface(&handle, ipcServer));
        m_pImpl = handle.Detach();
    }
    ShopServiceAccessServerProxy::GetInstance().Attach(this);
    NN_RESULT_SUCCESS;
}

//-----------------------------------------------------------------------------
Result ShopServiceAccessor::Finalize() NN_NOEXCEPT
{
    {
        std::lock_guard<decltype(m_Lock)> guard(m_Lock);
        const auto pImpl = m_pImpl;
        if (nullptr == pImpl)
        {
            NN_RESULT_SUCCESS;
        }

        // 未完了の非同期要求の完了待機
        TAsyncList& list = m_ListChild;
        while (NN_STATIC_CONDITION(true))
        {
            auto it = list.begin();
            if (it == list.end())
            {
                break;
            }
            it->Finalize();
        }

        // サーバープロセスへ終了通知 ( サーバーの依存リソースが破棄 )
        ::nn::sf::ReleaseSharedObject(pImpl);
        m_pImpl = nullptr;
    }
    // クライアントプロセスのオーナーへ解放完了通知
    ShopServiceAccessServerProxy::GetInstance().Detach(this);
    NN_RESULT_SUCCESS;
}

//-----------------------------------------------------------------------------
void ShopServiceAccessor::Attach(AsyncResponse* pAsyncResponse) NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Lock)> guard(m_Lock);
    NN_SDK_REQUIRES_NOT_NULL(pAsyncResponse);

    m_ListChild.push_back(*pAsyncResponse);
}

//-----------------------------------------------------------------------------
void ShopServiceAccessor::Detach(AsyncResponse* pAsyncResponse) NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Lock)> guard(m_Lock);
    NN_SDK_REQUIRES_NOT_NULL(pAsyncResponse);

    if (pAsyncResponse->IsLinked())
    {
        auto& list = m_ListChild;
        list.erase(list.iterator_to(*pAsyncResponse));
    }
}

//-----------------------------------------------------------------------------
Result ShopServiceAccessor::Request(AsyncResponse* pOutHandle
    , const ::nn::account::Uid& userId
    , const ShopService::Method method
    , const char* pPath
    , const PostData& postData
    , const ::nn::TimeSpan timeout
) NN_NOEXCEPT
{
    NN_SDK_ASSERT(0 <= static_cast<size_t>(method) && static_cast<size_t>(method) < ShopService::Method_EndOfConstants);
    NN_SDK_REQUIRES_NOT_NULL(pOutHandle);
    NN_SDK_REQUIRES_NOT_NULL(pPath);
    NN_SDK_REQUIRES(*pPath == '/');
    NN_SDK_REQUIRES(userId);

    // ユーザのインターネット接続要求提出受理済確認.
    NN_RESULT_THROW_UNLESS(::nn::nifm::IsAnyInternetRequestAccepted(::nn::nifm::GetClientId()), ResultInternetRequestNotAccepted());

    // URLパスフィールド中の無効文字検出.
    const auto pathLength = std::strlen(pPath);
    const auto pPathEnd = &pPath[pathLength];
    NN_RESULT_THROW_UNLESS(pPathEnd == std::find_if_not(pPath, pPathEnd, IsValidCharForPathAndQuery), ResultShopServiceAccessInvalidCharacter());

    // 要求.
    std::lock_guard<decltype(m_Lock)> guard(m_Lock);

    const auto pImpl = m_pImpl;
    NN_SDK_REQUIRES_NOT_NULL(pImpl);

    // タイムアウト丸め.
    const auto t = timeout.GetSeconds();
    const ::nn::TimeSpanType actualTimeout = (t < 1) ? static_cast<::nn::TimeSpanType>(DefaultTimeout)
        : ((t > INT_MAX) ? ::nn::TimeSpanType::FromSeconds(INT_MAX) : static_cast<::nn::TimeSpanType>(timeout));

    // 固定長パラメタは纏めて. ( 未使用は 0 埋め )
    ::nn::nim::ShopServiceAccessTypes::FixedParams fixed =
    {
        userId,
        actualTimeout,
        static_cast<::nn::nim::ShopServiceAccessTypes::Method>(method),
        0, 0, 0
    };

    // 非同期ハンドル生成要求.
    ::nn::sf::NativeHandle eventHandle;
    ::nn::sf::SharedPointer<AsyncResponse::IAsyncHandle> asyncHandle;
    NN_EC_RESULT_DO(pImpl->CreateAsyncInterface(&eventHandle, &asyncHandle, fixed));

    // パラメータ登録要求.
    const auto inPost = (false == postData) ? ::nn::sf::InArray<char>() : ::nn::sf::InArray<char>(postData.pBody, postData.size);
    NN_EC_RESULT_DO(asyncHandle->Prepare(::nn::sf::InArray<char>(pPath, pathLength), inPost));

    // 要求実施.
    NN_EC_RESULT_DO(asyncHandle->Request());

    // 成功したらユーザに譲渡.
    pOutHandle->Attach(this, asyncHandle.Detach(), eventHandle);
    eventHandle.Detach();

    NN_RESULT_SUCCESS;
}

//-----------------------------------------------------------------------------
bool ShopServiceAccessor::IsAvailable() const NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Lock)> guard(m_Lock);
    return (nullptr != m_pImpl);
}

//-----------------------------------------------------------------------------
ShopServiceAccessor::AsyncResponse::AsyncResponse() NN_NOEXCEPT
    : m_pOwner(nullptr)
    , m_pAsync(nullptr)
{
}

//-----------------------------------------------------------------------------
ShopServiceAccessor::AsyncResponse::~AsyncResponse() NN_NOEXCEPT
{
    Finalize();
}

//-----------------------------------------------------------------------------
void ShopServiceAccessor::AsyncResponse::Finalize() NN_NOEXCEPT
{
    Detach();
}

//-----------------------------------------------------------------------------
void ShopServiceAccessor::AsyncResponse::Attach(ShopServiceAccessor* const pOwner, IAsyncHandle* pAsyncHandle, const ::nn::sf::NativeHandle& notifyEventHandle) NN_NOEXCEPT
{
    Detach();
    m_Event.AttachReadableHandle(notifyEventHandle.GetOsHandle(), notifyEventHandle.IsManaged(), ::nn::os::EventClearMode_ManualClear);
    m_pAsync = pAsyncHandle;
    pOwner->Attach(this);
    m_pOwner = pOwner;
}

//-----------------------------------------------------------------------------
void ShopServiceAccessor::AsyncResponse::Detach() NN_NOEXCEPT
{
    const auto pAsyncHandle = m_pAsync;
    if (nullptr != pAsyncHandle)
    {
        pAsyncHandle->Cancel();
        m_Event.Wait();
        Emplace(m_Event);
        ::nn::sf::ReleaseSharedObject(pAsyncHandle);
        m_pAsync = nullptr;
    }
    auto pOwner = m_pOwner;
    if (nullptr != pOwner)
    {
        pOwner->Detach(this);
    }
    m_pOwner = nullptr;
}

//-----------------------------------------------------------------------------
Result ShopServiceAccessor::AsyncResponse::GetSizeImpl(size_t* const pOutValue) NN_NOEXCEPT
{
    const auto pAsyncHandle = m_pAsync;
    NN_RESULT_THROW_UNLESS(nullptr != pAsyncHandle, ResultShopServiceAccessCanceled());

    uint64_t outValue = 0;
    NN_EC_RESULT_TRY(pAsyncHandle->GetSize(&outValue))
    NN_RESULT_CATCH(::nn::nim::ResultShopServiceAccessServiceSpecificError)
    {
        // A service original specification error.
        *pOutValue = 0;
        NN_RESULT_MARK_AS_RETHROW;
        NN_RESULT_THROW(ToServiceSpecificationResult(GetErrorCode()));
    }
    NN_RESULT_CATCH_ALL
    {
        *pOutValue = 0;
        NN_RESULT_RETHROW;
    }
    NN_RESULT_END_TRY;

    *pOutValue = static_cast<size_t>(outValue);
    NN_RESULT_SUCCESS;
}

//-----------------------------------------------------------------------------
Result ShopServiceAccessor::AsyncResponse::GetSize(size_t* const pOutValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    Wait();

    NN_RESULT_DO(GetSizeImpl(pOutValue));
    NN_RESULT_SUCCESS;
}

//-----------------------------------------------------------------------------
Result ShopServiceAccessor::AsyncResponse::GetImpl(void* const pOutReceivedBuffer, const size_t bufferCapacity) NN_NOEXCEPT
{
    const auto pAsyncHandle = m_pAsync;
    NN_RESULT_THROW_UNLESS(nullptr != pAsyncHandle, ResultShopServiceAccessCanceled());

    uint64_t outSize;
    NN_EC_RESULT_TRY(pAsyncHandle->Read(&outSize, 0, ::nn::sf::OutBuffer(static_cast<char*>(pOutReceivedBuffer), bufferCapacity)))
    NN_RESULT_CATCH(::nn::nim::ResultShopServiceAccessServiceSpecificError)
    {
        // A service original specification error.
        NN_RESULT_MARK_AS_RETHROW;
        NN_RESULT_THROW(ToServiceSpecificationResult(GetErrorCode()));
    }
    NN_RESULT_END_TRY;
    NN_RESULT_SUCCESS;
}

//-----------------------------------------------------------------------------
Result ShopServiceAccessor::AsyncResponse::Get(void* const pOutReceivedBuffer, const size_t bufferCapacity) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutReceivedBuffer);

    Wait();

    size_t responseSize = 0;
    NN_RESULT_DO(GetSizeImpl(&responseSize));
    if (0 >= responseSize)
    {
        NN_RESULT_SUCCESS;
    }
    NN_RESULT_THROW_UNLESS(bufferCapacity >= responseSize, ResultShopServiceAccessInsufficientBuffer());
    NN_RESULT_DO(GetImpl(pOutReceivedBuffer, bufferCapacity));
    NN_RESULT_SUCCESS;
}

//-----------------------------------------------------------------------------
void ShopServiceAccessor::AsyncResponse::Wait() NN_NOEXCEPT
{
    if (nullptr != m_pAsync)
    {
        m_Event.Wait();
    }
}

//-----------------------------------------------------------------------------
bool ShopServiceAccessor::AsyncResponse::TryWait() NN_NOEXCEPT
{
    return (nullptr != m_pAsync) ? m_Event.TryWait() : true;
}

//-----------------------------------------------------------------------------
void ShopServiceAccessor::AsyncResponse::Cancel() NN_NOEXCEPT
{
    const auto pAsyncHandle = m_pAsync;
    if (nullptr != pAsyncHandle)
    {
        pAsyncHandle->Cancel();
    }
}

//-----------------------------------------------------------------------------
::nn::os::SystemEvent& ShopServiceAccessor::AsyncResponse::GetEvent() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pAsync);
    return m_Event;
}

//-----------------------------------------------------------------------------
::nn::err::ErrorCode ShopServiceAccessor::AsyncResponse::GetErrorCode() const NN_NOEXCEPT
{
    ::nn::err::ErrorCode code;
    const auto pAsyncHandle = m_pAsync;
    return (nullptr != pAsyncHandle && pAsyncHandle->GetErrorCode(&code).IsSuccess()) ? code : ::nn::err::ErrorCode::GetInvalidErrorCode();
}

}}
