﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief 消費型サービスアイテム・クライアントライブラリが使用するアロケートバッファ
 */

#pragma once
#include <nn/mem.h>
#include <nn/nn_Result.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/ec/ec_Result.h>

namespace nn { namespace ec {

template <typename T> class AllocatedBuffer
{
    NN_DISALLOW_COPY(AllocatedBuffer);
    NN_DISALLOW_MOVE(AllocatedBuffer);

public:
    explicit AllocatedBuffer(nn::mem::StandardAllocator* pAllocator) NN_NOEXCEPT
        : m_pAllocator(pAllocator)
        , m_Pointer(nullptr)
        , m_Count(0)
        , m_BufferSize(0u)
    {
    }

    ~AllocatedBuffer() NN_NOEXCEPT
    {
        if (m_Pointer != nullptr)
        {
            m_pAllocator->Free(m_Pointer);
        }
    }

    Result Allocate(int count) NN_NOEXCEPT
    {
        m_Count = count;
        m_BufferSize = sizeof(T) * static_cast<size_t>(count);
        m_Pointer = m_pAllocator->Allocate(m_BufferSize);

        //! メモリが確保できない場合、エラーを返す
        NN_RESULT_THROW_UNLESS(m_Pointer != nullptr, nn::ec::ResultInsufficientWorkMemory());
        NN_RESULT_SUCCESS;
    };

    T* Get() NN_NOEXCEPT
    {
        return reinterpret_cast<T*>(m_Pointer);
    }

    int GetCount() NN_NOEXCEPT
    {
        return m_Count;
    }

private:
    nn::mem::StandardAllocator* m_pAllocator;
    void* m_Pointer;
    int m_Count;

    //! デアロケーション時にメモリ操作する可能性を残して、確保したバッファサイズは保持しておく
    //! アプリケーションは触れない
    size_t m_BufferSize;
};

}}
