﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/ec/detail/ec_ShopApiInternal.h>

#include <nn/nn_Abort.h>
#include <nn/nn_SdkAssert.h>
#include <nn/ec/ec_ResultShop.h>
#include <nn/ec/detail/ec_ShowShopPageApi.h>
#include <nn/ec/detail/ec_ShowShopPageTypes.h>
#include <nn/ec/detail/ec_TypesInternal.h>
#include <nn/ec/detail/ec_Utils.h>
#include <nn/la/la_Result.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_StringUtil.h>

namespace nn { namespace ec { namespace detail {

namespace
{
    static const int MaxNewsIdLength = 127;

    ShopPageReturnValue g_ReturnValue;

    nn::Result ShopReturnValueToResult(const ShopPageReturnValue& returnValue) NN_NOEXCEPT
    {
        // success を表す戻り値返したときのみ ResultSuccess として
        // それ以外を nn::ec::ResultShopCanceled とする
        if (returnValue.GetShopExitReason() != ShopExitReason_CallbackUrlReached ||
            returnValue.GetLastUrlSize() != ShopSuccessLastUrlSize ||
            std::strncmp(returnValue.GetLastUrl(), ShopSuccessLastUrl, ShopSuccessLastUrlSize) != 0) // NULL文字込み
        {
            return nn::ec::ResultShopCanceled();
        }

        return nn::ResultSuccess();
    }
}

ShowShopPageArgStorage g_ShopPageArgStorage;
nn::os::MutexType g_ShopPageArgMutex = NN_OS_MUTEX_INITIALIZER(false);

bool ShowShopPageWrapper(ShowShopPageArg& arg, const nn::account::Uid& uid) NN_NOEXCEPT
{
    arg.SetUid(uid);
    arg.SetCallbackUrl(ShopCallbackUrl);

    auto result = ShowShopPage(&g_ReturnValue, arg);
    if (nn::la::ResultLibraryAppletCanceled::Includes(result))
    {
        return false;
    }
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    return true;
}

ShopPageReturnValue& GetShopPageReturnValue() NN_NOEXCEPT
{
    return g_ReturnValue;
}

void LaunchEShopImpl(LaunchMode launchMode, const nn::account::Uid& uid, SourceId sourceId) NN_NOEXCEPT
{
    ShowShopPageArgWrapper arg(g_ShopPageArgStorage, g_ShopPageArgMutex);

    GeneratePath(arg, PathShopBase, sourceId, nullptr,
        "scene", "top",
        "mode", LaunchModeToString(launchMode),
        nullptr, nullptr);

    ShowShopPageWrapper(arg, uid);
}

void ShowShopApplicationInformationImpl(LaunchMode launchMode, nn::ApplicationId applicationId, const nn::account::Uid& uid, SourceId sourceId, const char* newsId) NN_NOEXCEPT
{
    ShowShopPageArgWrapper arg(g_ShopPageArgStorage, g_ShopPageArgMutex);

    // BLOCK: path生成
    {
        char appIdString[MaxAppIdLength]; // 16進数16桁
        AppIdToString(appIdString, applicationId);
        if (newsId != nullptr)
        {
            GeneratePath(arg, PathShopBase, sourceId, nullptr,
                "scene", "product_detail", // タイトル/AOC/バンドル/サービスアイテム/体験版共通
                "mode", LaunchModeToString(launchMode),
                "dst_app_id", appIdString,
                "news_id", NN_DETAIL_EC_SPECIFY_PATH_VALUE_WITH_LENGTH(newsId, nn::util::Strnlen(newsId, MaxNewsIdLength)),
                nullptr, nullptr);
        }
        else
        {
            GeneratePath(arg, PathShopBase, sourceId, nullptr,
                "scene", "product_detail", // タイトル/AOC/バンドル/サービスアイテム/体験版共通
                "mode", LaunchModeToString(launchMode),
                "dst_app_id", appIdString,
                nullptr, nullptr);
        }
    }

    ShowShopPageWrapper(arg, uid);
}

void ShowShopApplicationInformationImpl(LaunchMode launchMode, NsUid nsuidApplication, const nn::account::Uid& uid, SourceId sourceId, const char* newsId) NN_NOEXCEPT
{
    ShowShopPageArgWrapper arg(g_ShopPageArgStorage, g_ShopPageArgMutex);

    // BLOCK: path生成
    {
        char nsuidString[MaxNsUidLength];
        NsUidToString(nsuidString, nsuidApplication);
        if (newsId != nullptr)
        {
            GeneratePath(arg, PathShopBase, sourceId, nullptr,
                "scene", "product_detail", // タイトル/AOC/バンドル/サービスアイテム/体験版共通
                "mode", LaunchModeToString(launchMode),
                "dst_nsuid", nsuidString,
                "news_id", NN_DETAIL_EC_SPECIFY_PATH_VALUE_WITH_LENGTH(newsId, nn::util::Strnlen(newsId, MaxNewsIdLength)),
                nullptr, nullptr);
        }
        else
        {
            GeneratePath(arg, PathShopBase, sourceId, nullptr,
                "scene", "product_detail", // タイトル/AOC/バンドル/サービスアイテム/体験版共通
                "mode", LaunchModeToString(launchMode),
                "dst_nsuid", nsuidString,
                nullptr, nullptr);
        }
    }

    ShowShopPageWrapper(arg, uid);
}

void ShowShopAddOnContentListImpl(LaunchMode launchMode, nn::ApplicationId applicationIdParent,
    const nn::account::Uid& uid, SourceId sourceId) NN_NOEXCEPT
{
    ShowShopPageArgWrapper arg(g_ShopPageArgStorage, g_ShopPageArgMutex);

    // BLOCK: path生成
    {
        char dstAppIdString[MaxAppIdLength]; // 16進数16桁
        AppIdToString(dstAppIdString, applicationIdParent);
        GeneratePath(arg, PathShopBase, sourceId, nullptr,
            "scene", "aocs",
            "mode", LaunchModeToString(launchMode),
            "dst_app_id", dstAppIdString,
            nullptr, nullptr);
    }

    ShowShopPageWrapper(arg, uid);
}

void ShowShopAddOnContentDetailsImpl(LaunchMode launchMode, NsUid nsuidAddOnContentItem,
    const nn::account::Uid& uid, SourceId sourceId) NN_NOEXCEPT
{
    ShowShopPageArgWrapper arg(g_ShopPageArgStorage, g_ShopPageArgMutex);

    // BLOCK: path生成
    {
        char nsuidString[MaxNsUidLength];
        NsUidToString(nsuidString, nsuidAddOnContentItem);
        GeneratePath(arg, PathShopBase, sourceId, nullptr,
            "scene", "product_detail", // タイトル/AOC/バンドル/サービスアイテム/体験版共通
            "mode", LaunchModeToString(launchMode),
            "dst_nsuid", nsuidString,
            nullptr, nullptr);
    }

    ShowShopPageWrapper(arg, uid);
}

void ShowShopSubscriptionListImpl(LaunchMode launchMode, nn::ApplicationId applicationIdParent,
    const nn::account::Uid& uid, SourceId sourceId) NN_NOEXCEPT
{
    ShowShopPageArgWrapper arg(g_ShopPageArgStorage, g_ShopPageArgMutex);

    // BLOCK: path生成
    {
        char dstAppIdString[MaxAppIdLength]; // 16進数16桁
        AppIdToString(dstAppIdString, applicationIdParent);
        GeneratePath(arg, PathShopBase, sourceId, nullptr,
            "scene", "subscriptions",
            "mode", LaunchModeToString(launchMode),
            "dst_app_id", dstAppIdString,
            nullptr, nullptr);
    }

    ShowShopPageWrapper(arg, uid);
}

void ShowShopSubscriptionListDetailImpl(LaunchMode launchMode, nn::ApplicationId applicationIdParent, const CourseId& courseId,
    const nn::account::Uid& uid, SourceId sourceId) NN_NOEXCEPT
{
    ShowShopPageArgWrapper arg(g_ShopPageArgStorage, g_ShopPageArgMutex);

    // BLOCK: path生成
    {
        char dstAppIdString[MaxAppIdLength]; // 16進数16桁
        AppIdToString(dstAppIdString, applicationIdParent);
        GeneratePath(arg, PathShopBase, sourceId, nullptr,
            "scene", "subscriptions",
            "mode", LaunchModeToString(launchMode),
            "dst_app_id", dstAppIdString,
            "dst_course_id", courseId.value,
            nullptr, nullptr);
    }

    ShowShopPageWrapper(arg, uid);
}

void ShowShopConsumableItemListImpl(LaunchMode launchMode, nn::ApplicationId applicationIdParent,
    const nn::account::Uid& uid, SourceId sourceId) NN_NOEXCEPT
{
    ShowShopPageArgWrapper arg(g_ShopPageArgStorage, g_ShopPageArgMutex);

    // BLOCK: path生成
    {
        char dstAppIdString[MaxAppIdLength]; // 16進数16桁
        AppIdToString(dstAppIdString, applicationIdParent);
        GeneratePath(arg, PathShopBase, sourceId, nullptr,
            "scene", "consumption",
            "mode", LaunchModeToString(launchMode),
            "dst_app_id", dstAppIdString,
            nullptr, nullptr);
    }

    ShowShopPageWrapper(arg, uid);
}
void ShowShopConsumableItemDetailImpl(LaunchMode launchMode, nn::ApplicationId applicationIdParent, const ConsumableId& consumableId, NsUid nsuidConsumableItem,
    const nn::account::Uid& uid, SourceId sourceId) NN_NOEXCEPT
{
    ShowShopPageArgWrapper arg(g_ShopPageArgStorage, g_ShopPageArgMutex);

    // BLOCK: path生成
    {
        char dstAppIdString[MaxAppIdLength]; // 16進数16桁
        AppIdToString(dstAppIdString, applicationIdParent);
        char nsuidString[MaxNsUidLength];
        NsUidToString(nsuidString, nsuidConsumableItem);
        GeneratePath(arg, PathShopBase, sourceId, nullptr,
            "scene", "consumption_item",
            "mode", LaunchModeToString(launchMode),
            "dst_app_id", dstAppIdString,
            "dst_consumption_id", consumableId.value,
            "dst_consumption_item_nsuid", nsuidString,
            nullptr, nullptr);
    }

    ShowShopPageWrapper(arg, uid);
}

void ShowShopCouponImpl(LaunchMode launchMode, CouponId couponId, const nn::account::Uid& uid, SourceId sourceId) NN_NOEXCEPT
{
    ShowShopPageArgWrapper arg(g_ShopPageArgStorage, g_ShopPageArgMutex);

    // BLOCK: path生成
    if (couponId == CouponId::GetInvalidId())
    {
        GeneratePath(arg, PathShopBase, sourceId, nullptr,
            "scene", "my_coupon",
            "mode", LaunchModeToString(launchMode),
            nullptr, nullptr);
    }
    else
    {
        char couponIdString[MaxCouponIdLength];
        CouponIdToString(couponIdString, couponId);
        GeneratePath(arg, PathShopBase, sourceId, nullptr,
            "scene", "my_coupon",
            "mode", LaunchModeToString(launchMode),
            "coupon_id", couponIdString,
            nullptr, nullptr);
    }

    ShowShopPageWrapper(arg, uid);
}

void ShowShopInformationWithParamsImpl(LaunchMode launchMode, const char* params, const nn::account::Uid& uid, SourceId sourceId) NN_NOEXCEPT
{
    ShowShopPageArgWrapper arg(g_ShopPageArgStorage, g_ShopPageArgMutex);

    GeneratePath(arg, PathShopBase, sourceId, params,
        "mode", LaunchModeToString(launchMode),
        nullptr, nullptr);

    ShowShopPageWrapper(arg, uid);
}

void ShowShopSecretApplicationInformationImpl(LaunchMode launchMode, nn::ApplicationId applicationId,
    const nn::account::Uid& uid, SourceId sourceId) NN_NOEXCEPT
{
    ShowShopPageArgWrapper arg(g_ShopPageArgStorage, g_ShopPageArgMutex);

    // BLOCK: path生成
    {
        char appIdString[MaxAppIdLength]; // 16進数16桁
        AppIdToString(appIdString, applicationId);
        GeneratePath(arg, PathShopBase, sourceId, nullptr,
            "scene", "product_detail", // タイトル/AOC/バンドル/サービスアイテム/体験版共通
            "mode", LaunchModeToString(launchMode),
            "allow_unsearchable", "true",
            "dst_app_id", appIdString,
            nullptr, nullptr);
    }

    ShowShopPageWrapper(arg, uid);
}

void ShowShopSecretApplicationInformationImpl(LaunchMode launchMode, NsUid nsuidApplication,
    const nn::account::Uid& uid, SourceId sourceId) NN_NOEXCEPT
{
    ShowShopPageArgWrapper arg(g_ShopPageArgStorage, g_ShopPageArgMutex);

    // BLOCK: path生成
    {
        char nsuidString[MaxNsUidLength];
        NsUidToString(nsuidString, nsuidApplication);
        GeneratePath(arg, PathShopBase, sourceId, nullptr,
            "scene", "product_detail", // タイトル/AOC/バンドル/サービスアイテム/体験版共通
            "mode", LaunchModeToString(launchMode),
            "allow_unsearchable", "true",
            "dst_nsuid", nsuidString,
            nullptr, nullptr);
    }

    ShowShopPageWrapper(arg, uid);
}

void ShowShopSettingsImpl(LaunchMode launchMode, const nn::account::Uid& uid, SourceId sourceId) NN_NOEXCEPT
{
    ShowShopPageArgWrapper arg(g_ShopPageArgStorage, g_ShopPageArgMutex);

    // BLOCK: path生成
    {
        GeneratePath(arg, PathShopBase, sourceId, nullptr,
            "scene", "setting",
            "mode", LaunchModeToString(launchMode),
            nullptr, nullptr);
    }

    ShowShopPageWrapper(arg, uid);
}

nn::Result ShowShopMembershipServiceInformationImpl(const nn::account::Uid& uid, SourceId sourceId) NN_NOEXCEPT
{
    ShowShopPageArgWrapper arg(g_ShopPageArgStorage, g_ShopPageArgMutex);

    // BLOCK: path生成
    {
        GeneratePath(arg, PathShopBase, sourceId, nullptr,
            "scene", "membership_service",
            "mode", LaunchModeToString(LaunchMode_Mini), // mini 固定
            nullptr, nullptr);
    }

    NN_RESULT_THROW_UNLESS(ShowShopPageWrapper(arg, uid), nn::ec::ResultShopCanceled());

    auto& returnValue = GetShopPageReturnValue();
    NN_RESULT_DO(ShopReturnValueToResult(returnValue));

    NN_RESULT_SUCCESS;
}

nn::Result GetShopResultFromStorageImpl(const nn::applet::StorageHandle& handle) NN_NOEXCEPT
{
    auto& returnValue = GetShopPageReturnValue();
    GetShopPageReturnValueFromStorage(&returnValue, handle);
    return ShopReturnValueToResult(returnValue);
}

void ShowShopPreinstallInformationImpl(LaunchMode launchMode, const nn::account::Uid& uid, SourceId sourceId) NN_NOEXCEPT
{
    ShowShopPageArgWrapper arg(g_ShopPageArgStorage, g_ShopPageArgMutex);

    // BLOCK: path生成
    {
        GeneratePath(arg, PathShopBase, sourceId, nullptr,
            "scene", "preinstall",
            "mode", LaunchModeToString(launchMode),
            nullptr, nullptr);
    }

    ShowShopPageWrapper(arg, uid);
}

}}}
