﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/socket/socket_Api.h>
#include <nn/dns/parser.h>

//#define NN_DNSPARSER_LOG_LEVEL NN_DNSPARSER_LOG_LEVEL_HEX
#define NN_DNSPARSER_LOG_MODULE_NAME "QuestionIterator" // NOLINT(preprocessor/const)
#include "dns_ParserLog.h"
#include "dns_ParserMacros.h"

extern "C"
{
#include <nnc/dns/parser.h>
};

NN_DNSPARSER_STATIC_ASSERT(sizeof(struct nndnsparserQuestionIterator) == sizeof(nn::dns::parser::QuestionIterator));
NN_DNSPARSER_STATIC_ASSERT(alignof(struct nndnsparserQuestionIterator) == alignof(nn::dns::parser::QuestionIterator));

namespace nn { namespace dns { namespace parser {

QuestionIterator::QuestionIterator() :
    m_pMessage(nullptr),
    m_Section { },
    m_pCurrent(nullptr),
    m_Count(0),
    m_Question(Question())
{
};

QuestionIterator::QuestionIterator(const QuestionIterator& rhs) :
    m_pMessage(rhs.m_pMessage),
    m_Section(rhs.m_Section),
    m_pCurrent(rhs.m_pCurrent),
    m_Count(rhs.m_Count),
    m_Question(rhs.m_Question)
{
};

QuestionIterator::~QuestionIterator()
{
};

QuestionIterator& QuestionIterator::operator=(const QuestionIterator& rhs)
{
    if (this == &rhs)
    {
        goto bail;
    };

    m_pMessage = rhs.m_pMessage;
    m_Section = rhs.m_Section;
    m_pCurrent = rhs.m_pCurrent;
    m_Count = rhs.m_Count;
    m_Question = rhs.m_Question;

bail:
    return *this;
};

int QuestionIterator::Initialize(const Message& message, const MessageSectionConstant tag) NN_NOEXCEPT
{
    NN_DNSPARSER_STATIC_ASSERT(offsetof(struct nndnsparserQuestionIterator, pMessage) ==
                               offsetof(nn::dns::parser::QuestionIterator, m_pMessage));

    NN_DNSPARSER_STATIC_ASSERT(offsetof(struct nndnsparserQuestionIterator, section) ==
                               offsetof(nn::dns::parser::QuestionIterator, m_Section));

    NN_DNSPARSER_STATIC_ASSERT(offsetof(struct nndnsparserQuestionIterator, pCurrent) ==
                               offsetof(nn::dns::parser::QuestionIterator, m_pCurrent));

    NN_DNSPARSER_STATIC_ASSERT(offsetof(struct nndnsparserQuestionIterator, count) ==
                               offsetof(nn::dns::parser::QuestionIterator, m_Count));

    NN_DNSPARSER_STATIC_ASSERT(offsetof(struct nndnsparserQuestionIterator, current) ==
                               offsetof(nn::dns::parser::QuestionIterator, m_Question));

    int rc = -1;
    const uint8_t* messageBuffer = nullptr;
    size_t messageBufferSize = 0;

    if (nullptr == (messageBuffer = message.GetBuffer()))
    {
        NN_DNSPARSER_LOG_DEBUG("\n");
        goto bail;
    }
    else if (0 == (messageBufferSize = message.GetBufferSize()))
    {
        NN_DNSPARSER_LOG_MINOR("\n");
        goto bail;
    };

    switch (tag)
    {
    case MessageSectionConstant::Question:
        m_Count = const_cast<Message&>(message).GetHeader().GetQuestionCount();
        m_Section = const_cast<Message&>(message).GetQuestionSection();
        break;
    case MessageSectionConstant::Answer:
        goto bail;
        break;
    case MessageSectionConstant::Authority:
        goto bail;
        break;
    case MessageSectionConstant::Additional:
        goto bail;
        break;
    default:
        goto bail;
    };

    if (nullptr == m_Section.pStart)
    {
        NN_DNSPARSER_LOG_DEBUG("\n");
        rc = 0;
        goto bail;
    }
    else if (nullptr == m_Section.pEnd)
    {
        NN_DNSPARSER_LOG_DEBUG("\n");
        rc = 0;
        goto bail;
    }
    else if (0 == m_Count)
    {
        NN_DNSPARSER_LOG_DEBUG("\n");
        rc = 0;
        goto bail;
    }
    else if (m_Section.pStart < messageBuffer)
    {
        NN_DNSPARSER_LOG_DEBUG("\n");
        goto bail;
    }
    else if (m_Section.pStart > (messageBuffer + messageBufferSize))
    {
        NN_DNSPARSER_LOG_DEBUG("\n");
        goto bail;
    }
    else if (m_Section.pEnd < messageBuffer)
    {
        NN_DNSPARSER_LOG_DEBUG("\n");
        goto bail;
    }
    else if (m_Section.pEnd > messageBuffer + messageBufferSize)
    {
        NN_DNSPARSER_LOG_DEBUG("\n");
        goto bail;
    };

    m_pMessage = &message;
    m_pCurrent = m_Section.pStart;

    rc = GetNext();

bail:
    return rc;
};

int QuestionIterator::GetNext() NN_NOEXCEPT
{
    int rc = -1;
    ssize_t bytes = 0;
    if (m_pCurrent >= m_Section.pEnd)
    {
        rc = 0;
        goto bail;
    };

    if (m_Count != 0)
    {
        m_Question = Question();

        m_Question.Initialize(m_pMessage);

        bytes = m_Question.FromBuffer(m_pCurrent, m_Section.pEnd - m_pCurrent);
        if (0 == bytes )
        {
            goto bail;
        }
        else if (-1 == bytes )
        {
            goto bail;
        }
        else if (bytes + m_pCurrent > m_Section.pEnd)
        {
            goto bail;
        }
        else
        {
            rc = 1;
            m_pCurrent += bytes;
            m_Count -= 1;
        };
    };

bail:
    return rc;
};

Question& QuestionIterator::GetCurrent() NN_NOEXCEPT
{
    return m_Question;
};

}}}; //nn::dns::parser
