﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once
#include <nn/nn_Macro.h>
#include <nn/nn_SdkAssert.h>
#include "dns_ParserLog.h"

namespace nn { namespace dns { namespace parser {

/**
 * @brief A macro that asserts in Develop / Debug builds and jumps
 * to the label provided in error in Release.
 *
 * @param condition The condition to check. If it is true then the
 * assert is hit in Debug / Develop builds.
 *
 * @param error The label to jump to in Release builds.
 */
#define NN_DNSPARSER_ERROR_IF(condition, error)                         \
    do                                                                  \
    {                                                                   \
        if ( (condition) )                                              \
        {                                                               \
            NN_DNSPARSER_LOG_DEBUG("Failure of condition: %s",          \
                                   #condition);                         \
            NN_SDK_ASSERT(!(condition));                                \
            goto error;                                                 \
        }                                                               \
    } while(NN_STATIC_CONDITION(false))

/**
 * @brief A macro for static_assert that adds the condition as string
 * parameter.
 *
 * @param condition The condition that must be true.
 */
#define NN_DNSPARSER_STATIC_ASSERT(condition)   \
    static_assert(condition, #condition);

/**
 * @brief Macro that checks whether the buffer is within the bounds
 * of the buffer.
 *
 * @param pBuffer The buffer.
 *
 * @param size The size of the buffer.
 *
 * @param ptr The pointer to check.
 *
 * @param error The label to jump to on error.
 */
#define NN_DNSPARSER_BOUNDS_CHECK(pBuffer, size, ptr, error)            \
    do                                                                  \
    {                                                                   \
        const uint8_t* start = (uint8_t*)pBuffer;                       \
        const uint8_t* eom = start + size;                              \
                                                                        \
        if ( ptr < start   )                                            \
        {                                                               \
            rc = -1;                                                    \
            goto error;                                                 \
        }                                                               \
        else if ( ptr > eom   )                                         \
        {                                                               \
            rc = -1;                                                    \
            goto error;                                                 \
        };                                                              \
    } while(NN_STATIC_CONDITION(false))

/**
 * @brief This function reads data from a byte stream, optionally
 * flips the endianness, and assigns the data to the provided memory
 * location.
 *
 * @details In Debug/Develop this function asserts if the provided
 * memory location is null, the buffer cursor is null, or if there are
 * insufficient bytes left in the buffer.
 *
 * @param pOutVariable The memory location to write to.
 *
 * @param pCursor The buffer cursor to read from.
 *
 * @param size The size of the buffer at pCursor.
 *
 * @param flip A pointer to a function that flips the data. If not
 * desired then nullptr is allowed.
 *
 * @return Returns the number of bytes processed on success, -1 on error.
 */
template <typename T>
inline ssize_t FromBuffer(T* pOutVariable,
                          const uint8_t * pCursor,
                          size_t size,
                          T (*flip)(const T))
{
    ssize_t rc = -1;
    NN_DNSPARSER_ERROR_IF((nullptr == pOutVariable), bail);
    NN_DNSPARSER_ERROR_IF((nullptr == pCursor), bail);
    NN_DNSPARSER_ERROR_IF((sizeof(T) > size), bail);

    if (nullptr != flip)
    {
        *pOutVariable = flip(*reinterpret_cast<const T*>(pCursor));
    }
    else
    {
        *pOutVariable = *reinterpret_cast<const T*>(pCursor);
    };

    rc = sizeof(T);

bail:
    return rc;
};

/**
 * @brief Macro that reduces FromBuffer / error handling code.
 *
 * @param rc The return code variable.
 *
 * @param type The c++ type for the given variable.
 *
 * @param var The variable.
 *
 * @param ptr The pointer to the buffer.
 *
 * @param sz The size of the buffer in bytes.
 *
 * @param flip The function to flip the buffer with.
 *
 * @param err The label to jump to on error.
 */
#define NN_DNSPARSER_FROMBUFFER_ERROR(rc, type, var, ptr, sz, flip, err)           \
    do                                                                             \
    {                                                                              \
        if (-1 == (rc = nn::dns::parser::FromBuffer(reinterpret_cast<type*>(&var), \
                                                    pCursor,                       \
                                                    size,                          \
                                                    flip)))                        \
            {                                                                      \
                goto err;                                                          \
            };                                                                     \
        size -= rc;                                                                \
        pCursor += rc;                                                             \
    }                                                                              \
    while (NN_STATIC_CONDITION(false))

/**
 * @brief This function writes data to a byte stream, optionally
 * flipping the endianness, from the provided memory location.
 *
 * @details In Debug/Develop this function asserts if the provided
 * memory location is null, the buffer cursor is null, or if there are
 * insufficient bytes left in the buffer.
 *
 * @param pOutCursor The buffer cursor to write to.
 *
 * @param size The size of the buffer at pCursor.
 *
 * @param flip A pointer to a function that flips the data. If not
 * desired then nullptr is allowed.
 *
 * @param pVariable The memory location to read from.
 *
 * @return Returns the number of bytes processed on success, -1 on error.
 */
template <typename T>
inline ssize_t ToBuffer(uint8_t * const pOutCursor,
                        size_t size,
                        T (*flip)(const T),
                        const T* pVariable)
{
    ssize_t rc = -1;
    NN_DNSPARSER_ERROR_IF((nullptr == pOutCursor), bail);
    NN_DNSPARSER_ERROR_IF((nullptr == pVariable), bail);
    NN_DNSPARSER_ERROR_IF((sizeof(T) > size), bail);

    if (nullptr != flip)
    {
        *reinterpret_cast<T*>(pOutCursor) = flip(*pVariable);
    }
    else
    {
        *reinterpret_cast<T*>(pOutCursor) = *pVariable;
    };

    rc = sizeof(T);

bail:
    return rc;
};

/**
 * @brief Macro that reduces ToBuffer / error handling code.
 *
 * @param rc The return code variable.
 *
 * @param ptr The pointer to the buffer.
 *
 * @param sz The size of the buffer in bytes.
 *
 * @param flip The function to flip the buffer with.
 *
 * @param type The c++ type for the given variable.
 *
 * @param var The variable.
 *
 * @param err The label to jump to on error.
 */
#define NN_DNSPARSER_TOBUFFER_ERROR(rc, ptr, sz, flip, type, var, err)                    \
    do                                                                                    \
    {                                                                                     \
        if ( -1 == (rc = nn::dns::parser::ToBuffer(ptr,                                   \
                                                   sz,                                    \
                                                   flip,                                  \
                                                   reinterpret_cast<const type*>(&var)))) \
        {                                                                                 \
            goto err;                                                                     \
        };                                                                                \
        sz -= rc;                                                                         \
        ptr += rc;                                                                        \
    }                                                                                     \
    while (NN_STATIC_CONDITION(false))

}}}; // nn::dns::parser
