﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_SdkLog.h>
#include <nn/nn_Log.h>
#include <nn/socket/socket_Api.h>
#include <nn/util/util_FormatString.h>
#include <cstring>
#include <cctype>
#include <algorithm>
#include "dns_ParserLog.h"

extern "C"
{                   // adding nolint because C++ versions do not exist
                    // in rynda and this causes a compilation error
#include <stdio.h>  // NOLINT(others/c_header)
#include <stdlib.h> // NOLINT(others/c_header)
}

namespace nn { namespace dns { namespace parser {

const char* g_LogLevelTable[] = {
    "Major",
    "Minor",
    "Info",
    "Debug",
    "Hex"
};

const unsigned int HexCharactersPerByte = 3;
const unsigned int NewlinePerLine = 1;
const unsigned int BytesPerLine = 16;
const unsigned int NewlineAndNull = 2;

static
const unsigned int GetHexStringBufferSize(unsigned bufferSize) NN_NOEXCEPT
{
#define LINES(size)  (                                          \
        (size / BytesPerLine) == 0 ?                            \
        1 :                                                     \
        ((size / BytesPerLine) + (size % BytesPerLine)))

#define GET_HEXDUMP_BUFFER_SIZE(size) (                 \
        (HexCharactersPerByte * size)           +       \
        (NewlinePerLine * (LINES(size)))        +       \
        NewlineAndNull                                  \
        )
    size_t sz = GET_HEXDUMP_BUFFER_SIZE(bufferSize);
    return sz + 2; // newline and null terminator
#undef GET_HEXDUMP_BUFFER_SIZE
#undef LINES
}

static
void CreateHexString(char* pOutString,
                     unsigned int strSize,
                     const uint8_t* pBytes,
                     unsigned int byteSize)  NN_NOEXCEPT
{
    char* pHex = pOutString;

    for (unsigned int idx=0; idx<byteSize; ++idx)
    {
        pHex += nn::util::SNPrintf(pHex,
                                   strSize - (pHex - pOutString),
                                   "%02x ", *pBytes);
        pBytes += 1;

        if (idx == (byteSize - 1))
        {
            *pHex = '\n';
            ++pHex;
        };
    };

    *pHex = 0;
};

const char* GetFilename(const char* line) NN_NOEXCEPT
{
    unsigned int length = 0;
    if (nullptr == line)
    {
        goto bail;
    };

    length = strlen(line);
    for (unsigned int idx=length; idx!=0; --idx)
    {
        if (line[idx] == '/' || line[idx] == '\\')
        {
            return &line[idx + 1];
        };
    };

bail:
    return line;
};

const char* GetLevelString(unsigned int level) NN_NOEXCEPT
{
    if (level > NN_DNSPARSER_LOG_LEVEL_MAX)
    {
        return "undefined";
    }
    return g_LogLevelTable[level];
};

void Log(const void* pDataBuffer,
         unsigned int dataBufferSize,
         const char* pModule,
         unsigned int line,
         const char* pFunctionName,
         unsigned int level,
         const char* pFormat, ...) NN_NOEXCEPT
{
    va_list args;
    va_start (args, pFormat);
    int hexStringBufferSize = nullptr == pDataBuffer ? 0 : GetHexStringBufferSize(dataBufferSize);

    int formatStringLength = vsnprintf(nullptr, 0, pFormat, args) + 1;
    char* pString = static_cast<char*>(alloca(formatStringLength + hexStringBufferSize + 1));

    memset(pString, 0, formatStringLength + hexStringBufferSize + 1);
    vsnprintf(pString, formatStringLength, pFormat, args);

    CreateHexString(pString + formatStringLength - 1, hexStringBufferSize, reinterpret_cast<const uint8_t*>(pDataBuffer), dataBufferSize);

    NN_SDK_LOG("%s %u %s %llu %s", pModule, line, pFunctionName, os::GetThreadId(os::GetCurrentThread()), pString);
    va_end(args);
};

}}}; // nn::dns::parser
