﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/socket/socket_Api.h>
#include <nn/dns/parser.h>

//#define NN_DNSPARSER_LOG_LEVEL NN_DNSPARSER_LOG_LEVEL_HEX
#define NN_DNSPARSER_LOG_MODULE_NAME "DnsLabel" // NOLINT(preprocessor/const)
#include "dns_ParserLog.h"
#include "dns_ParserMacros.h"
#include <nn/nn_SdkLog.h>

extern "C"
{
#include <nnc/dns/parser.h>
};

NN_DNSPARSER_STATIC_ASSERT(sizeof(struct nndnsparserLabel) == sizeof(nn::dns::parser::Label));
NN_DNSPARSER_STATIC_ASSERT(alignof(struct nndnsparserLabel) == alignof(nn::dns::parser::Label));

namespace nn { namespace dns { namespace parser {

/**
 * @brief Validate a label buffer.
 */
bool IsLabelValid(const uint8_t* pMessageData,
                  const uint8_t* pEndOfMessage,
                  const uint8_t* pCursor) NN_NOEXCEPT;

/**
 * @brief Determine if the label is of the compressed type..
 *
 * @param[in] value This is the first octet in the label bytes.
 *
 * @return Returns true if the two most significant bits are set.
 */
bool IsLabelTypePointer(const uint8_t value) NN_NOEXCEPT
{
    uint8_t v = ((value >> static_cast<int>(LabelConstant::TypeShift)) &
                 static_cast<int>(LabelConstant::TypeMask));
    bool rc =  v == static_cast<int>(LabelConstant::TypeCompressed);
    return rc;
};

/**
 * @brief Determine if the label is of the normal type.
 *
 * @param[in] value This is the first octet in the bytes.
 *
 * @return Returns true if the two most significant bits are unset.
 */
bool IsLabelTypeNormal(const uint8_t value) NN_NOEXCEPT
{
    uint8_t x = ((value >> static_cast<int>(LabelConstant::TypeShift)) &
                 static_cast<int>(LabelConstant::TypeMask));
    bool rc = x == static_cast<int>(LabelConstant::TypeNormal);
    return rc;
};

bool IsLabelPointerValid(const uint8_t* pMessageStart, const uint8_t* pMessageEnd,
                         const uint8_t* pLabelStart, const uint8_t* pLabelEnd,
                         const uint8_t* pointer)
{
    bool rc = false;

    // If one is null and not the other then bail.
    NN_DNSPARSER_ERROR_IF((pMessageStart == nullptr && pMessageEnd != nullptr) || (pMessageStart != nullptr && pMessageEnd == nullptr), bail);
    // If both are null and the end is before the beginning then bail.
    NN_DNSPARSER_ERROR_IF((pMessageStart != nullptr) && (pMessageEnd < pMessageStart), bail);
    // If the label start is null then bail.
    NN_DNSPARSER_ERROR_IF(pLabelStart == nullptr, bail);
    // if the label end is null then bail.
    NN_DNSPARSER_ERROR_IF(pLabelEnd == nullptr, bail);
    // if the end is before the beginning then bail
    NN_DNSPARSER_ERROR_IF(pLabelEnd < pLabelStart, bail);

    if (nullptr == pMessageStart)
    {
        if (pointer >= pLabelStart && pointer <= pLabelEnd)
        {
            rc = true;
        };
    }
    else
    {
        if ((pointer >= pLabelStart && pointer <= pLabelEnd) ||
            (pointer >= pMessageStart && pointer <= pMessageEnd))
        {
            rc = true;
        };
    };

bail:
    return rc;
};

/**
 * @brief Given a pointer to a buffer that has at least two bytes in it
 * this function returns the offset version of a pointer value.
 */
uint16_t LabelGetOffsetUnsafe(const uint8_t* pCursor) NN_NOEXCEPT
{
    uint16_t offset = *pCursor & static_cast<int>(LabelConstant::LengthOffsetMask);
    offset <<= static_cast<int>(LabelConstant::OffsetLeftShift);
    offset |= *(pCursor + 1);
    return offset;
};

/**
 * @brief Produce a character cursor from a cursor into the DNS
 * label structure byte.
 *
 * As stated elsewhere DNS Labels contain characters, a 14-bit
 * offset value into the DNS message buffer (called a "pointer"),
 * or both. This function is used to produce a location that produces
 * a character cursor from the provided byte cursor into the DNS
 * Label / message buffer. It also checks for errors such
 * as a DNS "pointer" value following another, pointer flags at the
 * end of the message, and a Label object that does not contain
 * a message; in error cases nullptr is returned.
 *
 * @param[in] pLabel This parameter is the pointer to the label object.
 *
 * @param[in] pCursor The byte cursor.
 *
 * @return This function returns the character cursor or nullptr on
 * error.
 */
static inline
const uint8_t* LabelGetCharacterCursor(const Label* pLabel,
                                       const uint8_t* pCursor) NN_NOEXCEPT
{
    NN_DNSPARSER_LOG_DEBUG("pLabel: %p, pCursor: %p\n", pLabel, pCursor);

    uint16_t offset = 0;
    const Message* pMessage = nullptr;
    const uint8_t* pMessageBuffer = nullptr;
    size_t messageBufferSize = 0;

    if (nullptr == pLabel)
    {
        NN_DNSPARSER_LOG_DEBUG("\n");
        pCursor = nullptr;
        goto bail;
    }
    else if (nullptr == pCursor)
    {
        NN_DNSPARSER_LOG_DEBUG("\n");
        goto bail;
    }
    else if (!IsLabelTypePointer(*pCursor))
    {
        goto bail;
    }
    else if (nullptr == (pMessage = const_cast<Label*>(pLabel)->GetMessage()))
    {
        NN_DNSPARSER_LOG_DEBUG("\n");
        pCursor = nullptr;
        goto bail;
    }
    else if (nullptr == (pMessageBuffer = pMessage->GetBuffer()))
    {
        NN_DNSPARSER_LOG_DEBUG("\n");
        pCursor = nullptr;
        goto bail;
    }
    else if (0 == (messageBufferSize = pMessage->GetBufferSize()))
    {
        NN_DNSPARSER_LOG_DEBUG("\n");
        goto bail;
    }
    else if (pCursor == pMessageBuffer + messageBufferSize)
    {
        NN_DNSPARSER_LOG_MINOR("label cursor is a pointer but at EOB\n");
        pCursor = NULL;
        goto bail;
    };

    offset = LabelGetOffsetUnsafe(pCursor);
    if ( 0 == offset)
    {
        NN_DNSPARSER_LOG_DEBUG("\n");
        pCursor = nullptr;
        goto bail;
    };

    pCursor = pMessageBuffer + offset;
    if (pCursor == pMessageBuffer + messageBufferSize)
    {
        NN_DNSPARSER_LOG_DEBUG("label cursor is a pointer but at EOB\n");
        pCursor = nullptr;
        goto bail;
    }
    else if (IsLabelTypePointer(*pCursor))
    {
        NN_DNSPARSER_LOG_DEBUG("pLabel: %p, pCursor: %p, pointer followed pointer!\n",
                               pLabel, pCursor);
        pCursor = nullptr;
    };

bail:
    NN_DNSPARSER_LOG_DEBUG("returning: %p\n", pCursor);
    return pCursor;
}

/**
 * @brief This function traverses the buffer to find the nullptr
 * or a DNS "pointer" offset.
 *
 * It also handles bounds checking
 *
 * @param pBuffer This parameter is the buffer pointer.
 *
 * @param size This parameter is the size of the buffer.
 *
 * @return The size of the label buffer or -1 on error such as
 * insufficient buffer size.
 */
ssize_t LabelSizeOfInternal(const uint8_t* pBuffer, size_t size) NN_NOEXCEPT
{
    NN_DNSPARSER_LOG_DEBUG("pBuffer: %p, size: %zu\n", pBuffer, size);
    ssize_t rc = 0;
    bool done = false;
    const uint8_t* pCursor = nullptr;

    for (pCursor = pBuffer; !done && pCursor - pBuffer < size; )
    {
        uint8_t val = *pCursor;

        if (true == IsLabelTypePointer(*pCursor))
        {
            if ((pCursor + 1) < (pBuffer + size))
            {
                rc += static_cast<unsigned int>(LabelConstant::TypeCompressedSize);
                goto bail;
            }
            else
            {
                NN_DNSPARSER_LOG_DEBUG("\n");
                rc = -1;
            };
            goto bail;
        }
        else if (true == IsLabelTypeNormal(*pCursor))
        {
            pCursor += val + 1;
            if (pCursor > (pBuffer + size))
            {
                NN_DNSPARSER_LOG_DEBUG("\n");
                rc = -1;
                goto bail;
            }
            else if (0 == val)
            {
                done = true;
            };
        }
        else
        {
            rc = -1;
            goto bail;
        };
    };
    rc = pCursor - pBuffer;

bail:
    NN_DNSPARSER_LOG_DEBUG("returning: %zd\n", rc);
    return rc;
};

Label::Label() :
    m_pMessage(nullptr),
    m_DirtySize(0),
    m_pData(nullptr)
{
    NN_DNSPARSER_LOG_DEBUG("\n");
    m_Range.pStart = m_Range.pEnd = nullptr;
};

Label::Label(const Label& rhs) :
    m_pMessage(rhs.m_pMessage),
    m_Range(rhs.m_Range),
    m_DirtySize(rhs.m_DirtySize),
    m_pData(rhs.m_pData)
{
    NN_DNSPARSER_LOG_DEBUG("\n");
};

Label::~Label()
{
    NN_DNSPARSER_LOG_DEBUG("\n");
};

Label& Label::operator=(const Label& rhs)
{
    NN_DNSPARSER_LOG_DEBUG("\n");
    if (this == &rhs)
    {
        goto bail;
    };

    m_pMessage = rhs.m_pMessage;
    m_Range = rhs.m_Range;
    m_DirtySize = rhs.m_DirtySize;
    m_pData = rhs.m_pData;

bail:
    return *this;
};

const Message* & Label::GetMessage() NN_NOEXCEPT
{
    NN_DNSPARSER_LOG_DEBUG("\n");
    return m_pMessage;
};

const uint8_t* Label::GetData() NN_NOEXCEPT
{
    NN_DNSPARSER_LOG_DEBUG("\n");
    return m_pData;
};

void Label::Initialize(const Message* pMessage) NN_NOEXCEPT
{
    NN_DNSPARSER_STATIC_ASSERT(offsetof(struct nndnsparserLabel, pMessage) ==
                               offsetof(nn::dns::parser::Label, m_pMessage));

    NN_DNSPARSER_STATIC_ASSERT(offsetof(struct nndnsparserLabel, range) ==
                               offsetof(nn::dns::parser::Label, m_Range));

    NN_DNSPARSER_STATIC_ASSERT(offsetof(struct nndnsparserLabel, dirtySize) ==
                               offsetof(nn::dns::parser::Label, m_DirtySize));

    NN_DNSPARSER_STATIC_ASSERT(offsetof(struct nndnsparserLabel, data) ==
                               offsetof(nn::dns::parser::Label, m_pData));

    memset(this, 0, sizeof(*this));
    m_pMessage = pMessage;
    return;
};

ssize_t Label::SizeOf() const NN_NOEXCEPT
{
    NN_DNSPARSER_LOG_DEBUG("\n");

    ssize_t rc = -1;
    if (nullptr == m_pData)
    {
        rc = 0;
        goto bail;
    }
    else if (0 != m_DirtySize)
    {
        rc = m_DirtySize;
    }
    else
    {
        rc = LabelSizeOfInternal(m_pData, m_DirtySize);
    };

bail:
    NN_DNSPARSER_LOG_DEBUG("returning: %zd\n", rc);
    return rc;
};

bool Label::operator==(const Label& that) const NN_NOEXCEPT
{
    NN_DNSPARSER_LOG_DEBUG("that: %p\n", &that);
    bool rc = false;
    unsigned characters = 0;
    const uint8_t* pCursor1 = nullptr;
    const uint8_t* pCursor2 = nullptr;

    if (this == &that)
    {
        NN_DNSPARSER_LOG_DEBUG("\n");
        rc = true;
        goto bail;
    }
    else if (m_pData == that.m_pData)
    {
        NN_DNSPARSER_LOG_DEBUG("\n");
        rc = true;
        goto bail;
    };

    for (pCursor1 = m_pData,
             pCursor2 = that.m_pData;

         pCursor1 != nullptr &&
             pCursor2 != nullptr &&
             characters < static_cast<unsigned int>(LabelConstant::LabelSequenceMaximumBufferSize);

         ++pCursor1,
             ++pCursor2,
             ++characters)
    {
        if (nullptr == (pCursor1 = LabelGetCharacterCursor(this, pCursor1)))
        {
            NN_DNSPARSER_LOG_DEBUG("\n");
            goto bail;
        }
        else if (nullptr == (pCursor2 = LabelGetCharacterCursor(&that, pCursor2)))
        {
            NN_DNSPARSER_LOG_DEBUG("\n");
            goto bail;
        }
        else if (0 == *pCursor1 && 0 == *pCursor2)
        {
            NN_DNSPARSER_LOG_DEBUG("\n");
            rc = true;
            goto bail;
        }
        else if (*pCursor1 != *pCursor2)
        {
            NN_DNSPARSER_LOG_DEBUG("\n");
            goto bail;
        };
    };

bail:
    NN_DNSPARSER_LOG_DEBUG("returning: %s\n", true == rc ? "true" : "false");
    return rc;
};

ssize_t Label::GetStringBufferSize() const NN_NOEXCEPT
{
    NN_DNSPARSER_LOG_DEBUG("\n");
    ssize_t rc = -1;
    unsigned characters = 0;
    const uint8_t* pCursor = nullptr;

    if (nullptr == m_pData)
    {
        NN_DNSPARSER_LOG_DEBUG("\n");
        goto bail;
    };

    for (pCursor = m_pData;
         characters < static_cast<unsigned int>(LabelConstant::LabelSequenceMaximumBufferSize);
         ++pCursor, ++characters)
    {
        if (nullptr == (pCursor = LabelGetCharacterCursor(this, pCursor)))
        {
            NN_DNSPARSER_LOG_DEBUG("\n");
            goto bail;
        }
        else if (0 == *pCursor)
        {
            break;
        };
    };

    rc = characters;

bail:
    NN_DNSPARSER_LOG_DEBUG("returning: %zd\n", rc);
    return rc;
};

ssize_t Label::FromBuffer(const uint8_t* pBuffer, size_t size) NN_NOEXCEPT
{
    NN_DNSPARSER_LOG_DEBUG("pBuffer: %p, size: %zd\n", pBuffer, size);

    ssize_t rc = -1;

    NN_DNSPARSER_ERROR_IF(nullptr == pBuffer, bail);

    m_Range.pStart = pBuffer;

    if (nullptr != m_pMessage && nullptr != m_pMessage->GetBuffer())
    {
        if (false == IsLabelValid(m_pMessage->GetBuffer(),
                                  m_pMessage->GetBuffer() + m_pMessage->GetBufferSize(),
                                  pBuffer))
        {
            rc = -1;
            goto bail;
        };
    };

    rc = m_DirtySize = LabelSizeOfInternal(pBuffer, size);
    m_pData = pBuffer;

    m_Range.pEnd = pBuffer + rc;

bail:
    NN_DNSPARSER_LOG_DEBUG("returning: %zd\n", rc);
    return rc;
};

ssize_t Label::ToBuffer(uint8_t * const pBuffer, size_t size) const NN_NOEXCEPT
{
    NN_DNSPARSER_LOG_DEBUG("pBuffer: %p, size: %zu\n", pBuffer, size);

    ssize_t rc = -1;
    size_t bytes = 0;

    NN_DNSPARSER_ERROR_IF(nullptr == pBuffer, bail);

    if (-1 == (bytes = SizeOf()))
    {
        NN_DNSPARSER_LOG_DEBUG("\n");
        goto bail;
    }
    else if (size < bytes)
    {
        NN_DNSPARSER_LOG_DEBUG("\n");
        goto bail;
    };

    memcpy(pBuffer, m_pData, bytes);
    rc = bytes;

bail:
    NN_DNSPARSER_LOG_DEBUG("returning: %zd\n", rc);
    return rc;
};

ssize_t Label::ToBufferFromHostnameString(uint8_t * const pBuffer,
                                          size_t bufferSize,
                                          const char* pString) NN_NOEXCEPT
{
    NN_DNSPARSER_LOG_DEBUG("pBuffer: %p, bufferSize: %zd, pString: %p (%s)\n",
                           pBuffer, bufferSize, pString, pString);

    ssize_t rc = -1;
    size_t stringLength;

    NN_DNSPARSER_ERROR_IF(nullptr == pBuffer, bail);
    NN_DNSPARSER_ERROR_IF(0 == bufferSize, bail);
    NN_DNSPARSER_ERROR_IF(nullptr == pString, bail);

    stringLength = strlen(pString);

    NN_DNSPARSER_ERROR_IF(stringLength > static_cast<unsigned int>(LibraryConstant::HostnameMaxStringLength), bail);
    NN_DNSPARSER_ERROR_IF(static_cast<unsigned int>(LabelConstant::StringLabelPadding) + stringLength > bufferSize, bail);
    NN_DNSPARSER_ERROR_IF(!(isdigit(pString[0]) || isalpha(pString[0])), bail);;
    NN_DNSPARSER_ERROR_IF(!(isdigit(pString[stringLength - 1]) || isalpha(pString[stringLength - 1])), bail);

    for (int idx = 1; idx < stringLength - 1; ++idx)
    {
        if (!(isdigit(pString[idx]) || isalpha(pString[idx]) || '-' == pString[idx]))
        {
            goto bail;
        };
    };

    rc = ToBufferFromDomainNameString(pBuffer, bufferSize, pString);

bail:
    NN_DNSPARSER_LOG_DEBUG("returning: %zd\n", rc);
    return rc;
};

ssize_t Label::ToBufferFromDomainNameString(uint8_t * const pBuffer,
                                            size_t bufferSize,
                                            const char* pString) NN_NOEXCEPT
{
    NN_DNSPARSER_LOG_DEBUG("pBuffer: %p, bufferSize: %zd, pString: %p (%s)\n",
                           pBuffer, bufferSize, pString, pString);

    ssize_t rc = -1;
    size_t stringLength = 0;
    unsigned stringIndex = 0;
    unsigned bufferIndex = 1;
    unsigned lastLabelLength = 0;
    uint8_t numberCharacters = 0;

    NN_DNSPARSER_ERROR_IF(nullptr == pBuffer, bail);
    NN_DNSPARSER_ERROR_IF(0 == bufferSize, bail);
    NN_DNSPARSER_ERROR_IF(nullptr == pString, bail);

    stringLength = strlen(pString);

    NN_DNSPARSER_ERROR_IF(stringLength > static_cast<unsigned int>(LibraryConstant::DomainNameMaxStringLength), bail);
    NN_DNSPARSER_ERROR_IF(bufferSize < static_cast<unsigned int>(LabelConstant::StringLabelPadding) + stringLength , bail);
    NN_DNSPARSER_ERROR_IF(pString[0] == '.', bail);

    memcpy(pBuffer + 1, pString, stringLength + 1);

    for (stringIndex = 0; stringIndex < stringLength; ++stringIndex)
    {
        NN_DNSPARSER_ERROR_IF(bufferIndex == bufferSize, bail);
        if ('.' == pString[stringIndex])
        {
            if (numberCharacters > static_cast<int>(LibraryConstant::HostnameMaxStringLength))
            {
                NN_DNSPARSER_LOG_DEBUG("numberCharacters: %d\n", numberCharacters);
                goto bail;
            }
            pBuffer[lastLabelLength] = numberCharacters;
            numberCharacters = 0;
            lastLabelLength = bufferIndex;
        }
        else
        {
            pBuffer[bufferIndex] = pString[stringIndex];
            ++numberCharacters;
        };

        ++bufferIndex;
    };

    if (numberCharacters > static_cast<int>(LibraryConstant::HostnameMaxStringLength))
    {
        NN_DNSPARSER_LOG_DEBUG("numberCharacters: %d\n", numberCharacters);
        goto bail;
    }

    pBuffer[lastLabelLength] = numberCharacters;
    rc = stringLength + static_cast<unsigned int>(LabelConstant::StringLabelPadding);

bail:
    NN_DNSPARSER_LOG_DEBUG("returning: %zd\n", rc);
    return rc;
};

ssize_t Label::ToString(char* pString, size_t size) const NN_NOEXCEPT
{
    NN_DNSPARSER_LOG_DEBUG("pString: %p (%s), size: %zu\n", pString, pString, size);

    ssize_t rc = -1;
    const uint8_t* pCursor = nullptr;
    size_t limit = (static_cast<unsigned int>(LabelConstant::LabelSequenceMaximumBufferSize) < size ?
                    size : static_cast<unsigned int>(LabelConstant::LabelSequenceMaximumBufferSize));

    if (nullptr == pString)
    {
        NN_DNSPARSER_LOG_DEBUG("\n");
        goto bail;
    }
    else if (nullptr == m_pData)
    {
        NN_DNSPARSER_LOG_DEBUG("\n");
        goto bail;
    };

    rc = 0;

    for (pCursor = m_pData, rc = 0; rc < limit;)
    {
        if (true == IsLabelTypePointer(*pCursor))
        {
            if (nullptr == (pCursor = LabelGetCharacterCursor(this, pCursor)))
            {
                NN_DNSPARSER_LOG_DEBUG("\n");
                goto bail;
            };
        }
        else if (true == IsLabelTypeNormal(*pCursor))
        {
            if (*pCursor == 0)
            {
                *pString = '\0';
                goto bail;
            }
            else if (rc != 0)
            {
                *pString = '.';
                ++pString;
                rc += 1;
            };

            memcpy(pString, pCursor + 1, *pCursor);
            pString += *pCursor;
            rc += *pCursor;
            pCursor += 1 + *pCursor;
        }
        else
        {
            NN_DNSPARSER_LOG_DEBUG("\n");
            rc = -1;
            goto bail;
        };
    };

bail:
    NN_DNSPARSER_LOG_DEBUG("returning: %zd\n", rc);
    return rc;
};

}}}; //nn::dnsserver

