﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/socket/socket_Api.h>
#include <nn/dns//parser.h>

//#define NN_DNSPARSER_LOG_LEVEL NN_DNSPARSER_LOG_LEVEL_HEX
#define NN_DNSPARSER_LOG_MODULE_NAME "Header" // NOLINT(preprocessor/const)
#include "dns_ParserLog.h"
#include "dns_ParserMacros.h"

extern "C"
{
#include <nnc/dns/parser.h>
};

NN_DNSPARSER_STATIC_ASSERT(sizeof(struct nndnsparserHeader) == sizeof(nn::dns::parser::Header));
NN_DNSPARSER_STATIC_ASSERT(alignof(struct nndnsparserHeader) == alignof(nn::dns::parser::Header));

namespace nn { namespace dns { namespace parser {

Header::Header() :
    m_pMessage(nullptr),
    m_DirtySize(0),
    m_TransactionId(0),
    m_Flags {},
    m_QuestionCount(0),
    m_AnswerCount(0),
    m_AuthorityCount(0),
    m_AdditionalCount(0)
{
    m_Range.pStart = m_Range.pEnd = nullptr;
};

Header::Header(const Header& rhs) :
    m_pMessage(rhs.m_pMessage),
    m_DirtySize(rhs.m_DirtySize),
    m_TransactionId(rhs.m_TransactionId),
    m_Flags(rhs.m_Flags),
    m_QuestionCount(rhs.m_QuestionCount),
    m_AnswerCount(rhs.m_AnswerCount),
    m_AuthorityCount(rhs.m_AuthorityCount),
    m_AdditionalCount(rhs.m_AdditionalCount)
{
};

Header::~Header()
{
};

Header& Header::operator=(const Header& rhs)
{
    if (this == &rhs)
    {
        goto bail;
    };

    m_pMessage = rhs.m_pMessage;
    m_DirtySize = rhs.m_DirtySize;
    m_TransactionId = rhs.m_TransactionId;
    m_Flags = rhs.m_Flags;
    m_QuestionCount = rhs.m_QuestionCount;
    m_AnswerCount = rhs.m_AnswerCount;
    m_AuthorityCount = rhs.m_AuthorityCount;
    m_AdditionalCount = rhs.m_AdditionalCount;

bail:
    return *this;
}

const Message* & Header::GetMessage() NN_NOEXCEPT
{
    return m_pMessage;
};

uint16_t& Header::GetTransactionId() NN_NOEXCEPT
{
    return m_TransactionId;
};

HeaderFlagsConstant& Header::GetFlags() NN_NOEXCEPT
{
    return m_Flags;
};

uint16_t& Header::GetQuestionCount() NN_NOEXCEPT
{
    return m_QuestionCount;
};

uint16_t& Header::GetAnswerCount() NN_NOEXCEPT
{
    return m_AnswerCount;
};

uint16_t& Header::GetAuthorityCount() NN_NOEXCEPT
{
    return m_AuthorityCount;
};

uint16_t& Header::GetAdditionalCount() NN_NOEXCEPT
{
    return m_AdditionalCount;
};

void Header::Initialize(const Message* pMessage) NN_NOEXCEPT
{
    NN_DNSPARSER_STATIC_ASSERT(offsetof(struct nndnsparserHeader, pMessage) ==
                               offsetof(nn::dns::parser::Header, m_pMessage));

    NN_DNSPARSER_STATIC_ASSERT(offsetof(struct nndnsparserHeader, range) ==
                               offsetof(nn::dns::parser::Header, m_Range));

    NN_DNSPARSER_STATIC_ASSERT(offsetof(struct nndnsparserHeader, dirtySize) ==
                               offsetof(nn::dns::parser::Header, m_DirtySize));

    NN_DNSPARSER_STATIC_ASSERT(offsetof(struct nndnsparserHeader, transactionId) ==
                               offsetof(nn::dns::parser::Header, m_TransactionId));

    NN_DNSPARSER_STATIC_ASSERT(offsetof(struct nndnsparserHeader, flags) ==
                               offsetof(nn::dns::parser::Header, m_Flags));

    NN_DNSPARSER_STATIC_ASSERT(offsetof(struct nndnsparserHeader, questionCount) ==
                               offsetof(nn::dns::parser::Header, m_QuestionCount));

    NN_DNSPARSER_STATIC_ASSERT(offsetof(struct nndnsparserHeader, answerCount) ==
                               offsetof(nn::dns::parser::Header, m_AnswerCount));

    NN_DNSPARSER_STATIC_ASSERT(offsetof(struct nndnsparserHeader, authorityCount) ==
                               offsetof(nn::dns::parser::Header, m_AuthorityCount));

    NN_DNSPARSER_STATIC_ASSERT(offsetof(struct nndnsparserHeader, additionalCount) ==
                               offsetof(nn::dns::parser::Header, m_AdditionalCount));

    memset(this, 0, sizeof(*this));
    m_pMessage = pMessage;

    return;
};

ssize_t Header::SizeOf() const NN_NOEXCEPT
{
    NN_DNSPARSER_LOG_DEBUG("\n");
    ssize_t rc = -1;

    rc = static_cast<unsigned int>(HeaderConstant::Size);

    NN_DNSPARSER_LOG_DEBUG("returning: %zd\n", rc);
    return rc;
};

bool Header::operator==(const Header& that) const NN_NOEXCEPT
{
    bool rc = false;

    if (this == &that)
    {
        NN_DNSPARSER_LOG_DEBUG("\n");
        rc = true;
        goto bail;
    };

    if (m_TransactionId != that.m_TransactionId)
    {
        NN_DNSPARSER_LOG_DEBUG("\n");
        goto bail;
    }
    else if (m_Flags != that.m_Flags)
    {
        NN_DNSPARSER_LOG_DEBUG("\n");
        goto bail;
    }
    else if (m_QuestionCount != that.m_QuestionCount)
    {
        NN_DNSPARSER_LOG_DEBUG("\n");
        goto bail;
    }
    else if (m_AnswerCount != that.m_AnswerCount)
    {
        NN_DNSPARSER_LOG_DEBUG("\n");
        goto bail;
    }
    else if (m_AuthorityCount != that.m_AuthorityCount)
    {
        NN_DNSPARSER_LOG_DEBUG("\n");
        goto bail;
    }
    else if (m_AdditionalCount != that.m_AdditionalCount)
    {
        NN_DNSPARSER_LOG_DEBUG("\n");
        goto bail;
    };

    rc = true;

bail:
    return rc;
};

ssize_t Header::FromBuffer(const uint8_t* pBuffer, size_t size) NN_NOEXCEPT
{
    NN_DNSPARSER_LOG_DEBUG("pBuffer: %p, size: %zu\n", pBuffer, size);

    int rc = -1;
    auto pCursor = pBuffer;

    NN_DNSPARSER_ERROR_IF(nullptr == pCursor, bail);
    NN_DNSPARSER_ERROR_IF((static_cast<unsigned int>(HeaderConstant::Size) > size), bail);

    m_Range.pStart = pCursor;

    NN_DNSPARSER_FROMBUFFER_ERROR(rc, uint16_t, m_TransactionId,   pCursor, size, nn::socket::InetNtohs, bail);
    NN_DNSPARSER_FROMBUFFER_ERROR(rc, uint16_t, m_Flags,           pCursor, size, nn::socket::InetNtohs, bail);
    NN_DNSPARSER_FROMBUFFER_ERROR(rc, uint16_t, m_QuestionCount,   pCursor, size, nn::socket::InetNtohs, bail);
    NN_DNSPARSER_FROMBUFFER_ERROR(rc, uint16_t, m_AnswerCount,     pCursor, size, nn::socket::InetNtohs, bail);
    NN_DNSPARSER_FROMBUFFER_ERROR(rc, uint16_t, m_AuthorityCount,  pCursor, size, nn::socket::InetNtohs, bail);
    NN_DNSPARSER_FROMBUFFER_ERROR(rc, uint16_t, m_AdditionalCount, pCursor, size, nn::socket::InetNtohs, bail);

    m_Range.pEnd = pCursor;
    rc = m_DirtySize = pCursor - pBuffer;

bail:
    NN_DNSPARSER_LOG_DEBUG("returning: %zd\n", rc);
    return rc;
};

ssize_t Header::ToBuffer(uint8_t * const pBuffer, size_t size) const NN_NOEXCEPT
{
    NN_DNSPARSER_LOG_DEBUG("pBuffer: %p, size: %zu\n", pBuffer, size);

    int rc = -1;
    uint8_t* pCursor = pBuffer;

    NN_DNSPARSER_ERROR_IF(nullptr == pCursor, bail);
    NN_DNSPARSER_ERROR_IF((static_cast<unsigned int>(HeaderConstant::Size) > size), bail);

    NN_DNSPARSER_TOBUFFER_ERROR(rc, pCursor, size, nn::socket::InetHtons, uint16_t, m_TransactionId,   bail);
    NN_DNSPARSER_TOBUFFER_ERROR(rc, pCursor, size, nn::socket::InetHtons, uint16_t, m_Flags,           bail);
    NN_DNSPARSER_TOBUFFER_ERROR(rc, pCursor, size, nn::socket::InetHtons, uint16_t, m_QuestionCount,   bail);
    NN_DNSPARSER_TOBUFFER_ERROR(rc, pCursor, size, nn::socket::InetHtons, uint16_t, m_AnswerCount,     bail);
    NN_DNSPARSER_TOBUFFER_ERROR(rc, pCursor, size, nn::socket::InetHtons, uint16_t, m_AuthorityCount,  bail);
    NN_DNSPARSER_TOBUFFER_ERROR(rc, pCursor, size, nn::socket::InetHtons, uint16_t, m_AdditionalCount, bail);

    rc = pCursor - pBuffer;

bail:
    NN_DNSPARSER_LOG_DEBUG("returning: %zd\n", rc);
    return rc;
};

}}}; //nn::dnsparser
