﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include "dhcps_Common.h"

/**
 * @file
 *
 * @brief This file is the header for the UDP socket manager.
 */

namespace nn { namespace dhcps { namespace detail {

class Coordinator;

/**
 * @brief The UDP Socket Manager is one of two classes that has a
 * socket. Unlike @ref BpfManager the UDP socket does little if any
 * work. Rather it is a "burner" socket that handles port allocations
 * in the network network stack so that DHCP messages do not result in
 * ICMP unreachable messages.
 */
class UdpSocketManager : public virtual IEventHandler
{
public:
    /**
     * @brief The states of the UdpSocketManager.
     */
    enum class State : uint8_t
    {
        Uninitialized = 0, ///< 0
        Initialized   = 1, ///< 1
        FileOpen      = 2, ///< 2
        Ready         = 3, ///< 3
        FileClosed    = 4, ///< 4
        FileError     = 5, ///< 5
    };

    /**
     * @brief Constructor.
     */
    UdpSocketManager() NN_NOEXCEPT;

    /**
     * @brief Destructor.
     */
    ~UdpSocketManager() NN_NOEXCEPT;

    /**
     * @brief Get the file descriptor.
     *
     * @details Intended only for use with select.
     *
     * @return The file descriptor.
     */
    int GetFileDescriptor() NN_NOEXCEPT;

    /**
     * @brief Initializes with the provided @ref Coordinator.
     *
     * @param[in] pCoordinator The provided Coordinator.
     *
     * @return On success return 0, on error return -1.
     */
    int Initialize(Coordinator* pCoordinator) NN_NOEXCEPT;

    /**
     * @brief Finalizes the UdpSocketManager.
     *
     * @return On success return 0, on error return -1.
     */
    int Finalize() NN_NOEXCEPT;

    /**
     * @brief The @ref IEventHandler event handler.
     *
     * @param[in] e The event to handle.
     */
    virtual void OnEvent(const InternalEvent& e) NN_NOEXCEPT;

    /**
     * @brief Get the desired time for the next timeout event.
     *
     * @param[out] pOutTimeval The desired time for the next timeout event.
     */
    virtual void GetTimeout(nn::socket::TimeVal* pOutTimeval) NN_NOEXCEPT;

private:
    /**
     * @brief The current state.
     */
    State m_State;

    /**
     * @brief The socket file descriptor.
     */
    int m_FileDescriptor;

    /**
     * @brief The event @ref Coordinator.
     */
    Coordinator* m_pCoordinator;

    /**
     * @brief Changes the state variable.
     *
     * @param[in] next The next state to transition to.
     */
    void ChangeState(State next) NN_NOEXCEPT;
};

}}}; // nn::dhcps::detail
