﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include "dhcps_Common.h"

/**
 * @file
 *
 * @brief Header file for the thread that runs the DHCP server.
 */

namespace nn { namespace dhcps { namespace detail {

const size_t ThreadStackSize = 16 * 1024;

/**
 * @brief The LibraryThread class manages the thread that runs the
 * DHCP server.
 */
class LibraryThread
{
public:
    /**
     * @brief The states of the LibraryThread.
     */
    enum class State : uint8_t
    {
        Uninitialized = 0, ///< 0
        Initialized   = 1, ///< 1
        MemorySet     = 2, ///< 2
        Starting      = 3, ///< 3
        Running       = 4, ///< 4
        Stopping      = 5, ///< 5
        Stopped       = 6, ///< 6
        Error         = 7, ///< 7
    };

    /**
     * @brief Initializes the @ref LibraryThread with values provided
     * by the user.
     *
     * @param[in] userConfig Contains values provided by the user
     * to configure the library.
     *
     * @return A success or failure result code.
     */
    Result Initialize(const UserConfiguration& userConfig) NN_NOEXCEPT;

    /**
     * @brief Gets the size of memory required for the lease table.
     *
     * @param[out] pOutSize On success the value pointed at is updated
     * with the required memory size.
     *
     * @return A success or failure result code.
     */
    Result GetMemorySize(size_t* pOutSize) NN_NOEXCEPT;

    /**
     * @brief This function sets memory for the DHCP server lease
     * table.
     *
     * @details The size of memory may be more than is required but
     * it must not be less.
     *
     * @param[in] pMemory The block of memory used for the lease table.
     *
     * @param[in] size The size of memory pointed to by \a pMemory.
     * It may be more than required by @ref GetMemorySize but it may not
     * be less.
     *
     * @return A success or failure result code.
     */
    Result SetMemory(void* pMemory, size_t size) NN_NOEXCEPT;

    /**
     * @brief Starts the @ref LibraryThread.
     *
     * @return A success or failure result code.
     */
    Result Start() NN_NOEXCEPT;

    /**
     * @brief Stops the @ref LibraryThread.
     *
     * @return A success or failure result code.
     */
    Result Stop() NN_NOEXCEPT;

    /**
     * @brief Finalizes the @ref LibraryThread.
     *
     * @return A success or failure result code.
     */
    Result Finalize() NN_NOEXCEPT;

    /**
     * @brief Gets the state of the @ref LibraryThread.
     */
    State GetState() NN_NOEXCEPT;

    /**
     * @brief Gets the @ref DhcpLease by a @a ClientIdentifierHash.
     *
     * @param[out] pOutLease Contains a copy of the lease record on success.
     *
     * @param[in] clientId The @a ClientIdentifierHash key to search for
     *
     * @return Result code specifying either success or error.
     */
    Result GetLeaseByDhcpClientIdentifierHash(DhcpLease* pOutLease, ClientIdentifierHash clientId) NN_NOEXCEPT;

    /**
     * @brief Gets the @ref DhcpLease by an IP Address in network-byte-order.
     *
     * @param[out] pOutLease Contains a copy of the lease record on success.
     *
     * @param[in] ipAddress The IP address in network-byte order to search for.
     *
     * @return Result code specifying either success or error.
     */
    Result GetLeaseByInternetAddress(DhcpLease* pOutLease, nn::socket::InAddr ipAddress) NN_NOEXCEPT;

    /**
     * @brief Gets the @ref DhcpLease by a @a EthernetMacAddress.
     *
     * @param[out] pOutLease Contains a copy of the lease record on success.
     *
     * @param[in] macAddress The @a EthernetMacAddress to search for.
     *
     * @return Result code specifying either success or error.
     */
    Result GetLeaseByEthernetMacAddress(DhcpLease* pOutLease, const EthernetMacAddress& macAddress) NN_NOEXCEPT;

private:
    /**
     * @brief Static function provided to nn::os::CreateThread().
     *
     * @param[in] self The pointer to the LibraryThread object.
     */
    static void Run(void* self) NN_NOEXCEPT;

    /**
     * @brief Changes the @ref LibraryThread state.
     *
     * @param[in] next The next state to advance to.
     */
    void ChangeState(State next) NN_NOEXCEPT;

    /**
     * @brief The state of the @ref LibraryThread.
     */
    State m_State;

    /**
     * @brief This is the stack memory set aside for the
     * @ref LibraryThread.
     */
    NN_OS_ALIGNAS_THREAD_STACK char m_ThreadStack[ThreadStackSize];

    /**
     * @brief The HOS thread primitive.
     */
    nn::os::ThreadType m_Thread;
};

}}}; // nn::dhcps::detail
