﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "dhcps_LibraryThread.h"
#include "dhcps_Coordinator.h"
#include "dhcps_DhcpManager.h"
#include "dhcps_LeaseTableManager.h"

//#define NN_DHCPS_LOG_LEVEL NN_DHCPS_LOG_LEVEL_DEBUG
#define NN_DETAIL_DHCPS_LOG_MODULE_NAME "LibraryThread"
#include "dhcps_Log.h"

namespace nn { namespace dhcps { namespace detail {

Coordinator g_Coordinator;
LeaseTableManager g_LeaseTableManager;
extern Config g_Config;

const char* LibraryThreadStateToString(LibraryThread::State in) NN_NOEXCEPT
{
    switch (in)
    {
        NN_DETAIL_DHCPS_STRINGIFY_CASE(LibraryThread::State::Uninitialized);
        NN_DETAIL_DHCPS_STRINGIFY_CASE(LibraryThread::State::Initialized);
        NN_DETAIL_DHCPS_STRINGIFY_CASE(LibraryThread::State::MemorySet);
        NN_DETAIL_DHCPS_STRINGIFY_CASE(LibraryThread::State::Starting);
        NN_DETAIL_DHCPS_STRINGIFY_CASE(LibraryThread::State::Running);
        NN_DETAIL_DHCPS_STRINGIFY_CASE(LibraryThread::State::Stopping);
        NN_DETAIL_DHCPS_STRINGIFY_CASE(LibraryThread::State::Stopped);
        NN_DETAIL_DHCPS_STRINGIFY_CASE(LibraryThread::State::Error);
    default:
        ;
    };
    return "Unknown Thread State";
};

LibraryThread::State LibraryThread::GetState() NN_NOEXCEPT
{
    return m_State;
};

void LibraryThread::ChangeState(State next) NN_NOEXCEPT
{
    ApiLock();
    m_State = next;
    SetGlobalState(GlobalState::Thread, static_cast<uint8_t>(next));
    ApiUnlock();
};

Result LibraryThread::Initialize(const UserConfiguration& userConfig) NN_NOEXCEPT
{
    Result result = ResultInvalidState();

    ApiLock();
    {
        switch (m_State)
        {
        case State::Starting:
        case State::Running:
        case State::Stopping:
        case State::Stopped:
        case State::Error:
        case State::Initialized:
        case State::MemorySet:
            NN_DETAIL_DHCPS_LOG_MAJOR("Cannot Initialize; state %s (%d)\n",
                                           LibraryThreadStateToString(m_State),
                                           static_cast<int>(m_State));
            goto bail;
        case State::Uninitialized:
        default:
            ;
        };

        if ((result = g_Config.FromUserConfiguration(userConfig)).IsFailure())
        {
            goto bail;
        }
        else if ((result = g_Config.AcquireNetworkValues()).IsFailure())
        {
            goto bail;
        };

        g_Config.Print();

        ChangeState(State::Initialized);
        result = ResultSuccess();
    };

bail:
    ApiUnlock();
    return result;
};

Result LibraryThread::GetMemorySize(size_t* pOutSize) NN_NOEXCEPT
{
    Result result = ResultInvalidState();
    if (nullptr == pOutSize)
    {
        return ResultNullPointer();
    };

    ApiLock();
    {
        switch (m_State)
        {
        case State::Uninitialized:
        case State::Starting:
        case State::Running:
        case State::Stopping:
        case State::Stopped:
        case State::Error:
        case State::MemorySet:
            NN_DETAIL_DHCPS_LOG_MAJOR("Cannot GetMemorySize; state %s (%d)\n",
                                           LibraryThreadStateToString(m_State),
                                           static_cast<int>(m_State));
            goto bail;
        case State::Initialized:
        default:
            ;
        };

        *pOutSize = sizeof(LeaseRecord) * g_LeaseTableManager.GetTotalRecordCount();
        NN_DETAIL_DHCPS_LOG_DEBUG("memorySize: %d\n", *pOutSize);
        result = ResultSuccess();
    };

bail:
    ApiUnlock();
    return result;
};

Result LibraryThread::SetMemory(void* pMemory, size_t size) NN_NOEXCEPT
{
    Result result = ResultInvalidState();
    size_t necessary = 0;

    if (nullptr == pMemory)
    {
        return ResultNullPointer();
    }
    else if (0 == size)
    {
        return ResultInvalidSize();
    }
    else if (sizeof(LeaseRecord) > size)
    {
        return ResultInvalidSize();
    };

    ApiLock();
    {
        switch (m_State)
        {
        case State::Uninitialized:
        case State::Starting:
        case State::Running:
        case State::Stopping:
        case State::Error:
        case State::MemorySet:
        case State::Stopped:
            NN_DETAIL_DHCPS_LOG_MAJOR("Cannot SetMemory; state %s (%d).\n",
                                           LibraryThreadStateToString(m_State),
                                           static_cast<int>(m_State));
            goto bail;
        case State::Initialized:
        default:
            ;
        };


        necessary = sizeof(LeaseRecord) * g_LeaseTableManager.GetTotalRecordCount();
        if (size < necessary)
        {
            result = ResultInvalidSize();
            goto bail;
        };

        g_Config.SetLeaseTableMemory(pMemory, size);
        ChangeState(State::MemorySet);
        result = ResultSuccess();
    };

bail:
    ApiUnlock();
    return result;
};

Result LibraryThread::Start() NN_NOEXCEPT
{
    Result result = ResultSuccess();

    ApiLock();
    {
        switch (m_State)
        {
        case State::Uninitialized:
        case State::Initialized:
        case State::Starting:
        case State::Running:
        case State::Stopping:
        case State::Error:
            NN_DETAIL_DHCPS_LOG_MAJOR("Cannot Start; state %s (%d).\n",
                                           LibraryThreadStateToString(m_State),
                                           static_cast<int>(m_State));
            goto bail;
        case State::MemorySet:
        case State::Stopped:
        default:
            ;
        };

        // start will proceed
        ApiUnlock();
        {
            g_Config.GetCallback()(g_Config.GetContext(),
                                   static_cast<Event>(static_cast<uint32_t>(Event::OnStart)),
                                   nullptr);
        }
        ApiLock();

        result = nn::os::CreateThread(&m_Thread,
                                      LibraryThread::Run,
                                      this,
                                      m_ThreadStack,
                                      ThreadStackSize,
                                      nn::os::DefaultThreadPriority);

        if (!result.IsSuccess())
        {
            NN_DETAIL_DHCPS_LOG_MAJOR("Cannot Start; unable to CreateThread\n");
            NN_SDK_ASSERT(false);
            goto bail;
        };

        ChangeState(State::Starting);
        nn::os::StartThread( &m_Thread );
    };

bail:
    ApiUnlock();
    return result;
};

void LibraryThread::Run(void* pSelf) NN_NOEXCEPT
{
    LibraryThread* pThread = reinterpret_cast<LibraryThread*>(pSelf);
    if (nullptr == pThread)
    {
        NN_DETAIL_DHCPS_LOG_MAJOR("Unable to run, thread pointer is NULL.\n");
        goto bail;
    };

    ApiLock(); // Coordinator run unlocks Api lock

    nn::os::SetThreadNamePointer(nn::os::GetCurrentThread(), "DHCP Server Thread");

    if (-1 == g_Coordinator.Initialize(&g_LeaseTableManager))
    {
        ApiUnlock();
        g_Coordinator.ExternalStop();
        pThread->ChangeState(State::Error);
        goto bail;
    };

    pThread->ChangeState(State::Running);

    // Here the API lock from LibraryThread::Start is still held
    // but once the Coordinator grabs the InternalLock it releases
    // the ApiLock

    g_Coordinator.Run();
    pThread->ChangeState(State::Stopped);

bail:
    return;
};

Result LibraryThread::Stop() NN_NOEXCEPT
{
    Result result = ResultInvalidState();
    ApiLock();
    {
        switch (m_State)
        {
        case State::Uninitialized:
        case State::Initialized:
        case State::Starting:
        case State::MemorySet:
        case State::Stopping:
        case State::Stopped:
        case State::Error:
            NN_DETAIL_DHCPS_LOG_MAJOR("Cannot Stop; state %s (%d).\n",
                                           LibraryThreadStateToString(m_State),
                                           static_cast<int>(m_State));
            goto bail;
        case State::Running:
        default:
            ;
        };
        g_Coordinator.ExternalStop();
        ChangeState(State::Stopping);
    }

    // we unlock and lock the API here
    // so that any event callback making
    // a call back into the dhcps API
    // does not cause a deadlock

    ApiUnlock();
    nn::os::WaitThread(&m_Thread);
    ApiLock();

    nn::os::DestroyThread(&m_Thread);
    ChangeState(State::Stopped);
    result = ResultSuccess();

bail:
    ApiUnlock();
    return result;
};

Result LibraryThread::Finalize() NN_NOEXCEPT
{
    Result result = ResultInvalidState();

    ApiLock();
    {
        switch (m_State)
        {
        case State::Uninitialized:
        case State::Starting:
        case State::Running:
        case State::Stopping:
            NN_DETAIL_DHCPS_LOG_MAJOR("Cannot Finalize; state %s (%d).\n",
                                           LibraryThreadStateToString(m_State),
                                           static_cast<int>(m_State));
            goto bail;
        case State::Error:
        case State::Initialized:
        case State::MemorySet:
        case State::Stopped:
        default:
            ;
        };

        ChangeState(State::Uninitialized);
        g_Coordinator.Finalize();
        g_Config = Config();
        result = ResultSuccess();
    };

bail:
    ApiUnlock();

    for (unsigned int idx=0; idx < static_cast<unsigned int>(GlobalState::Max); ++idx)
    {
        uint8_t state = 0;
        if (0 != (state = GetGlobalState(static_cast<GlobalState>(idx))))
        {
            NN_DETAIL_DHCPS_LOG_DEBUG("%s state is %02x when it ought to be 0\n",
                                           GlobalStateIdentifierToString(static_cast<GlobalState>(idx)),
                                           idx);
        };
    };


    return result;
};

Result LibraryThread::GetLeaseByDhcpClientIdentifierHash(DhcpLease* pOutLease, ClientIdentifierHash clientId) NN_NOEXCEPT
{
    Result result = ResultNullPointer();
    bool didApiLock = false, didInternalLock = false;

    if (nullptr == pOutLease)
    {
        goto bail;
    };

    if (ClientIdentifierHashUnknown == clientId)
    {
        result = ResultClientIdentifierHashNotFound();
        goto bail;
    };

    ApiLock();
    {
        didApiLock = true;
        result = ResultInvalidState();

        switch (m_State)
        {
        case State::Uninitialized:
        case State::Starting:
        case State::Stopping:
        case State::Stopped:
        case State::Error:
        case State::Initialized:
        case State::MemorySet:
            NN_DETAIL_DHCPS_LOG_MAJOR("Cannot %s; state %s (%d)\n",
                                           __FUNCTION__,
                                           LibraryThreadStateToString(m_State),
                                           static_cast<int>(m_State));
            goto bail;
        case State::Running:
        default:
            ;
        };

        didInternalLock = true;
        InternalLock();
        {
            LeaseRecord* pRecord = nullptr;

            if ( -1 == g_LeaseTableManager.GetRecordByClientIdentifierHash(pRecord, clientId))
            {
                result = ResultClientIdentifierHashNotFound();
                goto bail;
            }
            else if (nullptr == pRecord)
            {
                result = ResultGenericError();
                goto bail;
            }
            else if (pRecord->state == ClientState::Unbound || pRecord->state == ClientState::Reserved)
            {
                result = ResultQueryInvalidDhcpState();
                goto bail;
            };

            LeaseRecordToDhcpLease(pOutLease, *pRecord);
            result = ResultSuccess();
        };
    };

bail:
    if (didInternalLock)
    {
        InternalUnlock();
    };

    if (didApiLock)
    {
        ApiUnlock();
    };

    return result;
};

Result LibraryThread::GetLeaseByInternetAddress(DhcpLease* pOutLease, nn::socket::InAddr ipAddress) NN_NOEXCEPT
{
    Result result = ResultNullPointer();
    bool didApiLock = false,  didInternalLock = false;

    if (nullptr == pOutLease)
    {
        goto bail;
    };

    ApiLock();
    {
        didApiLock = true;
        result = ResultInvalidState();

        switch (m_State)
        {
        case State::Uninitialized:
        case State::Starting:
        case State::Stopping:
        case State::Stopped:
        case State::Error:
        case State::Initialized:
        case State::MemorySet:
            NN_DETAIL_DHCPS_LOG_MAJOR("Cannot %s; state %s (%d)\n",
                                           __FUNCTION__,
                                           LibraryThreadStateToString(m_State),
                                           static_cast<int>(m_State));
            goto bail;
        case State::Running:
        default:
            ;
        };

        didInternalLock = true;
        InternalLock();
        {
            LeaseRecord* pRecord = nullptr;
            if ( -1 == g_LeaseTableManager.GetRecordByIpAddress(pRecord, ipAddress) )
            {
                result = ResultIpAddressNotFound();
                goto bail;
            }
            else if (nullptr == pRecord)
            {
                result = ResultGenericError();
                goto bail;
            }
            else if (pRecord->state == ClientState::Unbound || pRecord->state == ClientState::Reserved)
            {
                result = ResultQueryInvalidDhcpState();
                goto bail;
            };

            LeaseRecordToDhcpLease(pOutLease, *pRecord);
            result = ResultSuccess();
        };
    };

bail:
    if (didInternalLock)
    {
        InternalUnlock();
    };

    if (didApiLock)
    {
        ApiUnlock();
    };
    return result;
};

Result LibraryThread::GetLeaseByEthernetMacAddress(DhcpLease* pOutLease, const EthernetMacAddress& macAddress) NN_NOEXCEPT
{
    Result result = ResultNullPointer();
    bool didApiLock = false, didInternalLock = false;

    if (nullptr == pOutLease)
    {
        goto bail;
    };

    ApiLock();
    {
        didApiLock = true;
        result = ResultInvalidState();

        switch (m_State)
        {
        case State::Uninitialized:
        case State::Starting:
        case State::Stopping:
        case State::Stopped:
        case State::Error:
        case State::Initialized:
        case State::MemorySet:
            NN_DETAIL_DHCPS_LOG_MAJOR("Cannot %s; state %s (%d)\n",
                                           __FUNCTION__,
                                           LibraryThreadStateToString(m_State),
                                           static_cast<int>(m_State));
            goto bail;
        case State::Running:
        default:
            ;
        };

        didInternalLock = true;
        InternalLock();
        {
            LeaseRecord* pRecord = nullptr;
            if (-1 == g_LeaseTableManager.GetRecordByEthernetMacAddress(pRecord, macAddress))
            {
                result = ResultMacAddressNotFound();
                goto bail;
            }
            else if (nullptr == pRecord)
            {
                result = ResultGenericError();
                goto bail;
            }
            else if (pRecord->state == ClientState::Unbound || pRecord->state == ClientState::Reserved)
            {
                result = ResultQueryInvalidDhcpState();
                goto bail;
            };

            LeaseRecordToDhcpLease(pOutLease, *pRecord);
            result = ResultSuccess();
        };
    };

bail:
    if (didInternalLock)
    {
        InternalUnlock();
    };

    if (didApiLock)
    {
        ApiUnlock();
    };

    return result;
};

}}}; // nn::dhcps::detail
